//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/PlotSpecular/SpecularPlot.h
//! @brief     Defines class SpecularPlot
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_PLOTSPECULAR_SPECULARPLOT_H
#define BORNAGAIN_GUI_VIEW_PLOTSPECULAR_SPECULARPLOT_H

#include "GUI/View/PlotUtil/ScientificPlot.h"
#include <QMap>
#include <memory>

class BasicAxisItem;
class QCPAxis;
class QCPErrorBars;
class QCPGraph;
class QCPRange;
class SpecularDataItem;
class UpdateTimer;

//! The SpecularPlot class presents 1D intensity data from SpecularDataItem.

//! Provides minimal functionality for data plotting and axes interaction. Should be a component
//! for more complicated plotting widgets. Corresponds to ColorMap for 2D intensity data.

class SpecularPlot : public ScientificPlot {
    Q_OBJECT
public:
    explicit SpecularPlot(QWidget* parent = nullptr);

    void setSpecularItems(const QList<SpecularDataItem*>& dataItems) override;

    QSize sizeHint() const override { return QSize(500, 400); }
    QSize minimumSizeHint() const override { return QSize(128, 128); }

    //! Returns PlotEventInfo corresponding to given axes coordinates.
    PlotEventInfo eventInfo(double xpos, double ypos) const override;

    QCustomPlot* customPlot() override { return m_custom_plot; }
    const QCustomPlot* customPlot() const override { return m_custom_plot; }

    //! sets logarithmic scale
    void setLog();

private:
    //! Propagate xmin, xmax back to IntensityDataItem
    void onXaxisRangeChanged(QCPRange newRange);

    //! Propagate ymin, ymax back to IntensityDataItem
    void onYaxisRangeChanged(QCPRange newRange);

    //! Replots SpecularPlot.
    void onTimeToReplot();

    void connectItems();

    //! creates and initializes the color map
    void initPlot();

    void setConnected(bool isConnected);

    //! Connects/disconnects signals related to SpecularPlot's X,Y axes rectangle change.
    void setAxesRangeConnected(bool isConnected);

    void setUpdateTimerConnected(bool isConnected);

    //! Sets initial state of SpecularPlot for all items.
    void setPlot();

    //! Sets axes ranges and logarithmic scale on y-axis if necessary.
    void setAxes();

    //! Sets (xmin,xmax) and (ymin,ymax) of SpecularPlot from the first item.
    void setAxesRanges();

    //! Sets X,Y axes labels from the first item
    void setAxesLabels();

    //! Sets label to axis
    void setLabel(const BasicAxisItem* item, QCPAxis* axis);

    //! Sets the data values to SpecularPlot.
    void setDataFromItem(SpecularDataItem* item);

    //! Schedule replot for later execution by onTimeReplot() slot.
    void replot();

    QCustomPlot* m_custom_plot;
    UpdateTimer* m_update_timer;

    QMap<SpecularDataItem*, QCPGraph*> m_graph_map;
    QMap<SpecularDataItem*, QCPErrorBars*> m_errorbar_map;
};

#endif // BORNAGAIN_GUI_VIEW_PLOTSPECULAR_SPECULARPLOT_H
