//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Job/JobResultsPresenter.cpp
//! @brief     Implements class JobResultsPresenter
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Job/JobResultsPresenter.h"
#include "GUI/View/Job/JobViewActivities.h"
#include "GUI/View/Plot2D/IntensityDataWidget.h"
#include "GUI/View/PlotComparison/FitComparisonWidget.h"
#include "GUI/View/PlotComparison/FitComparisonWidget1D.h"
#include "GUI/View/PlotSpecular/SpecularDataWidget.h"
#include "GUI/View/Projection/IntensityDataProjectionsWidget.h"

namespace {

QString fitPresentationOfInstrument(JobItem* ji)
{
    return ji->isSpecularJob() ? "Fit 1D Data" : "Fit 2D Data";
}

QStringList defaultActivePresentationsOfInstrument(JobItem* ji)
{
    return ji->isSpecularJob() ? QStringList{"Reflectometry"}
                               : QStringList{"Heat Map", "Projections"};
}

QString defaultPresentationOfInstrument(JobItem* ji)
{
    return ji->isSpecularJob() ? "Reflectometry" : "Heat Map";
}

} // namespace


JobResultsPresenter::JobResultsPresenter(QWidget* parent)
    : ItemComboWidget(parent)
{
    setWindowTitle("Job Datafield");
    setObjectName("JobResultsPresenter");

    registerWidget("Heat Map", create_new<IntensityDataWidget>);
    registerWidget("Projections", create_new<IntensityDataProjectionsWidget>);
    registerWidget("Fit 1D Data", create_new<FitComparisonWidget1D>);
    registerWidget("Fit 2D Data", create_new<FitComparisonWidget>);
    registerWidget("Reflectometry", create_new<SpecularDataWidget>);

    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    setMinimumSize(220, 200);
}

QString JobResultsPresenter::itemPresentation() const
{
    if (!jobItem())
        return ItemComboWidget::selectedPresentation();

    if (jobItem()->presentationType().isEmpty())
        return ItemComboWidget::itemPresentation();

    return jobItem()->presentationType();
}

void JobResultsPresenter::setJobItem(JobItem* jobItem)
{
    ItemComboWidget::setJobOrRealItem(jobItem);
}

void JobResultsPresenter::setPresentation(const QString& presentationType)
{
    ItemComboWidget::setPresentation(presentationType);
    if (jobItem() && !presentationType.isEmpty())
        jobItem()->setPresentationType(presentationType);
}

void JobResultsPresenter::setPresentationForActivity(JobViewActivity activity)
{
    if (!jobItem() || jobItem()->isRunning())
        return;

    QString presentation = "";
    switch (activity) {
    case JobViewActivity::Fitting:
        presentation = fitPresentationOfInstrument(jobItem());
        break;

    case JobViewActivity::RealTime:
    case JobViewActivity::JobView:
        presentation = defaultPresentationOfInstrument(jobItem());
        break;
    }
    setPresentation(presentation);
}

//! Returns list of presentation types, available for given item. JobItem with fitting abilities
//! is valid for IntensityDataWidget and FitComparisonWidget.

QStringList JobResultsPresenter::activePresentationList() const
{
    if (!jobItem() || jobItem()->isRunning())
        return {};

    auto result = defaultActivePresentationsOfInstrument(jobItem());

    if (jobItem()->isValidForFitting())
        result << fitPresentationOfInstrument(jobItem());

    return result;
}

QStringList JobResultsPresenter::presentationList() const
{
    if (!jobItem() || jobItem()->isRunning())
        return {};

    auto result = defaultActivePresentationsOfInstrument(jobItem());
    result << fitPresentationOfInstrument(jobItem());

    return result;
}
