# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from gtk import STOCK_FILE, WRAP_WORD
from threading import Thread
from ConfigParser import ConfigParser, NoSectionError
from os.path import join, isdir
from os import makedirs

from common.config import ConfigLoader
from common.functions import Functions
functions = Functions()

from modules.lyrics.config import Config
from modules.lyrics.lyricsdownloader import LyricsDownloader

class Lyrics:
    def __init__(self, extensionsloader):
        self.extensions = extensionsloader
        self.module = {'name': 'Lyrics',
                       'logo': STOCK_FILE,
                       'configurable': True}

    def start_module(self):
        """This function starts the module."""
        # Start configuration
        name = self.module['name'].lower() + '.cfg'
        self.config = ConfigLoader()
        self.config_file = join(self.config.confdir, 'modules', name)
        self.data_dir = join(self.config.datadir, 'modules', 'lyrics')

        if not isdir(self.data_dir):
            makedirs(self.data_dir)

        def open_bluemindo(glade):
            """Bluemindo has been opened."""
            self.state = 0
            self.glade = glade
            self.current = None

            conf = self.get_configuration()
            self.lcdown = LyricsDownloader(conf)
            self.cfg = Config(self.module, self.lcdown,
                              self.config_file, self.get_configuration)

        def show_lyrics_for_song(song):
            title = song[0]
            artist = song[1]
            self.current = (title, artist)

            thread = Thread(group=None, target=self.get_lyrics,
                            name='lyrics', args=(title, artist))
            thread.start()

        def force_show_lyrics(song):
            self.glade.get_widget('tool-lyrics').set_active(True)

        def configuration(args):
            self.cfg.configuration(args)

        def save_config(args):
            self.cfg.save_config(args)

        # Connect to Bluemindo's signals
        self.extensions.connect('OnBluemindoStarted', open_bluemindo)
        self.extensions.connect('OnPlayNewSong', show_lyrics_for_song)
        self.extensions.connect('OnLyricsRequest', show_lyrics_for_song)
        self.extensions.connect('OnLyricsRequest', force_show_lyrics)
        self.extensions.connect('OnToolLyricsPressed', self.hideorshow)
        self.extensions.connect('OnToolReloadLyricsPressed', self.reload_lyric)
        self.extensions.connect('OnToolSaveLyricsPressed', self.save_lyric)
        self.extensions.connect('OnModuleConfiguration', configuration)
        self.extensions.connect('OnModuleConfigurationSave', save_config)

    def get_configuration(self):
        configparser = ConfigParser()
        configparser.read(self.config_file)
        config = {}

        try:
            for item in configparser.items(self.module['name']):
                try:
                    value = int(item[1])
                except ValueError:
                    value = str(item[1])

                config[item[0]] = value
        except NoSectionError:
            config['server'] = 'lyricwiki.org'
            config['active'] = True

        name = self.module['name'].lower() + '.cfg'
        config['__config-file'] = self.config_file
        config['__data-dir'] = self.data_dir

        return config

    def get_lyrics(self, title, artist, force_download=False):
        """Set the lyrics in the textview."""
        text = self.glade.get_widget('textview1')
        text.set_wrap_mode(WRAP_WORD)
        buffer_ = text.get_buffer()

        buffer_.set_text(_('Downloading lyrics for %(title)s - %(artist)s…') % 
                         {'title': self.current[0], 'artist': self.current[1]})

        lyrics = self.lcdown.get_lyrics(title, artist, force_download)
        if lyrics == '':
            lyrics = (_('No lyrics found for %(title)s - %(artist)s!') %
                       {'title': self.current[0], 'artist': self.current[1]})

        buffer_.set_text(lyrics)

    def reload_lyric(self):
        """Reload the current lyric."""
        if self.current is None:
            return

        (title, artist) = self.current
        self.current = (title, artist)

        thread = Thread(group=None, target=self.get_lyrics,
                        name='lyrics', args=(title, artist, True))
        thread.start()

    def save_lyric(self):
        """Save the lyric (usefull if user has made corrections)."""
        if self.current is None:
            return

        (title, artist) = self.current

        song_hash = functions.get_hash(title, artist)
        lyrics_file = join(self.data_dir, '%s.lyrics' % song_hash)

        text = self.glade.get_widget('textview1')
        buffer_ = text.get_buffer()
        lyrics = buffer_.get_text(buffer_.get_start_iter(),
                                  buffer_.get_end_iter())
        file_ = open(lyrics_file, 'w')
        file_.write(lyrics)
        file_.close()

    def hideorshow(self, song=None):
        if self.state == 0:
            self.state = 1
            self.glade.get_widget('hbox3').show()
            self.glade.get_widget('toolbar3').show()
            self.glade.get_widget('vpaned1').hide()
            self.glade.get_widget('toolbar4').hide()

        elif self.state == 1:
            self.state = 0
            self.glade.get_widget('hbox3').hide()
            self.glade.get_widget('toolbar3').hide()
            self.glade.get_widget('vpaned1').show()
            self.glade.get_widget('toolbar4').show()