// author: Fabian Hassler
// year: 2019

// This module implements a list of colormaps
// the code has been converted from the python library
//           matplotlib 3.0.2 license under BSD
// Feel free to use or to modify the code

// example: the generate a palette wistia
//    pen[] Palette = wistia.palette()
//
// There are two types of palettes. For a complete list see below:
// 
// 1) The segmented palettes can be used as
//      <name>.palette(int NColors=256, real gamma=1.)
//         NColors are the number of colors in the palette
//         gamma is the gamma-factor
//
// 2) The listed palettes can only be used as
//      <name>.palette()
//
// Both functions return pen[] that can be used as a palette in the 
// module palette.

// list of palettes
//    see also https://matplotlib.org/tutorials/colors/colormaps.html

// segmented palettes:
//     CMRmap
//     autumn
//     binary
//     bone
//     cool
//     coolwarm
//     copper
//     gist_earth
//     gist_ncar
//     gist_stern
//     gray
//     hot
//     hsv
//     jet
//     nipy_spectral
//     pink
//     spring
//     summer
//     winter
//     wistia

// listed palettes:
//     Accent
//     Blues
//     BrBG
//     BuGn
//     BuPu
//     Dark2
//     GnBu
//     Greens
//     Greys
//     OrRd
//     Oranges
//     PRGn
//     Paired
//     Pastel1
//     Pastel2
//     PiYG
//     PuBuGn
//     PuBu
//     PuOr
//     PuRd
//     Purples
//     RdBu
//     RdGy
//     RdPu
//     RdYlBu
//     RdYlGn
//     Reds
//     Set1
//     Set2
//     Set3
//     Spectral
//     YlGnBu
//     YlGn
//     YlOrBr
//     YlOrRd
//     brg
//     bwr
//     seismic
//     tab10
//     tab20
//     tab20b
//     tab20c
//     cividis
//     inferno
//     magma
//     plasma
//     twilight
//     twilight_shifted
//     viridis


// Example of usage:

// import graph;
// import palette;
// import colormap;
// int NColors=5;
// pen[] Palette=spring.palette(NColors);
// palette(bounds(0,1),(0.,0),(500,50),Bottom,Palette);

//
// SOURCE CODE
//
private real[] makeMappingArray(int N, triple[] data, real gamma=1.) {
  real[] x; 
  real[] y0;
  real[] y1;

  for (int i=0; i<data.length; ++i) {
    x.push(data[i].x);
    y0.push(data[i].y);
    y1.push(data[i].z);
  };
  
  x = x*(N-1);
  real[] lut = new real[N]; 
  real[] xind = (N - 1) * uniform(0, 1, N-1) ** gamma;
  int[] ind = map(new int(real xi) {return search(x, xi);}, xind);
  ind = ind[1:N-1]; // note that the index is shifted from python
  
  real[] dist = (xind[1:N-1] - x[ind])/(x[ind+1] - x[ind]);

  lut[1:N-1] = dist * (y0[ind+1] - y1[ind]) + y1[ind];
  lut[0] = y1[0];
  lut[N-1] = y0[y0.length-1];
  return lut;
}

// struct for segmented data
struct seg_data {
  private triple[] r; // red
  private triple[] g; // green
  private triple[] b; // blue

  void operator init(triple[] r, triple[] g, triple[] b) {
    this.r=r;
    this.g=g;
    this.b=b;
  }

  pen[] palette(int NColors=256, real gamma=1.) {
    real[] red   = makeMappingArray(NColors, this.r, gamma);
    real[] green = makeMappingArray(NColors, this.g, gamma);
    real[] blue  = makeMappingArray(NColors, this.b, gamma);

    pen[] pal = 
      sequence(new pen(int i) {return rgb(red[i], green[i], blue[i]);}, 
        NColors);

    return pal;
  }
}

// struct for list data
struct list_data {
  private pen[] data;
  pen[] palette(){return data;}
  void operator init(pen[] d) {
    this.data=d;
  }
}

//
// DATA
//
list_data Accent = list_data(new pen[] {
  rgb (0.4980392156862745, 0.788235294117647, 0.4980392156862745) ,
  rgb (0.7450980392156863, 0.6823529411764706, 0.8313725490196079) ,
  rgb (0.9921568627450981, 0.7529411764705882, 0.5254901960784314) ,
  rgb (1.0, 1.0, 0.6) ,
  rgb (0.2196078431372549, 0.4235294117647059, 0.6901960784313725) ,
  rgb (0.9411764705882353, 0.00784313725490196, 0.4980392156862745) ,
  rgb (0.7490196078431373, 0.3568627450980392, 0.09019607843137253) ,
  rgb (0.4, 0.4, 0.4)
});


list_data Blues = list_data(new pen[] {
  rgb (0.9686274509803922, 0.984313725490196, 1.0) ,
  rgb (0.8705882352941177, 0.9215686274509803, 0.9686274509803922) ,
  rgb (0.7764705882352941, 0.8588235294117647, 0.9372549019607843) ,
  rgb (0.6196078431372549, 0.792156862745098, 0.8823529411764706) ,
  rgb (0.4196078431372549, 0.6823529411764706, 0.8392156862745098) ,
  rgb (0.25882352941176473, 0.5725490196078431, 0.7764705882352941) ,
  rgb (0.12941176470588237, 0.44313725490196076, 0.7098039215686275) ,
  rgb (0.03137254901960784, 0.3176470588235294, 0.611764705882353) ,
  rgb (0.03137254901960784, 0.18823529411764706, 0.4196078431372549)
});


list_data BrBG = list_data(new pen[] {
  rgb (0.32941176470588235, 0.18823529411764706, 0.0196078431372549) ,
  rgb (0.5490196078431373, 0.3176470588235294, 0.0392156862745098) ,
  rgb (0.7490196078431373, 0.5058823529411764, 0.17647058823529413) ,
  rgb (0.8745098039215686, 0.7607843137254902, 0.49019607843137253) ,
  rgb (0.9647058823529412, 0.9098039215686274, 0.7647058823529411) ,
  rgb (0.9607843137254902, 0.9607843137254902, 0.9607843137254902) ,
  rgb (0.7803921568627451, 0.9176470588235294, 0.8980392156862745) ,
  rgb (0.5019607843137255, 0.803921568627451, 0.7568627450980392) ,
  rgb (0.20784313725490197, 0.592156862745098, 0.5607843137254902) ,
  rgb (0.00392156862745098, 0.4, 0.3686274509803922) ,
  rgb (0.0, 0.23529411764705882, 0.18823529411764706)
});


list_data BuGn = list_data(new pen[] {
  rgb (0.9686274509803922, 0.9882352941176471, 0.9921568627450981) ,
  rgb (0.8980392156862745, 0.9607843137254902, 0.9764705882352941) ,
  rgb (0.8, 0.9254901960784314, 0.9019607843137255) ,
  rgb (0.6, 0.8470588235294118, 0.788235294117647) ,
  rgb (0.4, 0.7607843137254902, 0.6431372549019608) ,
  rgb (0.2549019607843137, 0.6823529411764706, 0.4627450980392157) ,
  rgb (0.13725490196078433, 0.5450980392156862, 0.27058823529411763) ,
  rgb (0.0, 0.42745098039215684, 0.17254901960784313) ,
  rgb (0.0, 0.26666666666666666, 0.10588235294117647)
});


list_data BuPu = list_data(new pen[] {
  rgb (0.9686274509803922, 0.9882352941176471, 0.9921568627450981) ,
  rgb (0.8784313725490196, 0.9254901960784314, 0.9568627450980393) ,
  rgb (0.7490196078431373, 0.8274509803921568, 0.9019607843137255) ,
  rgb (0.6196078431372549, 0.7372549019607844, 0.8549019607843137) ,
  rgb (0.5490196078431373, 0.5882352941176471, 0.7764705882352941) ,
  rgb (0.5490196078431373, 0.4196078431372549, 0.6941176470588235) ,
  rgb (0.5333333333333333, 0.2549019607843137, 0.615686274509804) ,
  rgb (0.5058823529411764, 0.05882352941176471, 0.48627450980392156) ,
  rgb (0.30196078431372547, 0.0, 0.29411764705882354)
});


seg_data CMRmap = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (0.125, 0.15, 0.15) ,
     (0.25, 0.3, 0.3) ,
     (0.375, 0.6, 0.6) ,
     (0.5, 1.0, 1.0) ,
     (0.625, 0.9, 0.9) ,
     (0.75, 0.9, 0.9) ,
     (0.875, 0.9, 0.9) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.125, 0.15, 0.15) ,
     (0.25, 0.15, 0.15) ,
     (0.375, 0.2, 0.2) ,
     (0.5, 0.25, 0.25) ,
     (0.625, 0.5, 0.5) ,
     (0.75, 0.75, 0.75) ,
     (0.875, 0.9, 0.9) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.125, 0.5, 0.5) ,
     (0.25, 0.75, 0.75) ,
     (0.375, 0.5, 0.5) ,
     (0.5, 0.15, 0.15) ,
     (0.625, 0.0, 0.0) ,
     (0.75, 0.1, 0.1) ,
     (0.875, 0.5, 0.5) ,
     (1.0, 1.0, 1.0)
  }
);


list_data Dark2 = list_data(new pen[] {
  rgb (0.10588235294117647, 0.6196078431372549, 0.4666666666666667) ,
  rgb (0.8509803921568627, 0.37254901960784315, 0.00784313725490196) ,
  rgb (0.4588235294117647, 0.4392156862745098, 0.7019607843137254) ,
  rgb (0.9058823529411765, 0.1607843137254902, 0.5411764705882353) ,
  rgb (0.4, 0.6509803921568628, 0.11764705882352941) ,
  rgb (0.9019607843137255, 0.6705882352941176, 0.00784313725490196) ,
  rgb (0.6509803921568628, 0.4627450980392157, 0.11372549019607843) ,
  rgb (0.4, 0.4, 0.4)
});


list_data GnBu = list_data(new pen[] {
  rgb (0.9686274509803922, 0.9882352941176471, 0.9411764705882353) ,
  rgb (0.8784313725490196, 0.9529411764705882, 0.8588235294117647) ,
  rgb (0.8, 0.9215686274509803, 0.7725490196078432) ,
  rgb (0.6588235294117647, 0.8666666666666667, 0.7098039215686275) ,
  rgb (0.4823529411764706, 0.8, 0.7686274509803922) ,
  rgb (0.3058823529411765, 0.7019607843137254, 0.8274509803921568) ,
  rgb (0.16862745098039217, 0.5490196078431373, 0.7450980392156863) ,
  rgb (0.03137254901960784, 0.40784313725490196, 0.6745098039215687) ,
  rgb (0.03137254901960784, 0.25098039215686274, 0.5058823529411764)
});


list_data Greens = list_data(new pen[] {
  rgb (0.9686274509803922, 0.9882352941176471, 0.9607843137254902) ,
  rgb (0.8980392156862745, 0.9607843137254902, 0.8784313725490196) ,
  rgb (0.7803921568627451, 0.9137254901960784, 0.7529411764705882) ,
  rgb (0.6313725490196078, 0.8509803921568627, 0.6078431372549019) ,
  rgb (0.4549019607843137, 0.7686274509803922, 0.4627450980392157) ,
  rgb (0.2549019607843137, 0.6705882352941176, 0.36470588235294116) ,
  rgb (0.13725490196078433, 0.5450980392156862, 0.27058823529411763) ,
  rgb (0.0, 0.42745098039215684, 0.17254901960784313) ,
  rgb (0.0, 0.26666666666666666, 0.10588235294117647)
});


list_data Greys = list_data(new pen[] {
  rgb (1.0, 1.0, 1.0) ,
  rgb (0.9411764705882353, 0.9411764705882353, 0.9411764705882353) ,
  rgb (0.8509803921568627, 0.8509803921568627, 0.8509803921568627) ,
  rgb (0.7411764705882353, 0.7411764705882353, 0.7411764705882353) ,
  rgb (0.5882352941176471, 0.5882352941176471, 0.5882352941176471) ,
  rgb (0.45098039215686275, 0.45098039215686275, 0.45098039215686275) ,
  rgb (0.3215686274509804, 0.3215686274509804, 0.3215686274509804) ,
  rgb (0.1450980392156863, 0.1450980392156863, 0.1450980392156863) ,
  rgb (0.0, 0.0, 0.0)
});


list_data OrRd = list_data(new pen[] {
  rgb (1.0, 0.9686274509803922, 0.9254901960784314) ,
  rgb (0.996078431372549, 0.9098039215686274, 0.7843137254901961) ,
  rgb (0.9921568627450981, 0.8313725490196079, 0.6196078431372549) ,
  rgb (0.9921568627450981, 0.7333333333333333, 0.5176470588235295) ,
  rgb (0.9882352941176471, 0.5529411764705883, 0.34901960784313724) ,
  rgb (0.9372549019607843, 0.396078431372549, 0.2823529411764706) ,
  rgb (0.8431372549019608, 0.18823529411764706, 0.12156862745098039) ,
  rgb (0.7019607843137254, 0.0, 0.0) ,
  rgb (0.4980392156862745, 0.0, 0.0)
});


list_data Oranges = list_data(new pen[] {
  rgb (1.0, 0.9607843137254902, 0.9215686274509803) ,
  rgb (0.996078431372549, 0.9019607843137255, 0.807843137254902) ,
  rgb (0.9921568627450981, 0.8156862745098039, 0.6352941176470588) ,
  rgb (0.9921568627450981, 0.6823529411764706, 0.4196078431372549) ,
  rgb (0.9921568627450981, 0.5529411764705883, 0.23529411764705882) ,
  rgb (0.9450980392156862, 0.4117647058823529, 0.07450980392156863) ,
  rgb (0.8509803921568627, 0.2823529411764706, 0.00392156862745098) ,
  rgb (0.6509803921568628, 0.21176470588235294, 0.01176470588235294) ,
  rgb (0.4980392156862745, 0.15294117647058825, 0.01568627450980392)
});


list_data PRGn = list_data(new pen[] {
  rgb (0.25098039215686274, 0.0, 0.29411764705882354) ,
  rgb (0.4627450980392157, 0.16470588235294117, 0.5137254901960784) ,
  rgb (0.6, 0.4392156862745098, 0.6705882352941176) ,
  rgb (0.7607843137254902, 0.6470588235294118, 0.8117647058823529) ,
  rgb (0.9058823529411765, 0.8313725490196079, 0.9098039215686274) ,
  rgb (0.9686274509803922, 0.9686274509803922, 0.9686274509803922) ,
  rgb (0.8509803921568627, 0.9411764705882353, 0.8274509803921568) ,
  rgb (0.6509803921568628, 0.8588235294117647, 0.6274509803921569) ,
  rgb (0.35294117647058826, 0.6823529411764706, 0.3803921568627451) ,
  rgb (0.10588235294117647, 0.47058823529411764, 0.21568627450980393) ,
  rgb (0.0, 0.26666666666666666, 0.10588235294117647)
});


list_data Paired = list_data(new pen[] {
  rgb (0.6509803921568628, 0.807843137254902, 0.8901960784313725) ,
  rgb (0.12156862745098039, 0.47058823529411764, 0.7058823529411765) ,
  rgb (0.6980392156862745, 0.8745098039215686, 0.5411764705882353) ,
  rgb (0.2, 0.6274509803921569, 0.17254901960784313) ,
  rgb (0.984313725490196, 0.6039215686274509, 0.6) ,
  rgb (0.8901960784313725, 0.10196078431372549, 0.10980392156862745) ,
  rgb (0.9921568627450981, 0.7490196078431373, 0.43529411764705883) ,
  rgb (1.0, 0.4980392156862745, 0.0) ,
  rgb (0.792156862745098, 0.6980392156862745, 0.8392156862745098) ,
  rgb (0.41568627450980394, 0.23921568627450981, 0.6039215686274509) ,
  rgb (1.0, 1.0, 0.6) ,
  rgb (0.6941176470588235, 0.34901960784313724, 0.1568627450980392)
});


list_data Pastel1 = list_data(new pen[] {
  rgb (0.984313725490196, 0.7058823529411765, 0.6823529411764706) ,
  rgb (0.7019607843137254, 0.803921568627451, 0.8901960784313725) ,
  rgb (0.8, 0.9215686274509803, 0.7725490196078432) ,
  rgb (0.8705882352941177, 0.796078431372549, 0.8941176470588236) ,
  rgb (0.996078431372549, 0.8509803921568627, 0.6509803921568628) ,
  rgb (1.0, 1.0, 0.8) ,
  rgb (0.8980392156862745, 0.8470588235294118, 0.7411764705882353) ,
  rgb (0.9921568627450981, 0.8549019607843137, 0.9254901960784314) ,
  rgb (0.9490196078431372, 0.9490196078431372, 0.9490196078431372)
});


list_data Pastel2 = list_data(new pen[] {
  rgb (0.7019607843137254, 0.8862745098039215, 0.803921568627451) ,
  rgb (0.9921568627450981, 0.803921568627451, 0.6745098039215687) ,
  rgb (0.796078431372549, 0.8352941176470589, 0.9098039215686274) ,
  rgb (0.9568627450980393, 0.792156862745098, 0.8941176470588236) ,
  rgb (0.9019607843137255, 0.9607843137254902, 0.788235294117647) ,
  rgb (1.0, 0.9490196078431372, 0.6823529411764706) ,
  rgb (0.9450980392156862, 0.8862745098039215, 0.8) ,
  rgb (0.8, 0.8, 0.8)
});


list_data PiYG = list_data(new pen[] {
  rgb (0.5568627450980392, 0.00392156862745098, 0.3215686274509804) ,
  rgb (0.7725490196078432, 0.10588235294117647, 0.49019607843137253) ,
  rgb (0.8705882352941177, 0.4666666666666667, 0.6823529411764706) ,
  rgb (0.9450980392156862, 0.7137254901960784, 0.8549019607843137) ,
  rgb (0.9921568627450981, 0.8784313725490196, 0.9372549019607843) ,
  rgb (0.9686274509803922, 0.9686274509803922, 0.9686274509803922) ,
  rgb (0.9019607843137255, 0.9607843137254902, 0.8156862745098039) ,
  rgb (0.7215686274509804, 0.8823529411764706, 0.5254901960784314) ,
  rgb (0.4980392156862745, 0.7372549019607844, 0.2549019607843137) ,
  rgb (0.30196078431372547, 0.5725490196078431, 0.12941176470588237) ,
  rgb (0.15294117647058825, 0.39215686274509803, 0.09803921568627451)
});


list_data PuBuGn = list_data(new pen[] {
  rgb (1.0, 0.9686274509803922, 0.984313725490196) ,
  rgb (0.9254901960784314, 0.8862745098039215, 0.9411764705882353) ,
  rgb (0.8156862745098039, 0.8196078431372549, 0.9019607843137255) ,
  rgb (0.6509803921568628, 0.7411764705882353, 0.8588235294117647) ,
  rgb (0.403921568627451, 0.6627450980392157, 0.8117647058823529) ,
  rgb (0.21176470588235294, 0.5647058823529412, 0.7529411764705882) ,
  rgb (0.00784313725490196, 0.5058823529411764, 0.5411764705882353) ,
  rgb (0.00392156862745098, 0.4235294117647059, 0.34901960784313724) ,
  rgb (0.00392156862745098, 0.27450980392156865, 0.21176470588235294)
});


list_data PuBu = list_data(new pen[] {
  rgb (1.0, 0.9686274509803922, 0.984313725490196) ,
  rgb (0.9254901960784314, 0.9058823529411765, 0.9490196078431372) ,
  rgb (0.8156862745098039, 0.8196078431372549, 0.9019607843137255) ,
  rgb (0.6509803921568628, 0.7411764705882353, 0.8588235294117647) ,
  rgb (0.4549019607843137, 0.6627450980392157, 0.8117647058823529) ,
  rgb (0.21176470588235294, 0.5647058823529412, 0.7529411764705882) ,
  rgb (0.0196078431372549, 0.4392156862745098, 0.6901960784313725) ,
  rgb (0.01568627450980392, 0.35294117647058826, 0.5529411764705883) ,
  rgb (0.00784313725490196, 0.2196078431372549, 0.34509803921568627)
});


list_data PuOr = list_data(new pen[] {
  rgb (0.4980392156862745, 0.23137254901960785, 0.03137254901960784) ,
  rgb (0.7019607843137254, 0.34509803921568627, 0.02352941176470588) ,
  rgb (0.8784313725490196, 0.5098039215686274, 0.0784313725490196) ,
  rgb (0.9921568627450981, 0.7215686274509804, 0.38823529411764707) ,
  rgb (0.996078431372549, 0.8784313725490196, 0.7137254901960784) ,
  rgb (0.9686274509803922, 0.9686274509803922, 0.9686274509803922) ,
  rgb (0.8470588235294118, 0.8549019607843137, 0.9215686274509803) ,
  rgb (0.6980392156862745, 0.6705882352941176, 0.8235294117647058) ,
  rgb (0.5019607843137255, 0.45098039215686275, 0.6745098039215687) ,
  rgb (0.32941176470588235, 0.15294117647058825, 0.5333333333333333) ,
  rgb (0.17647058823529413, 0.0, 0.29411764705882354)
});


list_data PuRd = list_data(new pen[] {
  rgb (0.9686274509803922, 0.9568627450980393, 0.9764705882352941) ,
  rgb (0.9058823529411765, 0.8823529411764706, 0.9372549019607843) ,
  rgb (0.8313725490196079, 0.7254901960784313, 0.8549019607843137) ,
  rgb (0.788235294117647, 0.5803921568627451, 0.7803921568627451) ,
  rgb (0.8745098039215686, 0.396078431372549, 0.6901960784313725) ,
  rgb (0.9058823529411765, 0.1607843137254902, 0.5411764705882353) ,
  rgb (0.807843137254902, 0.07058823529411765, 0.33725490196078434) ,
  rgb (0.596078431372549, 0.0, 0.2627450980392157) ,
  rgb (0.403921568627451, 0.0, 0.12156862745098039)
});


list_data Purples = list_data(new pen[] {
  rgb (0.9882352941176471, 0.984313725490196, 0.9921568627450981) ,
  rgb (0.9372549019607843, 0.9294117647058824, 0.9607843137254902) ,
  rgb (0.8549019607843137, 0.8549019607843137, 0.9215686274509803) ,
  rgb (0.7372549019607844, 0.7411764705882353, 0.8627450980392157) ,
  rgb (0.6196078431372549, 0.6039215686274509, 0.7843137254901961) ,
  rgb (0.5019607843137255, 0.49019607843137253, 0.7294117647058823) ,
  rgb (0.41568627450980394, 0.3176470588235294, 0.6392156862745098) ,
  rgb (0.32941176470588235, 0.15294117647058825, 0.5607843137254902) ,
  rgb (0.24705882352941178, 0.0, 0.49019607843137253)
});


list_data RdBu = list_data(new pen[] {
  rgb (0.403921568627451, 0.0, 0.12156862745098039) ,
  rgb (0.6980392156862745, 0.09411764705882353, 0.16862745098039217) ,
  rgb (0.8392156862745098, 0.3764705882352941, 0.30196078431372547) ,
  rgb (0.9568627450980393, 0.6470588235294118, 0.5098039215686274) ,
  rgb (0.9921568627450981, 0.8588235294117647, 0.7803921568627451) ,
  rgb (0.9686274509803922, 0.9686274509803922, 0.9686274509803922) ,
  rgb (0.8196078431372549, 0.8980392156862745, 0.9411764705882353) ,
  rgb (0.5725490196078431, 0.7725490196078432, 0.8705882352941177) ,
  rgb (0.2627450980392157, 0.5764705882352941, 0.7647058823529411) ,
  rgb (0.12941176470588237, 0.4, 0.6745098039215687) ,
  rgb (0.0196078431372549, 0.18823529411764706, 0.3803921568627451)
});


list_data RdGy = list_data(new pen[] {
  rgb (0.403921568627451, 0.0, 0.12156862745098039) ,
  rgb (0.6980392156862745, 0.09411764705882353, 0.16862745098039217) ,
  rgb (0.8392156862745098, 0.3764705882352941, 0.30196078431372547) ,
  rgb (0.9568627450980393, 0.6470588235294118, 0.5098039215686274) ,
  rgb (0.9921568627450981, 0.8588235294117647, 0.7803921568627451) ,
  rgb (1.0, 1.0, 1.0) ,
  rgb (0.8784313725490196, 0.8784313725490196, 0.8784313725490196) ,
  rgb (0.7294117647058823, 0.7294117647058823, 0.7294117647058823) ,
  rgb (0.5294117647058824, 0.5294117647058824, 0.5294117647058824) ,
  rgb (0.30196078431372547, 0.30196078431372547, 0.30196078431372547) ,
  rgb (0.10196078431372549, 0.10196078431372549, 0.10196078431372549)
});


list_data RdPu = list_data(new pen[] {
  rgb (1.0, 0.9686274509803922, 0.9529411764705882) ,
  rgb (0.9921568627450981, 0.8784313725490196, 0.8666666666666667) ,
  rgb (0.9882352941176471, 0.7725490196078432, 0.7529411764705882) ,
  rgb (0.9803921568627451, 0.6235294117647059, 0.7098039215686275) ,
  rgb (0.9686274509803922, 0.40784313725490196, 0.6313725490196078) ,
  rgb (0.8666666666666667, 0.20392156862745098, 0.592156862745098) ,
  rgb (0.6823529411764706, 0.00392156862745098, 0.49411764705882355) ,
  rgb (0.47843137254901963, 0.00392156862745098, 0.4666666666666667) ,
  rgb (0.28627450980392155, 0.0, 0.41568627450980394)
});


list_data RdYlBu = list_data(new pen[] {
  rgb (0.6470588235294118, 0.0, 0.14901960784313725) ,
  rgb (0.8431372549019608, 0.18823529411764706, 0.15294117647058825) ,
  rgb (0.9568627450980393, 0.42745098039215684, 0.2627450980392157) ,
  rgb (0.9921568627450981, 0.6823529411764706, 0.3803921568627451) ,
  rgb (0.996078431372549, 0.8784313725490196, 0.5647058823529412) ,
  rgb (1.0, 1.0, 0.7490196078431373) ,
  rgb (0.8784313725490196, 0.9529411764705882, 0.9725490196078431) ,
  rgb (0.6705882352941176, 0.8509803921568627, 0.9137254901960784) ,
  rgb (0.4549019607843137, 0.6784313725490196, 0.8196078431372549) ,
  rgb (0.27058823529411763, 0.4588235294117647, 0.7058823529411765) ,
  rgb (0.19215686274509805, 0.21176470588235294, 0.5843137254901961)
});


list_data RdYlGn = list_data(new pen[] {
  rgb (0.6470588235294118, 0.0, 0.14901960784313725) ,
  rgb (0.8431372549019608, 0.18823529411764706, 0.15294117647058825) ,
  rgb (0.9568627450980393, 0.42745098039215684, 0.2627450980392157) ,
  rgb (0.9921568627450981, 0.6823529411764706, 0.3803921568627451) ,
  rgb (0.996078431372549, 0.8784313725490196, 0.5450980392156862) ,
  rgb (1.0, 1.0, 0.7490196078431373) ,
  rgb (0.8509803921568627, 0.9372549019607843, 0.5450980392156862) ,
  rgb (0.6509803921568628, 0.8509803921568627, 0.41568627450980394) ,
  rgb (0.4, 0.7411764705882353, 0.38823529411764707) ,
  rgb (0.10196078431372549, 0.596078431372549, 0.3137254901960784) ,
  rgb (0.0, 0.40784313725490196, 0.21568627450980393)
});


list_data Reds = list_data(new pen[] {
  rgb (1.0, 0.9607843137254902, 0.9411764705882353) ,
  rgb (0.996078431372549, 0.8784313725490196, 0.8235294117647058) ,
  rgb (0.9882352941176471, 0.7333333333333333, 0.6313725490196078) ,
  rgb (0.9882352941176471, 0.5725490196078431, 0.4470588235294118) ,
  rgb (0.984313725490196, 0.41568627450980394, 0.2901960784313726) ,
  rgb (0.9372549019607843, 0.23137254901960785, 0.17254901960784313) ,
  rgb (0.796078431372549, 0.09411764705882353, 0.11372549019607843) ,
  rgb (0.6470588235294118, 0.058823529411764705, 0.08235294117647057) ,
  rgb (0.403921568627451, 0.0, 0.05098039215686274)
});


list_data Set1 = list_data(new pen[] {
  rgb (0.8941176470588236, 0.10196078431372549, 0.10980392156862745) ,
  rgb (0.21568627450980393, 0.49411764705882355, 0.7215686274509804) ,
  rgb (0.30196078431372547, 0.6862745098039216, 0.2901960784313726) ,
  rgb (0.596078431372549, 0.3058823529411765, 0.6392156862745098) ,
  rgb (1.0, 0.4980392156862745, 0.0) ,
  rgb (1.0, 1.0, 0.2) ,
  rgb (0.6509803921568628, 0.33725490196078434, 0.1568627450980392) ,
  rgb (0.9686274509803922, 0.5058823529411764, 0.7490196078431373) ,
  rgb (0.6, 0.6, 0.6)
});


list_data Set2 = list_data(new pen[] {
  rgb (0.4, 0.7607843137254902, 0.6470588235294118) ,
  rgb (0.9882352941176471, 0.5529411764705883, 0.3843137254901961) ,
  rgb (0.5529411764705883, 0.6274509803921569, 0.796078431372549) ,
  rgb (0.9058823529411765, 0.5411764705882353, 0.7647058823529411) ,
  rgb (0.6509803921568628, 0.8470588235294118, 0.32941176470588235) ,
  rgb (1.0, 0.8509803921568627, 0.1843137254901961) ,
  rgb (0.8980392156862745, 0.7686274509803922, 0.5803921568627451) ,
  rgb (0.7019607843137254, 0.7019607843137254, 0.7019607843137254)
});


list_data Set3 = list_data(new pen[] {
  rgb (0.5529411764705883, 0.8274509803921568, 0.7803921568627451) ,
  rgb (1.0, 1.0, 0.7019607843137254) ,
  rgb (0.7450980392156863, 0.7294117647058823, 0.8549019607843137) ,
  rgb (0.984313725490196, 0.5019607843137255, 0.4470588235294118) ,
  rgb (0.5019607843137255, 0.6941176470588235, 0.8274509803921568) ,
  rgb (0.9921568627450981, 0.7058823529411765, 0.3843137254901961) ,
  rgb (0.7019607843137254, 0.8705882352941177, 0.4117647058823529) ,
  rgb (0.9882352941176471, 0.803921568627451, 0.8980392156862745) ,
  rgb (0.8509803921568627, 0.8509803921568627, 0.8509803921568627) ,
  rgb (0.7372549019607844, 0.5019607843137255, 0.7411764705882353) ,
  rgb (0.8, 0.9215686274509803, 0.7725490196078432) ,
  rgb (1.0, 0.9294117647058824, 0.43529411764705883)
});


list_data Spectral = list_data(new pen[] {
  rgb (0.6196078431372549, 0.00392156862745098, 0.25882352941176473) ,
  rgb (0.8352941176470589, 0.24313725490196078, 0.30980392156862746) ,
  rgb (0.9568627450980393, 0.42745098039215684, 0.2627450980392157) ,
  rgb (0.9921568627450981, 0.6823529411764706, 0.3803921568627451) ,
  rgb (0.996078431372549, 0.8784313725490196, 0.5450980392156862) ,
  rgb (1.0, 1.0, 0.7490196078431373) ,
  rgb (0.9019607843137255, 0.9607843137254902, 0.596078431372549) ,
  rgb (0.6705882352941176, 0.8666666666666667, 0.6431372549019608) ,
  rgb (0.4, 0.7607843137254902, 0.6470588235294118) ,
  rgb (0.19607843137254902, 0.5333333333333333, 0.7411764705882353) ,
  rgb (0.3686274509803922, 0.30980392156862746, 0.6352941176470588)
});


list_data YlGnBu = list_data(new pen[] {
  rgb (1.0, 1.0, 0.8509803921568627) ,
  rgb (0.9294117647058824, 0.9725490196078431, 0.6941176470588235) ,
  rgb (0.7803921568627451, 0.9137254901960784, 0.7058823529411765) ,
  rgb (0.4980392156862745, 0.803921568627451, 0.7333333333333333) ,
  rgb (0.2549019607843137, 0.7137254901960784, 0.7686274509803922) ,
  rgb (0.11372549019607843, 0.5686274509803921, 0.7529411764705882) ,
  rgb (0.13333333333333333, 0.3686274509803922, 0.6588235294117647) ,
  rgb (0.1450980392156863, 0.20392156862745098, 0.5803921568627451) ,
  rgb (0.03137254901960784, 0.11372549019607843, 0.34509803921568627)
});


list_data YlGn = list_data(new pen[] {
  rgb (1.0, 1.0, 0.8980392156862745) ,
  rgb (0.9686274509803922, 0.9882352941176471, 0.7254901960784313) ,
  rgb (0.8509803921568627, 0.9411764705882353, 0.6392156862745098) ,
  rgb (0.6784313725490196, 0.8666666666666667, 0.5568627450980392) ,
  rgb (0.47058823529411764, 0.7764705882352941, 0.4745098039215686) ,
  rgb (0.2549019607843137, 0.6705882352941176, 0.36470588235294116) ,
  rgb (0.13725490196078433, 0.5176470588235295, 0.2627450980392157) ,
  rgb (0.0, 0.40784313725490196, 0.21568627450980393) ,
  rgb (0.0, 0.27058823529411763, 0.1607843137254902)
});


list_data YlOrBr = list_data(new pen[] {
  rgb (1.0, 1.0, 0.8980392156862745) ,
  rgb (1.0, 0.9686274509803922, 0.7372549019607844) ,
  rgb (0.996078431372549, 0.8901960784313725, 0.5686274509803921) ,
  rgb (0.996078431372549, 0.7686274509803922, 0.30980392156862746) ,
  rgb (0.996078431372549, 0.6, 0.1607843137254902) ,
  rgb (0.9254901960784314, 0.4392156862745098, 0.0784313725490196) ,
  rgb (0.8, 0.2980392156862745, 0.00784313725490196) ,
  rgb (0.6, 0.20392156862745098, 0.01568627450980392) ,
  rgb (0.4, 0.1450980392156863, 0.02352941176470588)
});


list_data YlOrRd = list_data(new pen[] {
  rgb (1.0, 1.0, 0.8) ,
  rgb (1.0, 0.9294117647058824, 0.6274509803921569) ,
  rgb (0.996078431372549, 0.8509803921568627, 0.4627450980392157) ,
  rgb (0.996078431372549, 0.6980392156862745, 0.2980392156862745) ,
  rgb (0.9921568627450981, 0.5529411764705883, 0.23529411764705882) ,
  rgb (0.9882352941176471, 0.3058823529411765, 0.16470588235294117) ,
  rgb (0.8901960784313725, 0.10196078431372549, 0.10980392156862745) ,
  rgb (0.7411764705882353, 0.0, 0.14901960784313725) ,
  rgb (0.5019607843137255, 0.0, 0.14901960784313725)
});


seg_data autumn = seg_data(
  new triple[] {  // red
     (0.0, 1.0, 1.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (1.0, 0.0, 0.0)
  }
);


seg_data binary = seg_data(
  new triple[] {  // red
     (0.0, 1.0, 1.0) ,
     (1.0, 0.0, 0.0)
  },
  new triple[] {  // green
     (0.0, 1.0, 1.0) ,
     (1.0, 0.0, 0.0)
  },
  new triple[] {  // blue
     (0.0, 1.0, 1.0) ,
     (1.0, 0.0, 0.0)
  }
);


seg_data bone = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (0.746032, 0.652778, 0.652778) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.365079, 0.319444, 0.319444) ,
     (0.746032, 0.777778, 0.777778) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.365079, 0.444444, 0.444444) ,
     (1.0, 1.0, 1.0)
  }
);


list_data brg = list_data(new pen[] {
  rgb (0.0, 0.0, 1.0) ,
  rgb (1.0, 0.0, 0.0) ,
  rgb (0.0, 1.0, 0.0)
});


list_data bwr = list_data(new pen[] {
  rgb (0.0, 0.0, 1.0) ,
  rgb (1.0, 1.0, 1.0) ,
  rgb (1.0, 0.0, 0.0)
});


seg_data cool = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 1.0, 1.0) ,
     (1.0, 0.0, 0.0)
  },
  new triple[] {  // blue
     (0.0, 1.0, 1.0) ,
     (1.0, 1.0, 1.0)
  }
);


seg_data coolwarm = seg_data(
  new triple[] {  // red
     (0.0, 0.2298057, 0.2298057) ,
     (0.03125, 0.26623388, 0.26623388) ,
     (0.0625, 0.30386891, 0.30386891) ,
     (0.09375, 0.342804478, 0.342804478) ,
     (0.125, 0.38301334, 0.38301334) ,
     (0.15625, 0.424369608, 0.424369608) ,
     (0.1875, 0.46666708, 0.46666708) ,
     (0.21875, 0.509635204, 0.509635204) ,
     (0.25, 0.552953156, 0.552953156) ,
     (0.28125, 0.596262162, 0.596262162) ,
     (0.3125, 0.639176211, 0.639176211) ,
     (0.34375, 0.681291281, 0.681291281) ,
     (0.375, 0.722193294, 0.722193294) ,
     (0.40625, 0.761464949, 0.761464949) ,
     (0.4375, 0.798691636, 0.798691636) ,
     (0.46875, 0.833466556, 0.833466556) ,
     (0.5, 0.865395197, 0.865395197) ,
     (0.53125, 0.897787179, 0.897787179) ,
     (0.5625, 0.924127593, 0.924127593) ,
     (0.59375, 0.944468518, 0.944468518) ,
     (0.625, 0.958852946, 0.958852946) ,
     (0.65625, 0.96732803, 0.96732803) ,
     (0.6875, 0.969954137, 0.969954137) ,
     (0.71875, 0.966811177, 0.966811177) ,
     (0.75, 0.958003065, 0.958003065) ,
     (0.78125, 0.943660866, 0.943660866) ,
     (0.8125, 0.923944917, 0.923944917) ,
     (0.84375, 0.89904617, 0.89904617) ,
     (0.875, 0.869186849, 0.869186849) ,
     (0.90625, 0.834620542, 0.834620542) ,
     (0.9375, 0.795631745, 0.795631745) ,
     (0.96875, 0.752534934, 0.752534934) ,
     (1.0, 0.705673158, 0.705673158)
  },
  new triple[] {  // green
     (0.0, 0.298717966, 0.298717966) ,
     (0.03125, 0.353094838, 0.353094838) ,
     (0.0625, 0.406535296, 0.406535296) ,
     (0.09375, 0.458757618, 0.458757618) ,
     (0.125, 0.50941904, 0.50941904) ,
     (0.15625, 0.558148092, 0.558148092) ,
     (0.1875, 0.604562568, 0.604562568) ,
     (0.21875, 0.648280772, 0.648280772) ,
     (0.25, 0.688929332, 0.688929332) ,
     (0.28125, 0.726149107, 0.726149107) ,
     (0.3125, 0.759599947, 0.759599947) ,
     (0.34375, 0.788964712, 0.788964712) ,
     (0.375, 0.813952739, 0.813952739) ,
     (0.40625, 0.834302879, 0.834302879) ,
     (0.4375, 0.849786142, 0.849786142) ,
     (0.46875, 0.860207984, 0.860207984) ,
     (0.5, 0.86541021, 0.86541021) ,
     (0.53125, 0.848937047, 0.848937047) ,
     (0.5625, 0.827384882, 0.827384882) ,
     (0.59375, 0.800927443, 0.800927443) ,
     (0.625, 0.769767752, 0.769767752) ,
     (0.65625, 0.734132809, 0.734132809) ,
     (0.6875, 0.694266682, 0.694266682) ,
     (0.71875, 0.650421156, 0.650421156) ,
     (0.75, 0.602842431, 0.602842431) ,
     (0.78125, 0.551750968, 0.551750968) ,
     (0.8125, 0.49730856, 0.49730856) ,
     (0.84375, 0.439559467, 0.439559467) ,
     (0.875, 0.378313092, 0.378313092) ,
     (0.90625, 0.312874446, 0.312874446) ,
     (0.9375, 0.24128379, 0.24128379) ,
     (0.96875, 0.157246067, 0.157246067) ,
     (1.0, 0.01555616, 0.01555616)
  },
  new triple[] {  // blue
     (0.0, 0.753683153, 0.753683153) ,
     (0.03125, 0.801466763, 0.801466763) ,
     (0.0625, 0.84495867, 0.84495867) ,
     (0.09375, 0.883725899, 0.883725899) ,
     (0.125, 0.917387822, 0.917387822) ,
     (0.15625, 0.945619588, 0.945619588) ,
     (0.1875, 0.968154911, 0.968154911) ,
     (0.21875, 0.98478814, 0.98478814) ,
     (0.25, 0.995375608, 0.995375608) ,
     (0.28125, 0.999836203, 0.999836203) ,
     (0.3125, 0.998151185, 0.998151185) ,
     (0.34375, 0.990363227, 0.990363227) ,
     (0.375, 0.976574709, 0.976574709) ,
     (0.40625, 0.956945269, 0.956945269) ,
     (0.4375, 0.931688648, 0.931688648) ,
     (0.46875, 0.901068838, 0.901068838) ,
     (0.5, 0.865395561, 0.865395561) ,
     (0.53125, 0.820880546, 0.820880546) ,
     (0.5625, 0.774508472, 0.774508472) ,
     (0.59375, 0.726736146, 0.726736146) ,
     (0.625, 0.678007945, 0.678007945) ,
     (0.65625, 0.628751763, 0.628751763) ,
     (0.6875, 0.579375448, 0.579375448) ,
     (0.71875, 0.530263762, 0.530263762) ,
     (0.75, 0.481775914, 0.481775914) ,
     (0.78125, 0.434243684, 0.434243684) ,
     (0.8125, 0.387970225, 0.387970225) ,
     (0.84375, 0.343229596, 0.343229596) ,
     (0.875, 0.300267182, 0.300267182) ,
     (0.90625, 0.259301199, 0.259301199) ,
     (0.9375, 0.220525627, 0.220525627) ,
     (0.96875, 0.184115123, 0.184115123) ,
     (1.0, 0.150232812, 0.150232812)
  }
);


seg_data copper = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (0.809524, 1.0, 1.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (1.0, 0.7812, 0.7812)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (1.0, 0.4975, 0.4975)
  }
);


seg_data gist_earth = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (0.2824, 0.1882, 0.1882) ,
     (0.4588, 0.2714, 0.2714) ,
     (0.549, 0.4719, 0.4719) ,
     (0.698, 0.7176, 0.7176) ,
     (0.7882, 0.7553, 0.7553) ,
     (1.0, 0.9922, 0.9922)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.0275, 0.0, 0.0) ,
     (0.1098, 0.1893, 0.1893) ,
     (0.1647, 0.3035, 0.3035) ,
     (0.2078, 0.3841, 0.3841) ,
     (0.2824, 0.502, 0.502) ,
     (0.5216, 0.6397, 0.6397) ,
     (0.698, 0.7171, 0.7171) ,
     (0.7882, 0.6392, 0.6392) ,
     (0.7922, 0.6413, 0.6413) ,
     (0.8, 0.6447, 0.6447) ,
     (0.8078, 0.6481, 0.6481) ,
     (0.8157, 0.6549, 0.6549) ,
     (0.8667, 0.6991, 0.6991) ,
     (0.8745, 0.7103, 0.7103) ,
     (0.8824, 0.7216, 0.7216) ,
     (0.8902, 0.7323, 0.7323) ,
     (0.898, 0.743, 0.743) ,
     (0.9412, 0.8275, 0.8275) ,
     (0.9569, 0.8635, 0.8635) ,
     (0.9647, 0.8816, 0.8816) ,
     (0.9961, 0.9733, 0.9733) ,
     (1.0, 0.9843, 0.9843)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.0039, 0.1684, 0.1684) ,
     (0.0078, 0.2212, 0.2212) ,
     (0.0275, 0.4329, 0.4329) ,
     (0.0314, 0.4549, 0.4549) ,
     (0.2824, 0.5004, 0.5004) ,
     (0.4667, 0.2748, 0.2748) ,
     (0.5451, 0.3205, 0.3205) ,
     (0.7843, 0.3961, 0.3961) ,
     (0.8941, 0.6651, 0.6651) ,
     (1.0, 0.9843, 0.9843)
  }
);


seg_data gist_ncar = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (0.3098, 0.0, 0.0) ,
     (0.3725, 0.3993, 0.3993) ,
     (0.4235, 0.5003, 0.5003) ,
     (0.5333, 1.0, 1.0) ,
     (0.7922, 1.0, 1.0) ,
     (0.8471, 0.6218, 0.6218) ,
     (0.898, 0.9235, 0.9235) ,
     (1.0, 0.9961, 0.9961)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.051, 0.3722, 0.3722) ,
     (0.1059, 0.0, 0.0) ,
     (0.1569, 0.7202, 0.7202) ,
     (0.1608, 0.7537, 0.7537) ,
     (0.1647, 0.7752, 0.7752) ,
     (0.2157, 1.0, 1.0) ,
     (0.2588, 0.9804, 0.9804) ,
     (0.2706, 0.9804, 0.9804) ,
     (0.3176, 1.0, 1.0) ,
     (0.3686, 0.8081, 0.8081) ,
     (0.4275, 1.0, 1.0) ,
     (0.5216, 1.0, 1.0) ,
     (0.6314, 0.7292, 0.7292) ,
     (0.6863, 0.2796, 0.2796) ,
     (0.7451, 0.0, 0.0) ,
     (0.7922, 0.0, 0.0) ,
     (0.8431, 0.1753, 0.1753) ,
     (0.898, 0.5, 0.5) ,
     (1.0, 0.9725, 0.9725)
  },
  new triple[] {  // blue
     (0.0, 0.502, 0.502) ,
     (0.051, 0.0222, 0.0222) ,
     (0.1098, 1.0, 1.0) ,
     (0.2039, 1.0, 1.0) ,
     (0.2627, 0.6145, 0.6145) ,
     (0.3216, 0.0, 0.0) ,
     (0.4157, 0.0, 0.0) ,
     (0.4745, 0.2342, 0.2342) ,
     (0.5333, 0.0, 0.0) ,
     (0.5804, 0.0, 0.0) ,
     (0.6314, 0.0549, 0.0549) ,
     (0.6902, 0.0, 0.0) ,
     (0.7373, 0.0, 0.0) ,
     (0.7922, 0.9738, 0.9738) ,
     (0.8, 1.0, 1.0) ,
     (0.8431, 1.0, 1.0) ,
     (0.898, 0.9341, 0.9341) ,
     (1.0, 0.9961, 0.9961)
  }
);


seg_data gist_stern = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (0.0547, 1.0, 1.0) ,
     (0.25, 0.027, 0.25) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0, 0, 0) ,
     (1, 1, 1)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.5, 1.0, 1.0) ,
     (0.735, 0.0, 0.0) ,
     (1.0, 1.0, 1.0)
  }
);


seg_data gray = seg_data(
  new triple[] {  // red
     (0.0, 0, 0) ,
     (1.0, 1, 1)
  },
  new triple[] {  // green
     (0.0, 0, 0) ,
     (1.0, 1, 1)
  },
  new triple[] {  // blue
     (0.0, 0, 0) ,
     (1.0, 1, 1)
  }
);


seg_data hot = seg_data(
  new triple[] {  // red
     (0.0, 0.0416, 0.0416) ,
     (0.365079, 1.0, 1.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.365079, 0.0, 0.0) ,
     (0.746032, 1.0, 1.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.746032, 0.0, 0.0) ,
     (1.0, 1.0, 1.0)
  }
);


seg_data hsv = seg_data(
  new triple[] {  // red
     (0.0, 1.0, 1.0) ,
     (0.15873, 1.0, 1.0) ,
     (0.174603, 0.96875, 0.96875) ,
     (0.333333, 0.03125, 0.03125) ,
     (0.349206, 0.0, 0.0) ,
     (0.666667, 0.0, 0.0) ,
     (0.68254, 0.03125, 0.03125) ,
     (0.84127, 0.96875, 0.96875) ,
     (0.857143, 1.0, 1.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.15873, 0.9375, 0.9375) ,
     (0.174603, 1.0, 1.0) ,
     (0.507937, 1.0, 1.0) ,
     (0.666667, 0.0625, 0.0625) ,
     (0.68254, 0.0, 0.0) ,
     (1.0, 0.0, 0.0)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.333333, 0.0, 0.0) ,
     (0.349206, 0.0625, 0.0625) ,
     (0.507937, 1.0, 1.0) ,
     (0.84127, 1.0, 1.0) ,
     (0.857143, 0.9375, 0.9375) ,
     (1.0, 0.09375, 0.09375)
  }
);


seg_data jet = seg_data(
  new triple[] {  // red
     (0.0, 0, 0) ,
     (0.35, 0, 0) ,
     (0.66, 1, 1) ,
     (0.89, 1, 1) ,
     (1, 0.5, 0.5)
  },
  new triple[] {  // green
     (0.0, 0, 0) ,
     (0.125, 0, 0) ,
     (0.375, 1, 1) ,
     (0.64, 1, 1) ,
     (0.91, 0, 0) ,
     (1, 0, 0)
  },
  new triple[] {  // blue
     (0.0, 0.5, 0.5) ,
     (0.11, 1, 1) ,
     (0.34, 1, 1) ,
     (0.65, 0, 0) ,
     (1, 0, 0)
  }
);


seg_data nipy_spectral = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (0.05, 0.4667, 0.4667) ,
     (0.1, 0.5333, 0.5333) ,
     (0.15, 0.0, 0.0) ,
     (0.2, 0.0, 0.0) ,
     (0.25, 0.0, 0.0) ,
     (0.3, 0.0, 0.0) ,
     (0.35, 0.0, 0.0) ,
     (0.4, 0.0, 0.0) ,
     (0.45, 0.0, 0.0) ,
     (0.5, 0.0, 0.0) ,
     (0.55, 0.0, 0.0) ,
     (0.6, 0.0, 0.0) ,
     (0.65, 0.7333, 0.7333) ,
     (0.7, 0.9333, 0.9333) ,
     (0.75, 1.0, 1.0) ,
     (0.8, 1.0, 1.0) ,
     (0.85, 1.0, 1.0) ,
     (0.9, 0.8667, 0.8667) ,
     (0.95, 0.8, 0.8) ,
     (1.0, 0.8, 0.8)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.05, 0.0, 0.0) ,
     (0.1, 0.0, 0.0) ,
     (0.15, 0.0, 0.0) ,
     (0.2, 0.0, 0.0) ,
     (0.25, 0.4667, 0.4667) ,
     (0.3, 0.6, 0.6) ,
     (0.35, 0.6667, 0.6667) ,
     (0.4, 0.6667, 0.6667) ,
     (0.45, 0.6, 0.6) ,
     (0.5, 0.7333, 0.7333) ,
     (0.55, 0.8667, 0.8667) ,
     (0.6, 1.0, 1.0) ,
     (0.65, 1.0, 1.0) ,
     (0.7, 0.9333, 0.9333) ,
     (0.75, 0.8, 0.8) ,
     (0.8, 0.6, 0.6) ,
     (0.85, 0.0, 0.0) ,
     (0.9, 0.0, 0.0) ,
     (0.95, 0.0, 0.0) ,
     (1.0, 0.8, 0.8)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.05, 0.5333, 0.5333) ,
     (0.1, 0.6, 0.6) ,
     (0.15, 0.6667, 0.6667) ,
     (0.2, 0.8667, 0.8667) ,
     (0.25, 0.8667, 0.8667) ,
     (0.3, 0.8667, 0.8667) ,
     (0.35, 0.6667, 0.6667) ,
     (0.4, 0.5333, 0.5333) ,
     (0.45, 0.0, 0.0) ,
     (0.5, 0.0, 0.0) ,
     (0.55, 0.0, 0.0) ,
     (0.6, 0.0, 0.0) ,
     (0.65, 0.0, 0.0) ,
     (0.7, 0.0, 0.0) ,
     (0.75, 0.0, 0.0) ,
     (0.8, 0.0, 0.0) ,
     (0.85, 0.0, 0.0) ,
     (0.9, 0.0, 0.0) ,
     (0.95, 0.0, 0.0) ,
     (1.0, 0.8, 0.8)
  }
);


seg_data pink = seg_data(
  new triple[] {  // red
     (0.0, 0.1178, 0.1178) ,
     (0.015873, 0.195857, 0.195857) ,
     (0.031746, 0.250661, 0.250661) ,
     (0.047619, 0.295468, 0.295468) ,
     (0.063492, 0.334324, 0.334324) ,
     (0.079365, 0.369112, 0.369112) ,
     (0.095238, 0.400892, 0.400892) ,
     (0.111111, 0.430331, 0.430331) ,
     (0.126984, 0.457882, 0.457882) ,
     (0.142857, 0.483867, 0.483867) ,
     (0.15873, 0.508525, 0.508525) ,
     (0.174603, 0.532042, 0.532042) ,
     (0.190476, 0.554563, 0.554563) ,
     (0.206349, 0.576204, 0.576204) ,
     (0.222222, 0.597061, 0.597061) ,
     (0.238095, 0.617213, 0.617213) ,
     (0.253968, 0.636729, 0.636729) ,
     (0.269841, 0.655663, 0.655663) ,
     (0.285714, 0.674066, 0.674066) ,
     (0.301587, 0.69198, 0.69198) ,
     (0.31746, 0.709441, 0.709441) ,
     (0.333333, 0.726483, 0.726483) ,
     (0.349206, 0.743134, 0.743134) ,
     (0.365079, 0.759421, 0.759421) ,
     (0.380952, 0.766356, 0.766356) ,
     (0.396825, 0.773229, 0.773229) ,
     (0.412698, 0.780042, 0.780042) ,
     (0.428571, 0.786796, 0.786796) ,
     (0.444444, 0.793492, 0.793492) ,
     (0.460317, 0.800132, 0.800132) ,
     (0.47619, 0.806718, 0.806718) ,
     (0.492063, 0.81325, 0.81325) ,
     (0.507937, 0.81973, 0.81973) ,
     (0.52381, 0.82616, 0.82616) ,
     (0.539683, 0.832539, 0.832539) ,
     (0.555556, 0.83887, 0.83887) ,
     (0.571429, 0.845154, 0.845154) ,
     (0.587302, 0.851392, 0.851392) ,
     (0.603175, 0.857584, 0.857584) ,
     (0.619048, 0.863731, 0.863731) ,
     (0.634921, 0.869835, 0.869835) ,
     (0.650794, 0.875897, 0.875897) ,
     (0.666667, 0.881917, 0.881917) ,
     (0.68254, 0.887896, 0.887896) ,
     (0.698413, 0.893835, 0.893835) ,
     (0.714286, 0.899735, 0.899735) ,
     (0.730159, 0.905597, 0.905597) ,
     (0.746032, 0.911421, 0.911421) ,
     (0.761905, 0.917208, 0.917208) ,
     (0.777778, 0.922958, 0.922958) ,
     (0.793651, 0.928673, 0.928673) ,
     (0.809524, 0.934353, 0.934353) ,
     (0.825397, 0.939999, 0.939999) ,
     (0.84127, 0.945611, 0.945611) ,
     (0.857143, 0.95119, 0.95119) ,
     (0.873016, 0.956736, 0.956736) ,
     (0.888889, 0.96225, 0.96225) ,
     (0.904762, 0.967733, 0.967733) ,
     (0.920635, 0.973185, 0.973185) ,
     (0.936508, 0.978607, 0.978607) ,
     (0.952381, 0.983999, 0.983999) ,
     (0.968254, 0.989361, 0.989361) ,
     (0.984127, 0.994695, 0.994695) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (0.015873, 0.102869, 0.102869) ,
     (0.031746, 0.145479, 0.145479) ,
     (0.047619, 0.178174, 0.178174) ,
     (0.063492, 0.205738, 0.205738) ,
     (0.079365, 0.230022, 0.230022) ,
     (0.095238, 0.251976, 0.251976) ,
     (0.111111, 0.272166, 0.272166) ,
     (0.126984, 0.290957, 0.290957) ,
     (0.142857, 0.308607, 0.308607) ,
     (0.15873, 0.3253, 0.3253) ,
     (0.174603, 0.341178, 0.341178) ,
     (0.190476, 0.356348, 0.356348) ,
     (0.206349, 0.370899, 0.370899) ,
     (0.222222, 0.3849, 0.3849) ,
     (0.238095, 0.39841, 0.39841) ,
     (0.253968, 0.411476, 0.411476) ,
     (0.269841, 0.424139, 0.424139) ,
     (0.285714, 0.436436, 0.436436) ,
     (0.301587, 0.448395, 0.448395) ,
     (0.31746, 0.460044, 0.460044) ,
     (0.333333, 0.471405, 0.471405) ,
     (0.349206, 0.482498, 0.482498) ,
     (0.365079, 0.493342, 0.493342) ,
     (0.380952, 0.517549, 0.517549) ,
     (0.396825, 0.540674, 0.540674) ,
     (0.412698, 0.562849, 0.562849) ,
     (0.428571, 0.584183, 0.584183) ,
     (0.444444, 0.604765, 0.604765) ,
     (0.460317, 0.624669, 0.624669) ,
     (0.47619, 0.643958, 0.643958) ,
     (0.492063, 0.662687, 0.662687) ,
     (0.507937, 0.6809, 0.6809) ,
     (0.52381, 0.698638, 0.698638) ,
     (0.539683, 0.715937, 0.715937) ,
     (0.555556, 0.732828, 0.732828) ,
     (0.571429, 0.749338, 0.749338) ,
     (0.587302, 0.765493, 0.765493) ,
     (0.603175, 0.781313, 0.781313) ,
     (0.619048, 0.796819, 0.796819) ,
     (0.634921, 0.812029, 0.812029) ,
     (0.650794, 0.82696, 0.82696) ,
     (0.666667, 0.841625, 0.841625) ,
     (0.68254, 0.85604, 0.85604) ,
     (0.698413, 0.870216, 0.870216) ,
     (0.714286, 0.884164, 0.884164) ,
     (0.730159, 0.897896, 0.897896) ,
     (0.746032, 0.911421, 0.911421) ,
     (0.761905, 0.917208, 0.917208) ,
     (0.777778, 0.922958, 0.922958) ,
     (0.793651, 0.928673, 0.928673) ,
     (0.809524, 0.934353, 0.934353) ,
     (0.825397, 0.939999, 0.939999) ,
     (0.84127, 0.945611, 0.945611) ,
     (0.857143, 0.95119, 0.95119) ,
     (0.873016, 0.956736, 0.956736) ,
     (0.888889, 0.96225, 0.96225) ,
     (0.904762, 0.967733, 0.967733) ,
     (0.920635, 0.973185, 0.973185) ,
     (0.936508, 0.978607, 0.978607) ,
     (0.952381, 0.983999, 0.983999) ,
     (0.968254, 0.989361, 0.989361) ,
     (0.984127, 0.994695, 0.994695) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 0.0, 0.0) ,
     (0.015873, 0.102869, 0.102869) ,
     (0.031746, 0.145479, 0.145479) ,
     (0.047619, 0.178174, 0.178174) ,
     (0.063492, 0.205738, 0.205738) ,
     (0.079365, 0.230022, 0.230022) ,
     (0.095238, 0.251976, 0.251976) ,
     (0.111111, 0.272166, 0.272166) ,
     (0.126984, 0.290957, 0.290957) ,
     (0.142857, 0.308607, 0.308607) ,
     (0.15873, 0.3253, 0.3253) ,
     (0.174603, 0.341178, 0.341178) ,
     (0.190476, 0.356348, 0.356348) ,
     (0.206349, 0.370899, 0.370899) ,
     (0.222222, 0.3849, 0.3849) ,
     (0.238095, 0.39841, 0.39841) ,
     (0.253968, 0.411476, 0.411476) ,
     (0.269841, 0.424139, 0.424139) ,
     (0.285714, 0.436436, 0.436436) ,
     (0.301587, 0.448395, 0.448395) ,
     (0.31746, 0.460044, 0.460044) ,
     (0.333333, 0.471405, 0.471405) ,
     (0.349206, 0.482498, 0.482498) ,
     (0.365079, 0.493342, 0.493342) ,
     (0.380952, 0.503953, 0.503953) ,
     (0.396825, 0.514344, 0.514344) ,
     (0.412698, 0.524531, 0.524531) ,
     (0.428571, 0.534522, 0.534522) ,
     (0.444444, 0.544331, 0.544331) ,
     (0.460317, 0.553966, 0.553966) ,
     (0.47619, 0.563436, 0.563436) ,
     (0.492063, 0.57275, 0.57275) ,
     (0.507937, 0.581914, 0.581914) ,
     (0.52381, 0.590937, 0.590937) ,
     (0.539683, 0.599824, 0.599824) ,
     (0.555556, 0.608581, 0.608581) ,
     (0.571429, 0.617213, 0.617213) ,
     (0.587302, 0.625727, 0.625727) ,
     (0.603175, 0.634126, 0.634126) ,
     (0.619048, 0.642416, 0.642416) ,
     (0.634921, 0.6506, 0.6506) ,
     (0.650794, 0.658682, 0.658682) ,
     (0.666667, 0.666667, 0.666667) ,
     (0.68254, 0.674556, 0.674556) ,
     (0.698413, 0.682355, 0.682355) ,
     (0.714286, 0.690066, 0.690066) ,
     (0.730159, 0.697691, 0.697691) ,
     (0.746032, 0.705234, 0.705234) ,
     (0.761905, 0.727166, 0.727166) ,
     (0.777778, 0.748455, 0.748455) ,
     (0.793651, 0.769156, 0.769156) ,
     (0.809524, 0.789314, 0.789314) ,
     (0.825397, 0.808969, 0.808969) ,
     (0.84127, 0.828159, 0.828159) ,
     (0.857143, 0.846913, 0.846913) ,
     (0.873016, 0.865261, 0.865261) ,
     (0.888889, 0.883229, 0.883229) ,
     (0.904762, 0.900837, 0.900837) ,
     (0.920635, 0.918109, 0.918109) ,
     (0.936508, 0.935061, 0.935061) ,
     (0.952381, 0.951711, 0.951711) ,
     (0.968254, 0.968075, 0.968075) ,
     (0.984127, 0.984167, 0.984167) ,
     (1.0, 1.0, 1.0)
  }
);


list_data seismic = list_data(new pen[] {
  rgb (0.0, 0.0, 0.3) ,
  rgb (0.0, 0.0, 1.0) ,
  rgb (1.0, 1.0, 1.0) ,
  rgb (1.0, 0.0, 0.0) ,
  rgb (0.5, 0.0, 0.0)
});


seg_data spring = seg_data(
  new triple[] {  // red
     (0.0, 1.0, 1.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 1.0, 1.0) ,
     (1.0, 0.0, 0.0)
  }
);


seg_data summer = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // green
     (0.0, 0.5, 0.5) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 0.4, 0.4) ,
     (1.0, 0.4, 0.4)
  }
);


list_data tab10 = list_data(new pen[] {
  rgb (0.12156862745098039, 0.4666666666666667, 0.7058823529411765) ,
  rgb (1.0, 0.4980392156862745, 0.054901960784313725) ,
  rgb (0.17254901960784313, 0.6274509803921569, 0.17254901960784313) ,
  rgb (0.8392156862745098, 0.15294117647058825, 0.1568627450980392) ,
  rgb (0.5803921568627451, 0.403921568627451, 0.7411764705882353) ,
  rgb (0.5490196078431373, 0.33725490196078434, 0.29411764705882354) ,
  rgb (0.8901960784313725, 0.4666666666666667, 0.7607843137254902) ,
  rgb (0.4980392156862745, 0.4980392156862745, 0.4980392156862745) ,
  rgb (0.7372549019607844, 0.7411764705882353, 0.13333333333333333) ,
  rgb (0.09019607843137255, 0.7450980392156863, 0.8117647058823529)
});


list_data tab20 = list_data(new pen[] {
  rgb (0.12156862745098039, 0.4666666666666667, 0.7058823529411765) ,
  rgb (0.6823529411764706, 0.7803921568627451, 0.9098039215686274) ,
  rgb (1.0, 0.4980392156862745, 0.054901960784313725) ,
  rgb (1.0, 0.7333333333333333, 0.47058823529411764) ,
  rgb (0.17254901960784313, 0.6274509803921569, 0.17254901960784313) ,
  rgb (0.596078431372549, 0.8745098039215686, 0.5411764705882353) ,
  rgb (0.8392156862745098, 0.15294117647058825, 0.1568627450980392) ,
  rgb (1.0, 0.596078431372549, 0.5882352941176471) ,
  rgb (0.5803921568627451, 0.403921568627451, 0.7411764705882353) ,
  rgb (0.7725490196078432, 0.6901960784313725, 0.8352941176470589) ,
  rgb (0.5490196078431373, 0.33725490196078434, 0.29411764705882354) ,
  rgb (0.7686274509803922, 0.611764705882353, 0.5803921568627451) ,
  rgb (0.8901960784313725, 0.4666666666666667, 0.7607843137254902) ,
  rgb (0.9686274509803922, 0.7137254901960784, 0.8235294117647058) ,
  rgb (0.4980392156862745, 0.4980392156862745, 0.4980392156862745) ,
  rgb (0.7803921568627451, 0.7803921568627451, 0.7803921568627451) ,
  rgb (0.7372549019607844, 0.7411764705882353, 0.13333333333333333) ,
  rgb (0.8588235294117647, 0.8588235294117647, 0.5529411764705883) ,
  rgb (0.09019607843137255, 0.7450980392156863, 0.8117647058823529) ,
  rgb (0.6196078431372549, 0.8549019607843137, 0.8980392156862745)
});


list_data tab20b = list_data(new pen[] {
  rgb (0.2235294117647059, 0.23137254901960785, 0.4745098039215686) ,
  rgb (0.3215686274509804, 0.32941176470588235, 0.6392156862745098) ,
  rgb (0.4196078431372549, 0.43137254901960786, 0.8117647058823529) ,
  rgb (0.611764705882353, 0.6196078431372549, 0.8705882352941177) ,
  rgb (0.38823529411764707, 0.4745098039215686, 0.2235294117647059) ,
  rgb (0.5490196078431373, 0.6352941176470588, 0.3215686274509804) ,
  rgb (0.7098039215686275, 0.8117647058823529, 0.4196078431372549) ,
  rgb (0.807843137254902, 0.8588235294117647, 0.611764705882353) ,
  rgb (0.5490196078431373, 0.42745098039215684, 0.19215686274509805) ,
  rgb (0.7411764705882353, 0.6196078431372549, 0.2235294117647059) ,
  rgb (0.9058823529411765, 0.7294117647058823, 0.3215686274509804) ,
  rgb (0.9058823529411765, 0.796078431372549, 0.5803921568627451) ,
  rgb (0.5176470588235295, 0.23529411764705882, 0.2235294117647059) ,
  rgb (0.6784313725490196, 0.28627450980392155, 0.2901960784313726) ,
  rgb (0.8392156862745098, 0.3803921568627451, 0.4196078431372549) ,
  rgb (0.9058823529411765, 0.5882352941176471, 0.611764705882353) ,
  rgb (0.4823529411764706, 0.2549019607843137, 0.45098039215686275) ,
  rgb (0.6470588235294118, 0.3176470588235294, 0.5803921568627451) ,
  rgb (0.807843137254902, 0.42745098039215684, 0.7411764705882353) ,
  rgb (0.8705882352941177, 0.6196078431372549, 0.8392156862745098)
});


list_data tab20c = list_data(new pen[] {
  rgb (0.19215686274509805, 0.5098039215686274, 0.7411764705882353) ,
  rgb (0.4196078431372549, 0.6823529411764706, 0.8392156862745098) ,
  rgb (0.6196078431372549, 0.792156862745098, 0.8823529411764706) ,
  rgb (0.7764705882352941, 0.8588235294117647, 0.9372549019607843) ,
  rgb (0.9019607843137255, 0.3333333333333333, 0.050980392156862744) ,
  rgb (0.9921568627450981, 0.5529411764705883, 0.23529411764705882) ,
  rgb (0.9921568627450981, 0.6823529411764706, 0.4196078431372549) ,
  rgb (0.9921568627450981, 0.8156862745098039, 0.6352941176470588) ,
  rgb (0.19215686274509805, 0.6392156862745098, 0.32941176470588235) ,
  rgb (0.4549019607843137, 0.7686274509803922, 0.4627450980392157) ,
  rgb (0.6313725490196078, 0.8509803921568627, 0.6078431372549019) ,
  rgb (0.7803921568627451, 0.9137254901960784, 0.7529411764705882) ,
  rgb (0.4588235294117647, 0.4196078431372549, 0.6941176470588235) ,
  rgb (0.6196078431372549, 0.6039215686274509, 0.7843137254901961) ,
  rgb (0.7372549019607844, 0.7411764705882353, 0.8627450980392157) ,
  rgb (0.8549019607843137, 0.8549019607843137, 0.9215686274509803) ,
  rgb (0.38823529411764707, 0.38823529411764707, 0.38823529411764707) ,
  rgb (0.5882352941176471, 0.5882352941176471, 0.5882352941176471) ,
  rgb (0.7411764705882353, 0.7411764705882353, 0.7411764705882353) ,
  rgb (0.8509803921568627, 0.8509803921568627, 0.8509803921568627)
});


seg_data winter = seg_data(
  new triple[] {  // red
     (0.0, 0.0, 0.0) ,
     (1.0, 0.0, 0.0)
  },
  new triple[] {  // green
     (0.0, 0.0, 0.0) ,
     (1.0, 1.0, 1.0)
  },
  new triple[] {  // blue
     (0.0, 1.0, 1.0) ,
     (1.0, 0.5, 0.5)
  }
);


seg_data wistia = seg_data(
  new triple[] {  // red
     (0.0, 0.8941176470588236, 0.8941176470588236) ,
     (0.25, 1.0, 1.0) ,
     (0.5, 1.0, 1.0) ,
     (0.75, 1.0, 1.0) ,
     (1.0, 0.9882352941176471, 0.9882352941176471)
  },
  new triple[] {  // green
     (0.0, 1.0, 1.0) ,
     (0.25, 0.9098039215686274, 0.9098039215686274) ,
     (0.5, 0.7411764705882353, 0.7411764705882353) ,
     (0.75, 0.6274509803921569, 0.6274509803921569) ,
     (1.0, 0.4980392156862745, 0.4980392156862745)
  },
  new triple[] {  // blue
     (0.0, 0.47843137254901963, 0.47843137254901963) ,
     (0.25, 0.10196078431372549, 0.10196078431372549) ,
     (0.5, 0.0, 0.0) ,
     (0.75, 0.0, 0.0) ,
     (1.0, 0.0, 0.0)
  }
);


list_data cividis = list_data(new pen[] {
  rgb (0.0, 0.135112, 0.304751) ,
  rgb (0.0, 0.138068, 0.311105) ,
  rgb (0.0, 0.141013, 0.317579) ,
  rgb (0.0, 0.143951, 0.323982) ,
  rgb (0.0, 0.146877, 0.330479) ,
  rgb (0.0, 0.149791, 0.337065) ,
  rgb (0.0, 0.152673, 0.343704) ,
  rgb (0.0, 0.155377, 0.3505) ,
  rgb (0.0, 0.157932, 0.357521) ,
  rgb (0.0, 0.160495, 0.364534) ,
  rgb (0.0, 0.163058, 0.371608) ,
  rgb (0.0, 0.165621, 0.378769) ,
  rgb (0.0, 0.168204, 0.385902) ,
  rgb (0.0, 0.1708, 0.3931) ,
  rgb (0.0, 0.17342, 0.400353) ,
  rgb (0.0, 0.176082, 0.407577) ,
  rgb (0.0, 0.178802, 0.414764) ,
  rgb (0.0, 0.18161, 0.421859) ,
  rgb (0.0, 0.18455, 0.428802) ,
  rgb (0.0, 0.186915, 0.435532) ,
  rgb (0.0, 0.188769, 0.439563) ,
  rgb (0.0, 0.19095, 0.441085) ,
  rgb (0.0, 0.193366, 0.441561) ,
  rgb (0.003602, 0.195911, 0.441564) ,
  rgb (0.017852, 0.198528, 0.441248) ,
  rgb (0.03211, 0.201199, 0.440785) ,
  rgb (0.046205, 0.203903, 0.440196) ,
  rgb (0.058378, 0.206629, 0.439531) ,
  rgb (0.068968, 0.209372, 0.438863) ,
  rgb (0.078624, 0.212122, 0.438105) ,
  rgb (0.087465, 0.214879, 0.437342) ,
  rgb (0.095645, 0.217643, 0.436593) ,
  rgb (0.103401, 0.220406, 0.43579) ,
  rgb (0.110658, 0.22317, 0.435067) ,
  rgb (0.117612, 0.225935, 0.434308) ,
  rgb (0.124291, 0.228697, 0.433547) ,
  rgb (0.130669, 0.231458, 0.43284) ,
  rgb (0.13683, 0.234216, 0.432148) ,
  rgb (0.142852, 0.236972, 0.431404) ,
  rgb (0.148638, 0.239724, 0.430752) ,
  rgb (0.154261, 0.242475, 0.43012) ,
  rgb (0.159733, 0.245221, 0.429528) ,
  rgb (0.165113, 0.247965, 0.428908) ,
  rgb (0.170362, 0.250707, 0.428325) ,
  rgb (0.17549, 0.253444, 0.42779) ,
  rgb (0.180503, 0.25618, 0.427299) ,
  rgb (0.185453, 0.258914, 0.426788) ,
  rgb (0.190303, 0.261644, 0.426329) ,
  rgb (0.195057, 0.264372, 0.425924) ,
  rgb (0.199764, 0.267099, 0.425497) ,
  rgb (0.204385, 0.269823, 0.425126) ,
  rgb (0.208926, 0.272546, 0.424809) ,
  rgb (0.213431, 0.275266, 0.42448) ,
  rgb (0.217863, 0.277985, 0.424206) ,
  rgb (0.222264, 0.280702, 0.423914) ,
  rgb (0.226598, 0.283419, 0.423678) ,
  rgb (0.230871, 0.286134, 0.423498) ,
  rgb (0.23512, 0.288848, 0.423304) ,
  rgb (0.239312, 0.291562, 0.423167) ,
  rgb (0.243485, 0.294274, 0.423014) ,
  rgb (0.247605, 0.296986, 0.422917) ,
  rgb (0.251675, 0.299698, 0.422873) ,
  rgb (0.255731, 0.302409, 0.422814) ,
  rgb (0.25974, 0.30512, 0.42281) ,
  rgb (0.263738, 0.307831, 0.422789) ,
  rgb (0.267693, 0.310542, 0.422821) ,
  rgb (0.271639, 0.313253, 0.422837) ,
  rgb (0.275513, 0.315965, 0.422979) ,
  rgb (0.279411, 0.318677, 0.423031) ,
  rgb (0.28324, 0.32139, 0.423211) ,
  rgb (0.287065, 0.324103, 0.423373) ,
  rgb (0.290884, 0.326816, 0.423517) ,
  rgb (0.294669, 0.329531, 0.423716) ,
  rgb (0.298421, 0.332247, 0.423973) ,
  rgb (0.302169, 0.334963, 0.424213) ,
  rgb (0.305886, 0.337681, 0.424512) ,
  rgb (0.309601, 0.340399, 0.42479) ,
  rgb (0.313287, 0.34312, 0.42512) ,
  rgb (0.316941, 0.345842, 0.425512) ,
  rgb (0.320595, 0.348565, 0.425889) ,
  rgb (0.32425, 0.351289, 0.42625) ,
  rgb (0.327875, 0.354016, 0.42667) ,
  rgb (0.331474, 0.356744, 0.427144) ,
  rgb (0.335073, 0.359474, 0.427605) ,
  rgb (0.338673, 0.362206, 0.428053) ,
  rgb (0.342246, 0.364939, 0.428559) ,
  rgb (0.345793, 0.367676, 0.429127) ,
  rgb (0.349341, 0.370414, 0.429685) ,
  rgb (0.352892, 0.373153, 0.430226) ,
  rgb (0.356418, 0.375896, 0.430823) ,
  rgb (0.359916, 0.378641, 0.431501) ,
  rgb (0.363446, 0.381388, 0.432075) ,
  rgb (0.366923, 0.384139, 0.432796) ,
  rgb (0.37043, 0.38689, 0.433428) ,
  rgb (0.373884, 0.389646, 0.434209) ,
  rgb (0.377371, 0.392404, 0.43489) ,
  rgb (0.38083, 0.395164, 0.435653) ,
  rgb (0.384268, 0.397928, 0.436475) ,
  rgb (0.387705, 0.400694, 0.437305) ,
  rgb (0.391151, 0.403464, 0.438096) ,
  rgb (0.394568, 0.406236, 0.438986) ,
  rgb (0.397991, 0.409011, 0.439848) ,
  rgb (0.401418, 0.41179, 0.440708) ,
  rgb (0.40482, 0.414572, 0.441642) ,
  rgb (0.408226, 0.417357, 0.44257) ,
  rgb (0.411607, 0.420145, 0.443577) ,
  rgb (0.414992, 0.422937, 0.444578) ,
  rgb (0.418383, 0.425733, 0.44556) ,
  rgb (0.421748, 0.428531, 0.44664) ,
  rgb (0.42512, 0.431334, 0.447692) ,
  rgb (0.428462, 0.43414, 0.448864) ,
  rgb (0.431817, 0.43695, 0.449982) ,
  rgb (0.435168, 0.439763, 0.451134) ,
  rgb (0.438504, 0.44258, 0.452341) ,
  rgb (0.44181, 0.445402, 0.453659) ,
  rgb (0.445148, 0.448226, 0.454885) ,
  rgb (0.448447, 0.451053, 0.456264) ,
  rgb (0.451759, 0.453887, 0.457582) ,
  rgb (0.455072, 0.456718, 0.458976) ,
  rgb (0.458366, 0.459552, 0.460457) ,
  rgb (0.461616, 0.462405, 0.461969) ,
  rgb (0.464947, 0.465241, 0.463395) ,
  rgb (0.468254, 0.468083, 0.464908) ,
  rgb (0.471501, 0.47096, 0.466357) ,
  rgb (0.474812, 0.473832, 0.467681) ,
  rgb (0.478186, 0.476699, 0.468845) ,
  rgb (0.481622, 0.479573, 0.469767) ,
  rgb (0.485141, 0.482451, 0.470384) ,
  rgb (0.488697, 0.485318, 0.471008) ,
  rgb (0.492278, 0.488198, 0.471453) ,
  rgb (0.495913, 0.491076, 0.471751) ,
  rgb (0.499552, 0.49396, 0.472032) ,
  rgb (0.503185, 0.496851, 0.472305) ,
  rgb (0.506866, 0.499743, 0.472432) ,
  rgb (0.51054, 0.502643, 0.47255) ,
  rgb (0.514226, 0.505546, 0.47264) ,
  rgb (0.51792, 0.508454, 0.472707) ,
  rgb (0.521643, 0.511367, 0.472639) ,
  rgb (0.525348, 0.514285, 0.47266) ,
  rgb (0.529086, 0.517207, 0.472543) ,
  rgb (0.532829, 0.520135, 0.472401) ,
  rgb (0.536553, 0.523067, 0.472352) ,
  rgb (0.540307, 0.526005, 0.472163) ,
  rgb (0.544069, 0.528948, 0.471947) ,
  rgb (0.54784, 0.531895, 0.471704) ,
  rgb (0.551612, 0.534849, 0.471439) ,
  rgb (0.555393, 0.537807, 0.471147) ,
  rgb (0.559181, 0.540771, 0.470829) ,
  rgb (0.562972, 0.543741, 0.470488) ,
  rgb (0.566802, 0.546715, 0.469988) ,
  rgb (0.570607, 0.549695, 0.469593) ,
  rgb (0.574417, 0.552682, 0.469172) ,
  rgb (0.578236, 0.555673, 0.468724) ,
  rgb (0.582087, 0.55867, 0.468118) ,
  rgb (0.585916, 0.561674, 0.467618) ,
  rgb (0.589753, 0.564682, 0.46709) ,
  rgb (0.593622, 0.567697, 0.466401) ,
  rgb (0.597469, 0.570718, 0.465821) ,
  rgb (0.601354, 0.573743, 0.465074) ,
  rgb (0.605211, 0.576777, 0.464441) ,
  rgb (0.609105, 0.579816, 0.463638) ,
  rgb (0.612977, 0.582861, 0.46295) ,
  rgb (0.616852, 0.585913, 0.462237) ,
  rgb (0.620765, 0.58897, 0.461351) ,
  rgb (0.624654, 0.592034, 0.460583) ,
  rgb (0.628576, 0.595104, 0.459641) ,
  rgb (0.632506, 0.59818, 0.458668) ,
  rgb (0.636412, 0.601264, 0.457818) ,
  rgb (0.640352, 0.604354, 0.456791) ,
  rgb (0.64427, 0.60745, 0.455886) ,
  rgb (0.648222, 0.610553, 0.454801) ,
  rgb (0.652178, 0.613664, 0.453689) ,
  rgb (0.656114, 0.61678, 0.452702) ,
  rgb (0.660082, 0.619904, 0.451534) ,
  rgb (0.664055, 0.623034, 0.450338) ,
  rgb (0.668008, 0.626171, 0.44927) ,
  rgb (0.671991, 0.629316, 0.448018) ,
  rgb (0.675981, 0.632468, 0.446736) ,
  rgb (0.679979, 0.635626, 0.445424) ,
  rgb (0.68395, 0.638793, 0.444251) ,
  rgb (0.687957, 0.641966, 0.442886) ,
  rgb (0.691971, 0.645145, 0.441491) ,
  rgb (0.695985, 0.648334, 0.440072) ,
  rgb (0.700008, 0.651529, 0.438624) ,
  rgb (0.704037, 0.654731, 0.437147) ,
  rgb (0.708067, 0.657942, 0.435647) ,
  rgb (0.712105, 0.66116, 0.434117) ,
  rgb (0.716177, 0.664384, 0.432386) ,
  rgb (0.720222, 0.667618, 0.430805) ,
  rgb (0.724274, 0.670859, 0.429194) ,
  rgb (0.728334, 0.674107, 0.427554) ,
  rgb (0.732422, 0.677364, 0.425717) ,
  rgb (0.736488, 0.680629, 0.424028) ,
  rgb (0.740589, 0.6839, 0.422131) ,
  rgb (0.744664, 0.687181, 0.420393) ,
  rgb (0.748772, 0.69047, 0.418448) ,
  rgb (0.752886, 0.693766, 0.416472) ,
  rgb (0.756975, 0.697071, 0.414659) ,
  rgb (0.761096, 0.700384, 0.412638) ,
  rgb (0.765223, 0.703705, 0.410587) ,
  rgb (0.769353, 0.707035, 0.408516) ,
  rgb (0.773486, 0.710373, 0.406422) ,
  rgb (0.777651, 0.713719, 0.404112) ,
  rgb (0.781795, 0.717074, 0.401966) ,
  rgb (0.785965, 0.720438, 0.399613) ,
  rgb (0.790116, 0.72381, 0.397423) ,
  rgb (0.794298, 0.72719, 0.395016) ,
  rgb (0.79848, 0.73058, 0.392597) ,
  rgb (0.802667, 0.733978, 0.390153) ,
  rgb (0.806859, 0.737385, 0.387684) ,
  rgb (0.811054, 0.740801, 0.385198) ,
  rgb (0.815274, 0.744226, 0.382504) ,
  rgb (0.819499, 0.747659, 0.379785) ,
  rgb (0.823729, 0.751101, 0.377043) ,
  rgb (0.827959, 0.754553, 0.374292) ,
  rgb (0.832192, 0.758014, 0.371529) ,
  rgb (0.836429, 0.761483, 0.368747) ,
  rgb (0.840693, 0.764962, 0.365746) ,
  rgb (0.844957, 0.76845, 0.362741) ,
  rgb (0.849223, 0.771947, 0.359729) ,
  rgb (0.853515, 0.775454, 0.3565) ,
  rgb (0.857809, 0.778969, 0.353259) ,
  rgb (0.862105, 0.782494, 0.350011) ,
  rgb (0.866421, 0.786028, 0.346571) ,
  rgb (0.870717, 0.789572, 0.343333) ,
  rgb (0.875057, 0.793125, 0.339685) ,
  rgb (0.879378, 0.796687, 0.336241) ,
  rgb (0.88372, 0.800258, 0.332599) ,
  rgb (0.888081, 0.803839, 0.32877) ,
  rgb (0.89244, 0.80743, 0.324968) ,
  rgb (0.896818, 0.81103, 0.320982) ,
  rgb (0.901195, 0.814639, 0.317021) ,
  rgb (0.905589, 0.818257, 0.312889) ,
  rgb (0.91, 0.821885, 0.308594) ,
  rgb (0.914407, 0.825522, 0.304348) ,
  rgb (0.918828, 0.829168, 0.29996) ,
  rgb (0.923279, 0.832822, 0.295244) ,
  rgb (0.927724, 0.836486, 0.290611) ,
  rgb (0.93218, 0.840159, 0.28588) ,
  rgb (0.93666, 0.843841, 0.280876) ,
  rgb (0.941147, 0.84753, 0.275815) ,
  rgb (0.945654, 0.851228, 0.270532) ,
  rgb (0.950178, 0.854933, 0.265085) ,
  rgb (0.954725, 0.858646, 0.259365) ,
  rgb (0.959284, 0.862365, 0.253563) ,
  rgb (0.963872, 0.866089, 0.247445) ,
  rgb (0.968469, 0.869819, 0.24131) ,
  rgb (0.973114, 0.87355, 0.234677) ,
  rgb (0.97778, 0.877281, 0.227954) ,
  rgb (0.982497, 0.881008, 0.220878) ,
  rgb (0.987293, 0.884718, 0.213336) ,
  rgb (0.992218, 0.888385, 0.205468) ,
  rgb (0.994847, 0.892954, 0.203445) ,
  rgb (0.995249, 0.898384, 0.207561) ,
  rgb (0.995503, 0.903866, 0.21237) ,
  rgb (0.995737, 0.909344, 0.217772)
});


list_data inferno = list_data(new pen[] {
  rgb (0.001462, 0.000466, 0.013866) ,
  rgb (0.002267, 0.00127, 0.01857) ,
  rgb (0.003299, 0.002249, 0.024239) ,
  rgb (0.004547, 0.003392, 0.030909) ,
  rgb (0.006006, 0.004692, 0.038558) ,
  rgb (0.007676, 0.006136, 0.046836) ,
  rgb (0.009561, 0.007713, 0.055143) ,
  rgb (0.011663, 0.009417, 0.06346) ,
  rgb (0.013995, 0.011225, 0.071862) ,
  rgb (0.016561, 0.013136, 0.080282) ,
  rgb (0.019373, 0.015133, 0.088767) ,
  rgb (0.022447, 0.017199, 0.097327) ,
  rgb (0.025793, 0.019331, 0.10593) ,
  rgb (0.029432, 0.021503, 0.114621) ,
  rgb (0.033385, 0.023702, 0.123397) ,
  rgb (0.037668, 0.025921, 0.132232) ,
  rgb (0.042253, 0.028139, 0.141141) ,
  rgb (0.046915, 0.030324, 0.150164) ,
  rgb (0.051644, 0.032474, 0.159254) ,
  rgb (0.056449, 0.034569, 0.168414) ,
  rgb (0.06134, 0.03659, 0.177642) ,
  rgb (0.066331, 0.038504, 0.186962) ,
  rgb (0.071429, 0.040294, 0.196354) ,
  rgb (0.076637, 0.041905, 0.205799) ,
  rgb (0.081962, 0.043328, 0.215289) ,
  rgb (0.087411, 0.044556, 0.224813) ,
  rgb (0.09299, 0.045583, 0.234358) ,
  rgb (0.098702, 0.046402, 0.243904) ,
  rgb (0.104551, 0.047008, 0.25343) ,
  rgb (0.110536, 0.047399, 0.262912) ,
  rgb (0.116656, 0.047574, 0.272321) ,
  rgb (0.122908, 0.047536, 0.281624) ,
  rgb (0.129285, 0.047293, 0.290788) ,
  rgb (0.135778, 0.046856, 0.299776) ,
  rgb (0.142378, 0.046242, 0.308553) ,
  rgb (0.149073, 0.045468, 0.317085) ,
  rgb (0.15585, 0.044559, 0.325338) ,
  rgb (0.162689, 0.043554, 0.333277) ,
  rgb (0.169575, 0.042489, 0.340874) ,
  rgb (0.176493, 0.041402, 0.348111) ,
  rgb (0.183429, 0.040329, 0.354971) ,
  rgb (0.190367, 0.039309, 0.361447) ,
  rgb (0.197297, 0.0384, 0.367535) ,
  rgb (0.204209, 0.037632, 0.373238) ,
  rgb (0.211095, 0.03703, 0.378563) ,
  rgb (0.217949, 0.036615, 0.383522) ,
  rgb (0.224763, 0.036405, 0.388129) ,
  rgb (0.231538, 0.036405, 0.3924) ,
  rgb (0.238273, 0.036621, 0.396353) ,
  rgb (0.244967, 0.037055, 0.400007) ,
  rgb (0.25162, 0.037705, 0.403378) ,
  rgb (0.258234, 0.038571, 0.406485) ,
  rgb (0.26481, 0.039647, 0.409345) ,
  rgb (0.271347, 0.040922, 0.411976) ,
  rgb (0.27785, 0.042353, 0.414392) ,
  rgb (0.284321, 0.043933, 0.416608) ,
  rgb (0.290763, 0.045644, 0.418637) ,
  rgb (0.297178, 0.04747, 0.420491) ,
  rgb (0.303568, 0.049396, 0.422182) ,
  rgb (0.309935, 0.051407, 0.423721) ,
  rgb (0.316282, 0.05349, 0.425116) ,
  rgb (0.32261, 0.055634, 0.426377) ,
  rgb (0.328921, 0.057827, 0.427511) ,
  rgb (0.335217, 0.06006, 0.428524) ,
  rgb (0.3415, 0.062325, 0.429425) ,
  rgb (0.347771, 0.064616, 0.430217) ,
  rgb (0.354032, 0.066925, 0.430906) ,
  rgb (0.360284, 0.069247, 0.431497) ,
  rgb (0.366529, 0.071579, 0.431994) ,
  rgb (0.372768, 0.073915, 0.4324) ,
  rgb (0.379001, 0.076253, 0.432719) ,
  rgb (0.385228, 0.078591, 0.432955) ,
  rgb (0.391453, 0.080927, 0.433109) ,
  rgb (0.397674, 0.083257, 0.433183) ,
  rgb (0.403894, 0.08558, 0.433179) ,
  rgb (0.410113, 0.087896, 0.433098) ,
  rgb (0.416331, 0.090203, 0.432943) ,
  rgb (0.422549, 0.092501, 0.432714) ,
  rgb (0.428768, 0.09479, 0.432412) ,
  rgb (0.434987, 0.097069, 0.432039) ,
  rgb (0.441207, 0.099338, 0.431594) ,
  rgb (0.447428, 0.101597, 0.43108) ,
  rgb (0.453651, 0.103848, 0.430498) ,
  rgb (0.459875, 0.106089, 0.429846) ,
  rgb (0.4661, 0.108322, 0.429125) ,
  rgb (0.472328, 0.110547, 0.428334) ,
  rgb (0.478558, 0.112764, 0.427475) ,
  rgb (0.484789, 0.114974, 0.426548) ,
  rgb (0.491022, 0.117179, 0.425552) ,
  rgb (0.497257, 0.119379, 0.424488) ,
  rgb (0.503493, 0.121575, 0.423356) ,
  rgb (0.50973, 0.123769, 0.422156) ,
  rgb (0.515967, 0.12596, 0.420887) ,
  rgb (0.522206, 0.12815, 0.419549) ,
  rgb (0.528444, 0.130341, 0.418142) ,
  rgb (0.534683, 0.132534, 0.416667) ,
  rgb (0.54092, 0.134729, 0.415123) ,
  rgb (0.547157, 0.136929, 0.413511) ,
  rgb (0.553392, 0.139134, 0.411829) ,
  rgb (0.559624, 0.141346, 0.410078) ,
  rgb (0.565854, 0.143567, 0.408258) ,
  rgb (0.572081, 0.145797, 0.406369) ,
  rgb (0.578304, 0.148039, 0.404411) ,
  rgb (0.584521, 0.150294, 0.402385) ,
  rgb (0.590734, 0.152563, 0.40029) ,
  rgb (0.59694, 0.154848, 0.398125) ,
  rgb (0.603139, 0.157151, 0.395891) ,
  rgb (0.60933, 0.159474, 0.393589) ,
  rgb (0.615513, 0.161817, 0.391219) ,
  rgb (0.621685, 0.164184, 0.388781) ,
  rgb (0.627847, 0.166575, 0.386276) ,
  rgb (0.633998, 0.168992, 0.383704) ,
  rgb (0.640135, 0.171438, 0.381065) ,
  rgb (0.64626, 0.173914, 0.378359) ,
  rgb (0.652369, 0.176421, 0.375586) ,
  rgb (0.658463, 0.178962, 0.372748) ,
  rgb (0.66454, 0.181539, 0.369846) ,
  rgb (0.670599, 0.184153, 0.366879) ,
  rgb (0.676638, 0.186807, 0.363849) ,
  rgb (0.682656, 0.189501, 0.360757) ,
  rgb (0.688653, 0.192239, 0.357603) ,
  rgb (0.694627, 0.195021, 0.354388) ,
  rgb (0.700576, 0.197851, 0.351113) ,
  rgb (0.7065, 0.200728, 0.347777) ,
  rgb (0.712396, 0.203656, 0.344383) ,
  rgb (0.718264, 0.206636, 0.340931) ,
  rgb (0.724103, 0.20967, 0.337424) ,
  rgb (0.729909, 0.212759, 0.333861) ,
  rgb (0.735683, 0.215906, 0.330245) ,
  rgb (0.741423, 0.219112, 0.326576) ,
  rgb (0.747127, 0.222378, 0.322856) ,
  rgb (0.752794, 0.225706, 0.319085) ,
  rgb (0.758422, 0.229097, 0.315266) ,
  rgb (0.76401, 0.232554, 0.311399) ,
  rgb (0.769556, 0.236077, 0.307485) ,
  rgb (0.775059, 0.239667, 0.303526) ,
  rgb (0.780517, 0.243327, 0.299523) ,
  rgb (0.785929, 0.247056, 0.295477) ,
  rgb (0.791293, 0.250856, 0.29139) ,
  rgb (0.796607, 0.254728, 0.287264) ,
  rgb (0.801871, 0.258674, 0.283099) ,
  rgb (0.807082, 0.262692, 0.278898) ,
  rgb (0.812239, 0.266786, 0.274661) ,
  rgb (0.817341, 0.270954, 0.27039) ,
  rgb (0.822386, 0.275197, 0.266085) ,
  rgb (0.827372, 0.279517, 0.26175) ,
  rgb (0.832299, 0.283913, 0.257383) ,
  rgb (0.837165, 0.288385, 0.252988) ,
  rgb (0.841969, 0.292933, 0.248564) ,
  rgb (0.846709, 0.297559, 0.244113) ,
  rgb (0.851384, 0.30226, 0.239636) ,
  rgb (0.855992, 0.307038, 0.235133) ,
  rgb (0.860533, 0.311892, 0.230606) ,
  rgb (0.865006, 0.316822, 0.226055) ,
  rgb (0.869409, 0.321827, 0.221482) ,
  rgb (0.873741, 0.326906, 0.216886) ,
  rgb (0.878001, 0.33206, 0.212268) ,
  rgb (0.882188, 0.337287, 0.207628) ,
  rgb (0.886302, 0.342586, 0.202968) ,
  rgb (0.890341, 0.347957, 0.198286) ,
  rgb (0.894305, 0.353399, 0.193584) ,
  rgb (0.898192, 0.358911, 0.18886) ,
  rgb (0.902003, 0.364492, 0.184116) ,
  rgb (0.905735, 0.37014, 0.17935) ,
  rgb (0.90939, 0.375856, 0.174563) ,
  rgb (0.912966, 0.381636, 0.169755) ,
  rgb (0.916462, 0.387481, 0.164924) ,
  rgb (0.919879, 0.393389, 0.16007) ,
  rgb (0.923215, 0.399359, 0.155193) ,
  rgb (0.92647, 0.405389, 0.150292) ,
  rgb (0.929644, 0.411479, 0.145367) ,
  rgb (0.932737, 0.417627, 0.140417) ,
  rgb (0.935747, 0.423831, 0.13544) ,
  rgb (0.938675, 0.430091, 0.130438) ,
  rgb (0.941521, 0.436405, 0.125409) ,
  rgb (0.944285, 0.442772, 0.120354) ,
  rgb (0.946965, 0.449191, 0.115272) ,
  rgb (0.949562, 0.45566, 0.110164) ,
  rgb (0.952075, 0.462178, 0.105031) ,
  rgb (0.954506, 0.468744, 0.099874) ,
  rgb (0.956852, 0.475356, 0.094695) ,
  rgb (0.959114, 0.482014, 0.089499) ,
  rgb (0.961293, 0.488716, 0.084289) ,
  rgb (0.963387, 0.495462, 0.079073) ,
  rgb (0.965397, 0.502249, 0.073859) ,
  rgb (0.967322, 0.509078, 0.068659) ,
  rgb (0.969163, 0.515946, 0.063488) ,
  rgb (0.970919, 0.522853, 0.058367) ,
  rgb (0.97259, 0.529798, 0.053324) ,
  rgb (0.974176, 0.53678, 0.048392) ,
  rgb (0.975677, 0.543798, 0.043618) ,
  rgb (0.977092, 0.55085, 0.03905) ,
  rgb (0.978422, 0.557937, 0.034931) ,
  rgb (0.979666, 0.565057, 0.031409) ,
  rgb (0.980824, 0.572209, 0.028508) ,
  rgb (0.981895, 0.579392, 0.02625) ,
  rgb (0.982881, 0.586606, 0.024661) ,
  rgb (0.983779, 0.593849, 0.02377) ,
  rgb (0.984591, 0.601122, 0.023606) ,
  rgb (0.985315, 0.608422, 0.024202) ,
  rgb (0.985952, 0.61575, 0.025592) ,
  rgb (0.986502, 0.623105, 0.027814) ,
  rgb (0.986964, 0.630485, 0.030908) ,
  rgb (0.987337, 0.63789, 0.034916) ,
  rgb (0.987622, 0.64532, 0.039886) ,
  rgb (0.987819, 0.652773, 0.045581) ,
  rgb (0.987926, 0.66025, 0.05175) ,
  rgb (0.987945, 0.667748, 0.058329) ,
  rgb (0.987874, 0.675267, 0.065257) ,
  rgb (0.987714, 0.682807, 0.072489) ,
  rgb (0.987464, 0.690366, 0.07999) ,
  rgb (0.987124, 0.697944, 0.087731) ,
  rgb (0.986694, 0.70554, 0.095694) ,
  rgb (0.986175, 0.713153, 0.103863) ,
  rgb (0.985566, 0.720782, 0.112229) ,
  rgb (0.984865, 0.728427, 0.120785) ,
  rgb (0.984075, 0.736087, 0.129527) ,
  rgb (0.983196, 0.743758, 0.138453) ,
  rgb (0.982228, 0.751442, 0.147565) ,
  rgb (0.981173, 0.759135, 0.156863) ,
  rgb (0.980032, 0.766837, 0.166353) ,
  rgb (0.978806, 0.774545, 0.176037) ,
  rgb (0.977497, 0.782258, 0.185923) ,
  rgb (0.976108, 0.789974, 0.196018) ,
  rgb (0.974638, 0.797692, 0.206332) ,
  rgb (0.973088, 0.805409, 0.216877) ,
  rgb (0.971468, 0.813122, 0.227658) ,
  rgb (0.969783, 0.820825, 0.238686) ,
  rgb (0.968041, 0.828515, 0.249972) ,
  rgb (0.966243, 0.836191, 0.261534) ,
  rgb (0.964394, 0.843848, 0.273391) ,
  rgb (0.962517, 0.851476, 0.285546) ,
  rgb (0.960626, 0.859069, 0.29801) ,
  rgb (0.95872, 0.866624, 0.31082) ,
  rgb (0.956834, 0.874129, 0.323974) ,
  rgb (0.954997, 0.881569, 0.337475) ,
  rgb (0.953215, 0.888942, 0.351369) ,
  rgb (0.951546, 0.896226, 0.365627) ,
  rgb (0.950018, 0.903409, 0.380271) ,
  rgb (0.948683, 0.910473, 0.395289) ,
  rgb (0.947594, 0.917399, 0.410665) ,
  rgb (0.946809, 0.924168, 0.426373) ,
  rgb (0.946392, 0.930761, 0.442367) ,
  rgb (0.946403, 0.937159, 0.458592) ,
  rgb (0.946903, 0.943348, 0.47497) ,
  rgb (0.947937, 0.949318, 0.491426) ,
  rgb (0.949545, 0.955063, 0.50786) ,
  rgb (0.95174, 0.960587, 0.524203) ,
  rgb (0.954529, 0.965896, 0.540361) ,
  rgb (0.957896, 0.971003, 0.556275) ,
  rgb (0.961812, 0.975924, 0.571925) ,
  rgb (0.966249, 0.980678, 0.587206) ,
  rgb (0.971162, 0.985282, 0.602154) ,
  rgb (0.976511, 0.989753, 0.61676) ,
  rgb (0.982257, 0.994109, 0.631017) ,
  rgb (0.988362, 0.998364, 0.644924)
});


list_data magma = list_data(new pen[] {
  rgb (0.001462, 0.000466, 0.013866) ,
  rgb (0.002258, 0.001295, 0.018331) ,
  rgb (0.003279, 0.002305, 0.023708) ,
  rgb (0.004512, 0.00349, 0.029965) ,
  rgb (0.00595, 0.004843, 0.03713) ,
  rgb (0.007588, 0.006356, 0.044973) ,
  rgb (0.009426, 0.008022, 0.052844) ,
  rgb (0.011465, 0.009828, 0.06075) ,
  rgb (0.013708, 0.011771, 0.068667) ,
  rgb (0.016156, 0.01384, 0.076603) ,
  rgb (0.018815, 0.016026, 0.084584) ,
  rgb (0.021692, 0.01832, 0.09261) ,
  rgb (0.024792, 0.020715, 0.100676) ,
  rgb (0.028123, 0.023201, 0.108787) ,
  rgb (0.031696, 0.025765, 0.116965) ,
  rgb (0.03552, 0.028397, 0.125209) ,
  rgb (0.039608, 0.03109, 0.133515) ,
  rgb (0.04383, 0.03383, 0.141886) ,
  rgb (0.048062, 0.036607, 0.150327) ,
  rgb (0.05232, 0.039407, 0.158841) ,
  rgb (0.056615, 0.04216, 0.167446) ,
  rgb (0.060949, 0.044794, 0.176129) ,
  rgb (0.06533, 0.047318, 0.184892) ,
  rgb (0.069764, 0.049726, 0.193735) ,
  rgb (0.074257, 0.052017, 0.20266) ,
  rgb (0.078815, 0.054184, 0.211667) ,
  rgb (0.083446, 0.056225, 0.220755) ,
  rgb (0.088155, 0.058133, 0.229922) ,
  rgb (0.092949, 0.059904, 0.239164) ,
  rgb (0.097833, 0.061531, 0.248477) ,
  rgb (0.102815, 0.06301, 0.257854) ,
  rgb (0.107899, 0.064335, 0.267289) ,
  rgb (0.113094, 0.065492, 0.276784) ,
  rgb (0.118405, 0.066479, 0.286321) ,
  rgb (0.123833, 0.067295, 0.295879) ,
  rgb (0.12938, 0.067935, 0.305443) ,
  rgb (0.135053, 0.068391, 0.315) ,
  rgb (0.140858, 0.068654, 0.324538) ,
  rgb (0.146785, 0.068738, 0.334011) ,
  rgb (0.152839, 0.068637, 0.343404) ,
  rgb (0.159018, 0.068354, 0.352688) ,
  rgb (0.165308, 0.067911, 0.361816) ,
  rgb (0.171713, 0.067305, 0.370771) ,
  rgb (0.178212, 0.066576, 0.379497) ,
  rgb (0.184801, 0.065732, 0.387973) ,
  rgb (0.19146, 0.064818, 0.396152) ,
  rgb (0.198177, 0.063862, 0.404009) ,
  rgb (0.204935, 0.062907, 0.411514) ,
  rgb (0.211718, 0.061992, 0.418647) ,
  rgb (0.218512, 0.061158, 0.425392) ,
  rgb (0.225302, 0.060445, 0.431742) ,
  rgb (0.232077, 0.059889, 0.437695) ,
  rgb (0.238826, 0.059517, 0.443256) ,
  rgb (0.245543, 0.059352, 0.448436) ,
  rgb (0.25222, 0.059415, 0.453248) ,
  rgb (0.258857, 0.059706, 0.45771) ,
  rgb (0.265447, 0.060237, 0.46184) ,
  rgb (0.271994, 0.060994, 0.46566) ,
  rgb (0.278493, 0.061978, 0.46919) ,
  rgb (0.284951, 0.063168, 0.472451) ,
  rgb (0.291366, 0.064553, 0.475462) ,
  rgb (0.29774, 0.066117, 0.478243) ,
  rgb (0.304081, 0.067835, 0.480812) ,
  rgb (0.310382, 0.069702, 0.483186) ,
  rgb (0.316654, 0.07169, 0.48538) ,
  rgb (0.322899, 0.073782, 0.487408) ,
  rgb (0.329114, 0.075972, 0.489287) ,
  rgb (0.335308, 0.078236, 0.491024) ,
  rgb (0.341482, 0.080564, 0.492631) ,
  rgb (0.347636, 0.082946, 0.494121) ,
  rgb (0.353773, 0.085373, 0.495501) ,
  rgb (0.359898, 0.087831, 0.496778) ,
  rgb (0.366012, 0.090314, 0.49796) ,
  rgb (0.372116, 0.092816, 0.499053) ,
  rgb (0.378211, 0.095332, 0.500067) ,
  rgb (0.384299, 0.097855, 0.501002) ,
  rgb (0.390384, 0.100379, 0.501864) ,
  rgb (0.396467, 0.102902, 0.502658) ,
  rgb (0.402548, 0.10542, 0.503386) ,
  rgb (0.408629, 0.10793, 0.504052) ,
  rgb (0.414709, 0.110431, 0.504662) ,
  rgb (0.420791, 0.11292, 0.505215) ,
  rgb (0.426877, 0.115395, 0.505714) ,
  rgb (0.432967, 0.117855, 0.50616) ,
  rgb (0.439062, 0.120298, 0.506555) ,
  rgb (0.445163, 0.122724, 0.506901) ,
  rgb (0.451271, 0.125132, 0.507198) ,
  rgb (0.457386, 0.127522, 0.507448) ,
  rgb (0.463508, 0.129893, 0.507652) ,
  rgb (0.46964, 0.132245, 0.507809) ,
  rgb (0.47578, 0.134577, 0.507921) ,
  rgb (0.481929, 0.136891, 0.507989) ,
  rgb (0.488088, 0.139186, 0.508011) ,
  rgb (0.494258, 0.141462, 0.507988) ,
  rgb (0.500438, 0.143719, 0.50792) ,
  rgb (0.506629, 0.145958, 0.507806) ,
  rgb (0.512831, 0.148179, 0.507648) ,
  rgb (0.519045, 0.150383, 0.507443) ,
  rgb (0.52527, 0.152569, 0.507192) ,
  rgb (0.531507, 0.154739, 0.506895) ,
  rgb (0.537755, 0.156894, 0.506551) ,
  rgb (0.544015, 0.159033, 0.506159) ,
  rgb (0.550287, 0.161158, 0.505719) ,
  rgb (0.556571, 0.163269, 0.50523) ,
  rgb (0.562866, 0.165368, 0.504692) ,
  rgb (0.569172, 0.167454, 0.504105) ,
  rgb (0.57549, 0.16953, 0.503466) ,
  rgb (0.581819, 0.171596, 0.502777) ,
  rgb (0.588158, 0.173652, 0.502035) ,
  rgb (0.594508, 0.175701, 0.501241) ,
  rgb (0.600868, 0.177743, 0.500394) ,
  rgb (0.607238, 0.179779, 0.499492) ,
  rgb (0.613617, 0.181811, 0.498536) ,
  rgb (0.620005, 0.18384, 0.497524) ,
  rgb (0.626401, 0.185867, 0.496456) ,
  rgb (0.632805, 0.187893, 0.495332) ,
  rgb (0.639216, 0.189921, 0.49415) ,
  rgb (0.645633, 0.191952, 0.49291) ,
  rgb (0.652056, 0.193986, 0.491611) ,
  rgb (0.658483, 0.196027, 0.490253) ,
  rgb (0.664915, 0.198075, 0.488836) ,
  rgb (0.671349, 0.200133, 0.487358) ,
  rgb (0.677786, 0.202203, 0.485819) ,
  rgb (0.684224, 0.204286, 0.484219) ,
  rgb (0.690661, 0.206384, 0.482558) ,
  rgb (0.697098, 0.208501, 0.480835) ,
  rgb (0.703532, 0.210638, 0.479049) ,
  rgb (0.709962, 0.212797, 0.477201) ,
  rgb (0.716387, 0.214982, 0.47529) ,
  rgb (0.722805, 0.217194, 0.473316) ,
  rgb (0.729216, 0.219437, 0.471279) ,
  rgb (0.735616, 0.221713, 0.46918) ,
  rgb (0.742004, 0.224025, 0.467018) ,
  rgb (0.748378, 0.226377, 0.464794) ,
  rgb (0.754737, 0.228772, 0.462509) ,
  rgb (0.761077, 0.231214, 0.460162) ,
  rgb (0.767398, 0.233705, 0.457755) ,
  rgb (0.773695, 0.236249, 0.455289) ,
  rgb (0.779968, 0.238851, 0.452765) ,
  rgb (0.786212, 0.241514, 0.450184) ,
  rgb (0.792427, 0.244242, 0.447543) ,
  rgb (0.798608, 0.24704, 0.444848) ,
  rgb (0.804752, 0.249911, 0.442102) ,
  rgb (0.810855, 0.252861, 0.439305) ,
  rgb (0.816914, 0.255895, 0.436461) ,
  rgb (0.822926, 0.259016, 0.433573) ,
  rgb (0.828886, 0.262229, 0.430644) ,
  rgb (0.834791, 0.26554, 0.427671) ,
  rgb (0.840636, 0.268953, 0.424666) ,
  rgb (0.846416, 0.272473, 0.421631) ,
  rgb (0.852126, 0.276106, 0.418573) ,
  rgb (0.857763, 0.279857, 0.415496) ,
  rgb (0.86332, 0.283729, 0.412403) ,
  rgb (0.868793, 0.287728, 0.409303) ,
  rgb (0.874176, 0.291859, 0.406205) ,
  rgb (0.879464, 0.296125, 0.403118) ,
  rgb (0.884651, 0.30053, 0.400047) ,
  rgb (0.889731, 0.305079, 0.397002) ,
  rgb (0.8947, 0.309773, 0.393995) ,
  rgb (0.899552, 0.314616, 0.391037) ,
  rgb (0.904281, 0.31961, 0.388137) ,
  rgb (0.908884, 0.324755, 0.385308) ,
  rgb (0.913354, 0.330052, 0.382563) ,
  rgb (0.917689, 0.3355, 0.379915) ,
  rgb (0.921884, 0.341098, 0.377376) ,
  rgb (0.925937, 0.346844, 0.374959) ,
  rgb (0.929845, 0.352734, 0.372677) ,
  rgb (0.933606, 0.358764, 0.370541) ,
  rgb (0.937221, 0.364929, 0.368567) ,
  rgb (0.940687, 0.371224, 0.366762) ,
  rgb (0.944006, 0.377643, 0.365136) ,
  rgb (0.94718, 0.384178, 0.363701) ,
  rgb (0.95021, 0.39082, 0.362468) ,
  rgb (0.953099, 0.397563, 0.361438) ,
  rgb (0.955849, 0.4044, 0.360619) ,
  rgb (0.958464, 0.411324, 0.360014) ,
  rgb (0.960949, 0.418323, 0.35963) ,
  rgb (0.96331, 0.42539, 0.359469) ,
  rgb (0.965549, 0.432519, 0.359529) ,
  rgb (0.967671, 0.439703, 0.35981) ,
  rgb (0.96968, 0.446936, 0.360311) ,
  rgb (0.971582, 0.45421, 0.36103) ,
  rgb (0.973381, 0.46152, 0.361965) ,
  rgb (0.975082, 0.468861, 0.363111) ,
  rgb (0.97669, 0.476226, 0.364466) ,
  rgb (0.97821, 0.483612, 0.366025) ,
  rgb (0.979645, 0.491014, 0.367783) ,
  rgb (0.981, 0.498428, 0.369734) ,
  rgb (0.982279, 0.505851, 0.371874) ,
  rgb (0.983485, 0.51328, 0.374198) ,
  rgb (0.984622, 0.520713, 0.376698) ,
  rgb (0.985693, 0.528148, 0.379371) ,
  rgb (0.9867, 0.535582, 0.38221) ,
  rgb (0.987646, 0.543015, 0.38521) ,
  rgb (0.988533, 0.550446, 0.388365) ,
  rgb (0.989363, 0.557873, 0.391671) ,
  rgb (0.990138, 0.565296, 0.395122) ,
  rgb (0.990871, 0.572706, 0.398714) ,
  rgb (0.991558, 0.580107, 0.402441) ,
  rgb (0.992196, 0.587502, 0.406299) ,
  rgb (0.992785, 0.594891, 0.410283) ,
  rgb (0.993326, 0.602275, 0.41439) ,
  rgb (0.993834, 0.609644, 0.418613) ,
  rgb (0.994309, 0.616999, 0.42295) ,
  rgb (0.994738, 0.62435, 0.427397) ,
  rgb (0.995122, 0.631696, 0.431951) ,
  rgb (0.99548, 0.639027, 0.436607) ,
  rgb (0.99581, 0.646344, 0.441361) ,
  rgb (0.996096, 0.653659, 0.446213) ,
  rgb (0.996341, 0.660969, 0.45116) ,
  rgb (0.99658, 0.668256, 0.456192) ,
  rgb (0.996775, 0.675541, 0.461314) ,
  rgb (0.996925, 0.682828, 0.466526) ,
  rgb (0.997077, 0.690088, 0.471811) ,
  rgb (0.997186, 0.697349, 0.477182) ,
  rgb (0.997254, 0.704611, 0.482635) ,
  rgb (0.997325, 0.711848, 0.488154) ,
  rgb (0.997351, 0.719089, 0.493755) ,
  rgb (0.997351, 0.726324, 0.499428) ,
  rgb (0.997341, 0.733545, 0.505167) ,
  rgb (0.997285, 0.740772, 0.510983) ,
  rgb (0.997228, 0.747981, 0.516859) ,
  rgb (0.997138, 0.75519, 0.522806) ,
  rgb (0.997019, 0.762398, 0.528821) ,
  rgb (0.996898, 0.769591, 0.534892) ,
  rgb (0.996727, 0.776795, 0.541039) ,
  rgb (0.996571, 0.783977, 0.547233) ,
  rgb (0.996369, 0.791167, 0.553499) ,
  rgb (0.996162, 0.798348, 0.55982) ,
  rgb (0.995932, 0.805527, 0.566202) ,
  rgb (0.99568, 0.812706, 0.572645) ,
  rgb (0.995424, 0.819875, 0.57914) ,
  rgb (0.995131, 0.827052, 0.585701) ,
  rgb (0.994851, 0.834213, 0.592307) ,
  rgb (0.994524, 0.841387, 0.598983) ,
  rgb (0.994222, 0.84854, 0.605696) ,
  rgb (0.993866, 0.855711, 0.612482) ,
  rgb (0.993545, 0.862859, 0.619299) ,
  rgb (0.99317, 0.870024, 0.626189) ,
  rgb (0.992831, 0.877168, 0.633109) ,
  rgb (0.99244, 0.88433, 0.640099) ,
  rgb (0.992089, 0.89147, 0.647116) ,
  rgb (0.991688, 0.898627, 0.654202) ,
  rgb (0.991332, 0.905763, 0.661309) ,
  rgb (0.99093, 0.912915, 0.668481) ,
  rgb (0.99057, 0.920049, 0.675675) ,
  rgb (0.990175, 0.927196, 0.682926) ,
  rgb (0.989815, 0.934329, 0.690198) ,
  rgb (0.989434, 0.94147, 0.697519) ,
  rgb (0.989077, 0.948604, 0.704863) ,
  rgb (0.988717, 0.955742, 0.712242) ,
  rgb (0.988367, 0.962878, 0.719649) ,
  rgb (0.988033, 0.970012, 0.727077) ,
  rgb (0.987691, 0.977154, 0.734536) ,
  rgb (0.987387, 0.984288, 0.742002) ,
  rgb (0.987053, 0.991438, 0.749504)
});


list_data plasma = list_data(new pen[] {
  rgb (0.050383, 0.029803, 0.527975) ,
  rgb (0.063536, 0.028426, 0.533124) ,
  rgb (0.075353, 0.027206, 0.538007) ,
  rgb (0.086222, 0.026125, 0.542658) ,
  rgb (0.096379, 0.025165, 0.547103) ,
  rgb (0.10598, 0.024309, 0.551368) ,
  rgb (0.115124, 0.023556, 0.555468) ,
  rgb (0.123903, 0.022878, 0.559423) ,
  rgb (0.132381, 0.022258, 0.56325) ,
  rgb (0.140603, 0.021687, 0.566959) ,
  rgb (0.148607, 0.021154, 0.570562) ,
  rgb (0.156421, 0.020651, 0.574065) ,
  rgb (0.16407, 0.020171, 0.577478) ,
  rgb (0.171574, 0.019706, 0.580806) ,
  rgb (0.17895, 0.019252, 0.584054) ,
  rgb (0.186213, 0.018803, 0.587228) ,
  rgb (0.193374, 0.018354, 0.59033) ,
  rgb (0.200445, 0.017902, 0.593364) ,
  rgb (0.207435, 0.017442, 0.596333) ,
  rgb (0.21435, 0.016973, 0.599239) ,
  rgb (0.221197, 0.016497, 0.602083) ,
  rgb (0.227983, 0.016007, 0.604867) ,
  rgb (0.234715, 0.015502, 0.607592) ,
  rgb (0.241396, 0.014979, 0.610259) ,
  rgb (0.248032, 0.014439, 0.612868) ,
  rgb (0.254627, 0.013882, 0.615419) ,
  rgb (0.261183, 0.013308, 0.617911) ,
  rgb (0.267703, 0.012716, 0.620346) ,
  rgb (0.274191, 0.012109, 0.622722) ,
  rgb (0.280648, 0.011488, 0.625038) ,
  rgb (0.287076, 0.010855, 0.627295) ,
  rgb (0.293478, 0.010213, 0.62949) ,
  rgb (0.299855, 0.009561, 0.631624) ,
  rgb (0.30621, 0.008902, 0.633694) ,
  rgb (0.312543, 0.008239, 0.6357) ,
  rgb (0.318856, 0.007576, 0.63764) ,
  rgb (0.32515, 0.006915, 0.639512) ,
  rgb (0.331426, 0.006261, 0.641316) ,
  rgb (0.337683, 0.005618, 0.643049) ,
  rgb (0.343925, 0.004991, 0.64471) ,
  rgb (0.35015, 0.004382, 0.646298) ,
  rgb (0.356359, 0.003798, 0.64781) ,
  rgb (0.362553, 0.003243, 0.649245) ,
  rgb (0.368733, 0.002724, 0.650601) ,
  rgb (0.374897, 0.002245, 0.651876) ,
  rgb (0.381047, 0.001814, 0.653068) ,
  rgb (0.387183, 0.001434, 0.654177) ,
  rgb (0.393304, 0.001114, 0.655199) ,
  rgb (0.399411, 0.000859, 0.656133) ,
  rgb (0.405503, 0.000678, 0.656977) ,
  rgb (0.41158, 0.000577, 0.65773) ,
  rgb (0.417642, 0.000564, 0.65839) ,
  rgb (0.423689, 0.000646, 0.658956) ,
  rgb (0.429719, 0.000831, 0.659425) ,
  rgb (0.435734, 0.001127, 0.659797) ,
  rgb (0.441732, 0.00154, 0.660069) ,
  rgb (0.447714, 0.00208, 0.66024) ,
  rgb (0.453677, 0.002755, 0.66031) ,
  rgb (0.459623, 0.003574, 0.660277) ,
  rgb (0.46555, 0.004545, 0.660139) ,
  rgb (0.471457, 0.005678, 0.659897) ,
  rgb (0.477344, 0.00698, 0.659549) ,
  rgb (0.48321, 0.00846, 0.659095) ,
  rgb (0.489055, 0.010127, 0.658534) ,
  rgb (0.494877, 0.01199, 0.657865) ,
  rgb (0.500678, 0.014055, 0.657088) ,
  rgb (0.506454, 0.016333, 0.656202) ,
  rgb (0.512206, 0.018833, 0.655209) ,
  rgb (0.517933, 0.021563, 0.654109) ,
  rgb (0.523633, 0.024532, 0.652901) ,
  rgb (0.529306, 0.027747, 0.651586) ,
  rgb (0.534952, 0.031217, 0.650165) ,
  rgb (0.54057, 0.03495, 0.64864) ,
  rgb (0.546157, 0.038954, 0.64701) ,
  rgb (0.551715, 0.043136, 0.645277) ,
  rgb (0.557243, 0.047331, 0.643443) ,
  rgb (0.562738, 0.051545, 0.641509) ,
  rgb (0.568201, 0.055778, 0.639477) ,
  rgb (0.573632, 0.060028, 0.637349) ,
  rgb (0.579029, 0.064296, 0.635126) ,
  rgb (0.584391, 0.068579, 0.632812) ,
  rgb (0.589719, 0.072878, 0.630408) ,
  rgb (0.595011, 0.07719, 0.627917) ,
  rgb (0.600266, 0.081516, 0.625342) ,
  rgb (0.605485, 0.085854, 0.622686) ,
  rgb (0.610667, 0.090204, 0.619951) ,
  rgb (0.615812, 0.094564, 0.61714) ,
  rgb (0.620919, 0.098934, 0.614257) ,
  rgb (0.625987, 0.103312, 0.611305) ,
  rgb (0.631017, 0.107699, 0.608287) ,
  rgb (0.636008, 0.112092, 0.605205) ,
  rgb (0.640959, 0.116492, 0.602065) ,
  rgb (0.645872, 0.120898, 0.598867) ,
  rgb (0.650746, 0.125309, 0.595617) ,
  rgb (0.65558, 0.129725, 0.592317) ,
  rgb (0.660374, 0.134144, 0.588971) ,
  rgb (0.665129, 0.138566, 0.585582) ,
  rgb (0.669845, 0.142992, 0.582154) ,
  rgb (0.674522, 0.147419, 0.578688) ,
  rgb (0.67916, 0.151848, 0.575189) ,
  rgb (0.683758, 0.156278, 0.57166) ,
  rgb (0.688318, 0.160709, 0.568103) ,
  rgb (0.69284, 0.165141, 0.564522) ,
  rgb (0.697324, 0.169573, 0.560919) ,
  rgb (0.701769, 0.174005, 0.557296) ,
  rgb (0.706178, 0.178437, 0.553657) ,
  rgb (0.710549, 0.182868, 0.550004) ,
  rgb (0.714883, 0.187299, 0.546338) ,
  rgb (0.719181, 0.191729, 0.542663) ,
  rgb (0.723444, 0.196158, 0.538981) ,
  rgb (0.72767, 0.200586, 0.535293) ,
  rgb (0.731862, 0.205013, 0.531601) ,
  rgb (0.736019, 0.209439, 0.527908) ,
  rgb (0.740143, 0.213864, 0.524216) ,
  rgb (0.744232, 0.218288, 0.520524) ,
  rgb (0.748289, 0.222711, 0.516834) ,
  rgb (0.752312, 0.227133, 0.513149) ,
  rgb (0.756304, 0.231555, 0.509468) ,
  rgb (0.760264, 0.235976, 0.505794) ,
  rgb (0.764193, 0.240396, 0.502126) ,
  rgb (0.76809, 0.244817, 0.498465) ,
  rgb (0.771958, 0.249237, 0.494813) ,
  rgb (0.775796, 0.253658, 0.491171) ,
  rgb (0.779604, 0.258078, 0.487539) ,
  rgb (0.783383, 0.2625, 0.483918) ,
  rgb (0.787133, 0.266922, 0.480307) ,
  rgb (0.790855, 0.271345, 0.476706) ,
  rgb (0.794549, 0.27577, 0.473117) ,
  rgb (0.798216, 0.280197, 0.469538) ,
  rgb (0.801855, 0.284626, 0.465971) ,
  rgb (0.805467, 0.289057, 0.462415) ,
  rgb (0.809052, 0.293491, 0.45887) ,
  rgb (0.812612, 0.297928, 0.455338) ,
  rgb (0.816144, 0.302368, 0.451816) ,
  rgb (0.819651, 0.306812, 0.448306) ,
  rgb (0.823132, 0.311261, 0.444806) ,
  rgb (0.826588, 0.315714, 0.441316) ,
  rgb (0.830018, 0.320172, 0.437836) ,
  rgb (0.833422, 0.324635, 0.434366) ,
  rgb (0.836801, 0.329105, 0.430905) ,
  rgb (0.840155, 0.33358, 0.427455) ,
  rgb (0.843484, 0.338062, 0.424013) ,
  rgb (0.846788, 0.342551, 0.420579) ,
  rgb (0.850066, 0.347048, 0.417153) ,
  rgb (0.853319, 0.351553, 0.413734) ,
  rgb (0.856547, 0.356066, 0.410322) ,
  rgb (0.85975, 0.360588, 0.406917) ,
  rgb (0.862927, 0.365119, 0.403519) ,
  rgb (0.866078, 0.36966, 0.400126) ,
  rgb (0.869203, 0.374212, 0.396738) ,
  rgb (0.872303, 0.378774, 0.393355) ,
  rgb (0.875376, 0.383347, 0.389976) ,
  rgb (0.878423, 0.387932, 0.3866) ,
  rgb (0.881443, 0.392529, 0.383229) ,
  rgb (0.884436, 0.397139, 0.37986) ,
  rgb (0.887402, 0.401762, 0.376494) ,
  rgb (0.89034, 0.406398, 0.37313) ,
  rgb (0.89325, 0.411048, 0.369768) ,
  rgb (0.896131, 0.415712, 0.366407) ,
  rgb (0.898984, 0.420392, 0.363047) ,
  rgb (0.901807, 0.425087, 0.359688) ,
  rgb (0.904601, 0.429797, 0.356329) ,
  rgb (0.907365, 0.434524, 0.35297) ,
  rgb (0.910098, 0.439268, 0.34961) ,
  rgb (0.9128, 0.444029, 0.346251) ,
  rgb (0.915471, 0.448807, 0.34289) ,
  rgb (0.918109, 0.453603, 0.339529) ,
  rgb (0.920714, 0.458417, 0.336166) ,
  rgb (0.923287, 0.463251, 0.332801) ,
  rgb (0.925825, 0.468103, 0.329435) ,
  rgb (0.928329, 0.472975, 0.326067) ,
  rgb (0.930798, 0.477867, 0.322697) ,
  rgb (0.933232, 0.48278, 0.319325) ,
  rgb (0.93563, 0.487712, 0.315952) ,
  rgb (0.93799, 0.492667, 0.312575) ,
  rgb (0.940313, 0.497642, 0.309197) ,
  rgb (0.942598, 0.502639, 0.305816) ,
  rgb (0.944844, 0.507658, 0.302433) ,
  rgb (0.947051, 0.512699, 0.299049) ,
  rgb (0.949217, 0.517763, 0.295662) ,
  rgb (0.951344, 0.52285, 0.292275) ,
  rgb (0.953428, 0.52796, 0.288883) ,
  rgb (0.95547, 0.533093, 0.28549) ,
  rgb (0.957469, 0.53825, 0.282096) ,
  rgb (0.959424, 0.543431, 0.278701) ,
  rgb (0.961336, 0.548636, 0.275305) ,
  rgb (0.963203, 0.553865, 0.271909) ,
  rgb (0.965024, 0.559118, 0.268513) ,
  rgb (0.966798, 0.564396, 0.265118) ,
  rgb (0.968526, 0.5697, 0.261721) ,
  rgb (0.970205, 0.575028, 0.258325) ,
  rgb (0.971835, 0.580382, 0.254931) ,
  rgb (0.973416, 0.585761, 0.25154) ,
  rgb (0.974947, 0.591165, 0.248151) ,
  rgb (0.976428, 0.596595, 0.244767) ,
  rgb (0.977856, 0.602051, 0.241387) ,
  rgb (0.979233, 0.607532, 0.238013) ,
  rgb (0.980556, 0.613039, 0.234646) ,
  rgb (0.981826, 0.618572, 0.231287) ,
  rgb (0.983041, 0.624131, 0.227937) ,
  rgb (0.984199, 0.629718, 0.224595) ,
  rgb (0.985301, 0.63533, 0.221265) ,
  rgb (0.986345, 0.640969, 0.217948) ,
  rgb (0.987332, 0.646633, 0.214648) ,
  rgb (0.98826, 0.652325, 0.211364) ,
  rgb (0.989128, 0.658043, 0.2081) ,
  rgb (0.989935, 0.663787, 0.204859) ,
  rgb (0.990681, 0.669558, 0.201642) ,
  rgb (0.991365, 0.675355, 0.198453) ,
  rgb (0.991985, 0.681179, 0.195295) ,
  rgb (0.992541, 0.68703, 0.19217) ,
  rgb (0.993032, 0.692907, 0.189084) ,
  rgb (0.993456, 0.69881, 0.186041) ,
  rgb (0.993814, 0.704741, 0.183043) ,
  rgb (0.994103, 0.710698, 0.180097) ,
  rgb (0.994324, 0.716681, 0.177208) ,
  rgb (0.994474, 0.722691, 0.174381) ,
  rgb (0.994553, 0.728728, 0.171622) ,
  rgb (0.994561, 0.734791, 0.168938) ,
  rgb (0.994495, 0.74088, 0.166335) ,
  rgb (0.994355, 0.746995, 0.163821) ,
  rgb (0.994141, 0.753137, 0.161404) ,
  rgb (0.993851, 0.759304, 0.159092) ,
  rgb (0.993482, 0.765499, 0.156891) ,
  rgb (0.993033, 0.77172, 0.154808) ,
  rgb (0.992505, 0.777967, 0.152855) ,
  rgb (0.991897, 0.784239, 0.151042) ,
  rgb (0.991209, 0.790537, 0.149377) ,
  rgb (0.990439, 0.796859, 0.14787) ,
  rgb (0.989587, 0.803205, 0.146529) ,
  rgb (0.988648, 0.809579, 0.145357) ,
  rgb (0.987621, 0.815978, 0.144363) ,
  rgb (0.986509, 0.822401, 0.143557) ,
  rgb (0.985314, 0.828846, 0.142945) ,
  rgb (0.984031, 0.835315, 0.142528) ,
  rgb (0.982653, 0.841812, 0.142303) ,
  rgb (0.98119, 0.848329, 0.142279) ,
  rgb (0.979644, 0.854866, 0.142453) ,
  rgb (0.977995, 0.861432, 0.142808) ,
  rgb (0.976265, 0.868016, 0.143351) ,
  rgb (0.974443, 0.874622, 0.144061) ,
  rgb (0.97253, 0.88125, 0.144923) ,
  rgb (0.970533, 0.887896, 0.145919) ,
  rgb (0.968443, 0.894564, 0.147014) ,
  rgb (0.966271, 0.901249, 0.14818) ,
  rgb (0.964021, 0.90795, 0.14937) ,
  rgb (0.961681, 0.914672, 0.15052) ,
  rgb (0.959276, 0.921407, 0.151566) ,
  rgb (0.956808, 0.928152, 0.152409) ,
  rgb (0.954287, 0.934908, 0.152921) ,
  rgb (0.951726, 0.941671, 0.152925) ,
  rgb (0.949151, 0.948435, 0.152178) ,
  rgb (0.946602, 0.95519, 0.150328) ,
  rgb (0.944152, 0.961916, 0.146861) ,
  rgb (0.941896, 0.96859, 0.140956) ,
  rgb (0.940015, 0.975158, 0.131326)
});


list_data twilight = list_data(new pen[] {
  rgb (0.8857501584075443, 0.8500092494306783, 0.8879736506427196) ,
  rgb (0.8837852019553906, 0.8507294054031063, 0.8872322209694989) ,
  rgb (0.8817223105928579, 0.8512759407765347, 0.8863805692551482) ,
  rgb (0.8795410528270573, 0.8516567540749572, 0.8854143767924102) ,
  rgb (0.8772488085896548, 0.8518702833887027, 0.8843412038131143) ,
  rgb (0.8748534750857597, 0.8519152612302319, 0.8831692696761383) ,
  rgb (0.8723313408512408, 0.8518016547808089, 0.8818970435500162) ,
  rgb (0.8697047485350982, 0.8515240300479789, 0.8805388339000336) ,
  rgb (0.8669601550533358, 0.8510896085314068, 0.8790976697717334) ,
  rgb (0.86408985081464, 0.8505039116750779, 0.8775792578489263) ,
  rgb (0.8611024543689985, 0.8497675485700126, 0.8759924292343957) ,
  rgb (0.8579825924567037, 0.8488893481028184, 0.8743403855344628) ,
  rgb (0.8547259318925698, 0.8478748812467282, 0.8726282980930582) ,
  rgb (0.8513371457085719, 0.8467273579611647, 0.8708608165735044) ,
  rgb (0.8478071070257792, 0.8454546229209523, 0.8690403678369444) ,
  rgb (0.8441261828674842, 0.8440648271103739, 0.8671697332269007) ,
  rgb (0.8403042080595778, 0.8425605950855084, 0.865250882410458) ,
  rgb (0.8363403180919118, 0.8409479651895194, 0.8632852800107016) ,
  rgb (0.8322270571293441, 0.8392349062775448, 0.8612756350042788) ,
  rgb (0.8279689431601354, 0.837426007513952, 0.8592239945130679) ,
  rgb (0.8235742968025285, 0.8355248776479544, 0.8571319132851495) ,
  rgb (0.8190465467793753, 0.8335364929949034, 0.855002062870101) ,
  rgb (0.8143898212114309, 0.8314655869419785, 0.8528375906214702) ,
  rgb (0.8095999819094809, 0.8293189667350546, 0.8506444160105037) ,
  rgb (0.8046916442981458, 0.8270983878056066, 0.8484244929697402) ,
  rgb (0.79967075421268, 0.8248078181208093, 0.8461821002957853) ,
  rgb (0.7945430508923111, 0.8224511622630462, 0.8439218478682798) ,
  rgb (0.7893144556460892, 0.8200321318870201, 0.8416486380471222) ,
  rgb (0.7839910104276492, 0.8175542640053343, 0.8393674746403673) ,
  rgb (0.7785789200822759, 0.8150208937874255, 0.8370834463093898) ,
  rgb (0.7730841659017094, 0.8124352473546601, 0.8348017295057968) ,
  rgb (0.7675110850441786, 0.8098007598713145, 0.8325281663805967) ,
  rgb (0.7618690793798029, 0.8071194938764749, 0.830266486168872) ,
  rgb (0.7561644358438198, 0.8043940873347794, 0.8280213899472) ,
  rgb (0.750403467654067, 0.8016269900896532, 0.8257973785108242) ,
  rgb (0.7445924777189017, 0.7988204771958325, 0.8235986758615652) ,
  rgb (0.7387377170049494, 0.7959766573503101, 0.8214292278043301) ,
  rgb (0.7328454364552346, 0.7930974646884407, 0.8192926338423038) ,
  rgb (0.726921775128297, 0.7901846863592763, 0.8171921746672638) ,
  rgb (0.7209728066553678, 0.7872399592345264, 0.8151307392087926) ,
  rgb (0.7150040307625213, 0.7842648709158119, 0.8131111655994991) ,
  rgb (0.709020781345393, 0.7812608871607091, 0.8111359185511793) ,
  rgb (0.7030297722540817, 0.7782290497335813, 0.8092061884805697) ,
  rgb (0.6970365443886174, 0.7751705000806606, 0.8073233538006345) ,
  rgb (0.691046410093091, 0.7720862946067809, 0.8054884169067907) ,
  rgb (0.6850644615439593, 0.7689774029354699, 0.8037020626717691) ,
  rgb (0.6790955449988215, 0.765844721313959, 0.8019646617300199) ,
  rgb (0.6731442255942621, 0.7626890873389048, 0.8002762854580953) ,
  rgb (0.6672147980375281, 0.7595112803730375, 0.7986367465453776) ,
  rgb (0.6613112930078745, 0.7563120270871903, 0.7970456043491897) ,
  rgb (0.6554369232645472, 0.7530920875676843, 0.7955027112903105) ,
  rgb (0.6495957300425348, 0.7498520122194177, 0.7940067402149911) ,
  rgb (0.6437910831099849, 0.7465923800833657, 0.7925565320130605) ,
  rgb (0.6380258682854598, 0.7433137671403319, 0.7911510045957317) ,
  rgb (0.6323027138710603, 0.740016721601314, 0.7897889276264043) ,
  rgb (0.6266240202260459, 0.7367017540369944, 0.7884690131633456) ,
  rgb (0.6209919306481755, 0.733369347989232, 0.7871899462469658) ,
  rgb (0.6154084641177048, 0.7300199523273969, 0.7859502270675048) ,
  rgb (0.6098754317609306, 0.7266539875975829, 0.7847483573269471) ,
  rgb (0.6043943420027486, 0.7232718614323369, 0.7835829559353559) ,
  rgb (0.5989665814482068, 0.7198739489224673, 0.7824525989934664) ,
  rgb (0.5935933569683722, 0.7164606049658685, 0.781355882376401) ,
  rgb (0.588275797805555, 0.7130321464645814, 0.7802914140563652) ,
  rgb (0.5830148703693241, 0.7095888767699747, 0.7792578182047659) ,
  rgb (0.5778116438998202, 0.7061310615715398, 0.7782534512102552) ,
  rgb (0.5726668948158774, 0.7026589535425779, 0.7772770268091199) ,
  rgb (0.5675811785386197, 0.6991727930264627, 0.776327485342753) ,
  rgb (0.5625551535721934, 0.6956727838162965, 0.7754035914230984) ,
  rgb (0.5575894041960517, 0.6921591145825405, 0.7745041337932782) ,
  rgb (0.5526845058934713, 0.6886319451516638, 0.7736279426902245) ,
  rgb (0.5478409815301863, 0.6850914221850988, 0.7727738647344087) ,
  rgb (0.5430593242401823, 0.6815376725306588, 0.7719407969783508) ,
  rgb (0.5383401557517628, 0.677970811290954, 0.7711273443905772) ,
  rgb (0.533683891477284, 0.6743909370521273, 0.7703325054879735) ,
  rgb (0.529090861832473, 0.6707981230280622, 0.7695555229231313) ,
  rgb (0.5245615147059358, 0.6671924299614223, 0.7687954171423095) ,
  rgb (0.5200962739223556, 0.6635739143403039, 0.768051194033441) ,
  rgb (0.5156955988596057, 0.65994260812898, 0.7673219148959617) ,
  rgb (0.5113599254160193, 0.6562985398183186, 0.7666066378064533) ,
  rgb (0.5070896957645166, 0.6526417240314645, 0.7659044566083585) ,
  rgb (0.5028853540415561, 0.6489721673409526, 0.7652144671817491) ,
  rgb (0.4987473366135607, 0.6452898684900934, 0.7645357873418008) ,
  rgb (0.4946761847863938, 0.6415948411950443, 0.7638671900213091) ,
  rgb (0.4906722493856122, 0.6378870485884708, 0.7632081276316384) ,
  rgb (0.4867359599430568, 0.6341664625110051, 0.7625578008592404) ,
  rgb (0.4828677867260272, 0.6304330455306234, 0.761915371498953) ,
  rgb (0.47906816236197386, 0.6266867625186013, 0.7612800037566242) ,
  rgb (0.47533752394906287, 0.6229275728383581, 0.7606508557181775) ,
  rgb (0.4716762951887709, 0.6191554324288464, 0.7600270922788305) ,
  rgb (0.46808490970531597, 0.6153702869579029, 0.7594078989109274) ,
  rgb (0.4645637671630393, 0.6115720882286415, 0.7587924262302581) ,
  rgb (0.4611132664702388, 0.607760777169989, 0.7581798643680714) ,
  rgb (0.45773377230160567, 0.6039363004658646, 0.7575693690185916) ,
  rgb (0.45442563977552913, 0.6000985950385866, 0.7569601366060649) ,
  rgb (0.45118918687617743, 0.5962476205135354, 0.7563512064324664) ,
  rgb (0.4480247093358917, 0.5923833145214658, 0.7557417647410792) ,
  rgb (0.4449324685421538, 0.5885055998308617, 0.7551311041857901) ,
  rgb (0.441912717666964, 0.5846144110017557, 0.7545183888441067) ,
  rgb (0.43896563958048396, 0.5807096924109849, 0.7539027620828594) ,
  rgb (0.4360913895835637, 0.5767913799818608, 0.7532834105961016) ,
  rgb (0.43329008867358393, 0.5728594162560667, 0.7526594653256667) ,
  rgb (0.4305617907305757, 0.5689137457245718, 0.752030080993127) ,
  rgb (0.42790652284925834, 0.5649543060909209, 0.7513944352191484) ,
  rgb (0.42532423665011354, 0.560981049599503, 0.7507516498900512) ,
  rgb (0.4228148567577266, 0.5569939212699658, 0.7501008698822764) ,
  rgb (0.42037822361396326, 0.5529928715810817, 0.7494412559451894) ,
  rgb (0.4180141407923363, 0.5489778542188889, 0.7487719316700112) ,
  rgb (0.4157223260454232, 0.544948827153504, 0.7480920445900052) ,
  rgb (0.4135024574331473, 0.5409057477109848, 0.7474007329754309) ,
  rgb (0.4113541469730457, 0.5368485776500593, 0.7466971285506578) ,
  rgb (0.4092768899914751, 0.5327773017713032, 0.7459803063570782) ,
  rgb (0.4072701869421907, 0.5286918801105741, 0.7452494263758127) ,
  rgb (0.4053334378930318, 0.5245922817498312, 0.7445036583670813) ,
  rgb (0.40346600333905397, 0.5204784765384003, 0.7437421522356709) ,
  rgb (0.40166714010896104, 0.5163504496968876, 0.7429640345324835) ,
  rgb (0.39993606933454834, 0.5122081814321852, 0.7421684457131799) ,
  rgb (0.3982719152586337, 0.5080516653927614, 0.7413545091809972) ,
  rgb (0.3966737490566561, 0.5038808905384797, 0.7405213858051674) ,
  rgb (0.3951405880820763, 0.4996958532637776, 0.7396682021171571) ,
  rgb (0.39367135736822567, 0.4954965577745118, 0.738794102296364) ,
  rgb (0.39226494876209317, 0.4912830033289926, 0.7378982478447508) ,
  rgb (0.390920175719949, 0.4870552025122304, 0.7369797713388125) ,
  rgb (0.38963580160340855, 0.48281316715123496, 0.7360378254693274) ,
  rgb (0.3884105330084243, 0.47855691131792805, 0.7350715764115726) ,
  rgb (0.3872430145933025, 0.4742864593363539, 0.7340801678785439) ,
  rgb (0.386131841788921, 0.4700018340988123, 0.7330627749243106) ,
  rgb (0.3850755679365139, 0.46570306719930193, 0.732018540336905) ,
  rgb (0.38407269378943537, 0.46139018782416635, 0.7309466543290268) ,
  rgb (0.3831216808440275, 0.457063235814072, 0.7298462679135326) ,
  rgb (0.38222094988570376, 0.45272225034283325, 0.7287165614400378) ,
  rgb (0.3813688793045416, 0.4483672766927786, 0.7275567131714135) ,
  rgb (0.3805638069656562, 0.4439983720863372, 0.7263658704513531) ,
  rgb (0.3798040374484875, 0.4396155882122263, 0.7251432377876109) ,
  rgb (0.3790878928311076, 0.43521897612544935, 0.7238879869132313) ,
  rgb (0.378413635091359, 0.43080859411413064, 0.7225993199306104) ,
  rgb (0.3777794975351373, 0.4263845142616835, 0.7212763999353023) ,
  rgb (0.3771837184425123, 0.4219468022345483, 0.7199184152447577) ,
  rgb (0.37662448930806297, 0.41749553747893614, 0.7185245473617611) ,
  rgb (0.37610001286385814, 0.4130307995247706, 0.7170939691992023) ,
  rgb (0.375608469194424, 0.40855267638072096, 0.7156258509158755) ,
  rgb (0.37514802505380473, 0.4040612609993941, 0.7141193695725726) ,
  rgb (0.3747168601930223, 0.3995566498711684, 0.7125736851650046) ,
  rgb (0.3743131319931234, 0.3950389482828331, 0.7109879652237746) ,
  rgb (0.3739349933047578, 0.3905082752937583, 0.7093613429347845) ,
  rgb (0.3735806215098284, 0.3859647438605754, 0.7076929760731058) ,
  rgb (0.37324816143326384, 0.38140848555753937, 0.7059820097480604) ,
  rgb (0.3729357864666503, 0.3768396383521984, 0.7042275578058994) ,
  rgb (0.37264166757849604, 0.3722583500483685, 0.7024287314570723) ,
  rgb (0.37236397858465387, 0.36766477862108266, 0.7005846349652077) ,
  rgb (0.3721008970244382, 0.3630590973698238, 0.6986943461507372) ,
  rgb (0.3718506155898596, 0.3584414828587522, 0.6967569581025654) ,
  rgb (0.3716113323440048, 0.3538121372967869, 0.6947714991938089) ,
  rgb (0.37138124223736607, 0.34917126878479027, 0.6927370347192883) ,
  rgb (0.37115856636209105, 0.3445191141023017, 0.6906525358646499) ,
  rgb (0.3709415155133733, 0.33985591488818123, 0.6885170337950512) ,
  rgb (0.3707283327942267, 0.33518193808489577, 0.6863294816960677) ,
  rgb (0.37051738634484427, 0.3304974124430785, 0.6840888878885721) ,
  rgb (0.37030682071842685, 0.32580269697872455, 0.6817941168448668) ,
  rgb (0.37009487130772695, 0.3210981375964933, 0.6794440539905685) ,
  rgb (0.3698798032902536, 0.31638410101153364, 0.6770375543809057) ,
  rgb (0.36965987626565955, 0.3116609876295197, 0.6745734474341955) ,
  rgb (0.3694333459127623, 0.3069292355186234, 0.6720505284912062) ,
  rgb (0.36919847837592484, 0.3021893217650707, 0.6694675433161452) ,
  rgb (0.3689535530659678, 0.29744175492366276, 0.6668232208982426) ,
  rgb (0.3686968223189527, 0.292687098561501, 0.6641162529823691) ,
  rgb (0.36842655638020444, 0.2879259643777846, 0.661345269109446) ,
  rgb (0.3681410147989972, 0.2831590122118299, 0.6585088880697231) ,
  rgb (0.3678384369653108, 0.2783869718129776, 0.655605668384537) ,
  rgb (0.36751707094367697, 0.2736106331709098, 0.6526341171161864) ,
  rgb (0.36717513650699446, 0.26883085667326956, 0.6495927229789225) ,
  rgb (0.3668108554010799, 0.26404857724525643, 0.6464799165290824) ,
  rgb (0.3664224325155063, 0.25926481158628106, 0.6432940914076554) ,
  rgb (0.36600853966739794, 0.25448043878086224, 0.6400336180336859) ,
  rgb (0.3655669837353898, 0.24969683475296395, 0.6366967518748858) ,
  rgb (0.3650957984588681, 0.24491536803550484, 0.6332817352005559) ,
  rgb (0.3645930889012501, 0.24013747024823828, 0.629786801550261) ,
  rgb (0.3640569302208851, 0.23536470386204195, 0.6262101345195302) ,
  rgb (0.36348537610385145, 0.2305987621839642, 0.6225498862239288) ,
  rgb (0.3628764356004103, 0.2258414929328703, 0.6188041741082302) ,
  rgb (0.36222809558295926, 0.22109488427338303, 0.6149711234609613) ,
  rgb (0.36153829010998356, 0.21636111429594002, 0.6110488067964093) ,
  rgb (0.36080493826624654, 0.21164251793458128, 0.6070353217206471) ,
  rgb (0.36002681809096376, 0.20694122817889948, 0.6029284543191687) ,
  rgb (0.35920088560930186, 0.20226037920758122, 0.5987265295935138) ,
  rgb (0.3583248996661781, 0.197602942459778, 0.5944276851750107) ,
  rgb (0.35739663292915563, 0.1929720819784246, 0.5900301125106313) ,
  rgb (0.35641381143126327, 0.18837119869242164, 0.5855320765920552) ,
  rgb (0.3553741530690672, 0.18380392577704466, 0.580931914318328) ,
  rgb (0.3542753496066376, 0.17927413271618647, 0.5762280966066872) ,
  rgb (0.35311574421123737, 0.17478570377561287, 0.5714187152355529) ,
  rgb (0.3518924860887379, 0.17034320478524959, 0.5665028491121665) ,
  rgb (0.3506030444193101, 0.1659512998472086, 0.5614796470399323) ,
  rgb (0.34924513554955644, 0.16161477763045118, 0.5563483747416378) ,
  rgb (0.3478165323877778, 0.1573386351115298, 0.5511085345270326) ,
  rgb (0.3463150717579309, 0.15312802296627787, 0.5457599924248665) ,
  rgb (0.34473901574536375, 0.1489882058982641, 0.5403024592040654) ,
  rgb (0.34308600291572294, 0.14492465359918028, 0.534737042820671) ,
  rgb (0.34135411074506483, 0.1409427920655632, 0.5290650094033675) ,
  rgb (0.33954168752669694, 0.1370480189671817, 0.5232879753508524) ,
  rgb (0.3376473209067111, 0.13324562282438077, 0.5174080757397947) ,
  rgb (0.33566978565015315, 0.12954074251271822, 0.5114280721516895) ,
  rgb (0.33360804901486, 0.1259381830100592, 0.505351647966549) ,
  rgb (0.33146154891145124, 0.12244245263391232, 0.4991827458843107) ,
  rgb (0.3292300520323141, 0.11905764321981127, 0.49292595612342666) ,
  rgb (0.3269137124539796, 0.1157873496841953, 0.4865864649569746) ,
  rgb (0.32451307931207785, 0.11263459791730848, 0.48017007211645196) ,
  rgb (0.3220288227606932, 0.10960114111258401, 0.4736849472572688) ,
  rgb (0.31946262395497965, 0.1066887988239266, 0.46713728801395243) ,
  rgb (0.316816480890235, 0.10389861387653518, 0.46053414662739794) ,
  rgb (0.3140927841475553, 0.10123077676403242, 0.45388335612058467) ,
  rgb (0.31129434479712365, 0.0986847719340522, 0.4471931371516162) ,
  rgb (0.30842444457210105, 0.09625938534057774, 0.44047194882050544) ,
  rgb (0.30548675819945936, 0.09395276484082374, 0.4337284999936111) ,
  rgb (0.3024853636457425, 0.0917611873973036, 0.42697404043749887) ,
  rgb (0.2994248396021477, 0.08968225371675004, 0.42021619665853854) ,
  rgb (0.2963100038890529, 0.08771325096046395, 0.41346259134143476) ,
  rgb (0.2931459309698525, 0.08585065688962071, 0.40672178082365834) ,
  rgb (0.2899379244517661, 0.08409078829085731, 0.40000214725256295) ,
  rgb (0.28669151388283165, 0.08242987384848069, 0.39331182532243375) ,
  rgb (0.28341239797185225, 0.08086415336549937, 0.38665868550105914) ,
  rgb (0.2801063857697547, 0.07938999480226153, 0.38005028528138707) ,
  rgb (0.2767793961581559, 0.07800394103378822, 0.37349382846504675) ,
  rgb (0.2734373934245081, 0.07670280023749607, 0.36699616136347685) ,
  rgb (0.2700863774911405, 0.07548367558427554, 0.36056376228111864) ,
  rgb (0.26673233211995284, 0.0743440180285462, 0.3542027606624096) ,
  rgb (0.26338121807151404, 0.07328165793989708, 0.34791888996380105) ,
  rgb (0.26003895187439957, 0.0722947810433622, 0.3417175669546984) ,
  rgb (0.256711916510839, 0.07138010624208224, 0.3356064898460009) ,
  rgb (0.25340685873736807, 0.07053358292685183, 0.3295945757321303) ,
  rgb (0.2501284530619938, 0.06975820642910699, 0.32368100685760637) ,
  rgb (0.24688226237959, 0.06905363944920445, 0.31786993834254956) ,
  rgb (0.24367372557466271, 0.06841985515092269, 0.3121652405088837) ,
  rgb (0.2405081333229594, 0.0678571038148556, 0.3065705449367832) ,
  rgb (0.23739062429054825, 0.06736588805055552, 0.3010892218406587) ,
  rgb (0.23433055727563878, 0.0669355996616394, 0.295740099298676) ,
  rgb (0.23132955273021344, 0.06657618693909059, 0.29051361067988485) ,
  rgb (0.2283917709422868, 0.06628997924139618, 0.28541074411068496) ,
  rgb (0.22552164337737857, 0.0660781731193956, 0.28043398847505197) ,
  rgb (0.22272706739121817, 0.06593379067565194, 0.275597146520537) ,
  rgb (0.22001251100779617, 0.0658579189189076, 0.2709027999432586) ,
  rgb (0.21737845072382705, 0.06585966123356204, 0.2663420934966951) ,
  rgb (0.21482843531473683, 0.06594038561377849, 0.26191675992376573) ,
  rgb (0.21237411048541005, 0.06608502466175845, 0.2576516509356954) ,
  rgb (0.21001214221188125, 0.06630857391894718, 0.2535289048041211) ,
  rgb (0.2077442377448806, 0.06661453200418091, 0.24954644291943817) ,
  rgb (0.20558051999470117, 0.06699046239786874, 0.24572497420147632) ,
  rgb (0.20352007949514977, 0.06744417961242422, 0.24205576625191821) ,
  rgb (0.2015613376412984, 0.06798327102620025, 0.23852974228695395) ,
  rgb (0.19971571438603364, 0.06859271055370472, 0.23517094067076993) ,
  rgb (0.19794834061899208, 0.06931406607166066, 0.23194647381302336) ,
  rgb (0.1960826032659409, 0.07032122724242362, 0.22874673279569585) ,
  rgb (0.19410351363791453, 0.07160830485689157, 0.22558727307410353) ,
  rgb (0.19199449184606268, 0.0731828306492733, 0.22243385243433622) ,
  rgb (0.18975853639094634, 0.07501986186214377, 0.2193005075652994) ,
  rgb (0.18739228342697645, 0.07710209689958833, 0.21618875376309582) ,
  rgb (0.18488035509396164, 0.07942573027972388, 0.21307651648984993) ,
  rgb (0.18774482037046955, 0.07725158846803931, 0.21387448578597812) ,
  rgb (0.19049578401722037, 0.07531127841678764, 0.2146562337112265) ,
  rgb (0.1931548636579131, 0.07360681904011795, 0.21542362939081539) ,
  rgb (0.19571853588267552, 0.07215778103960274, 0.21617499187076789) ,
  rgb (0.19819343656336558, 0.07097462525273879, 0.21690975060032436) ,
  rgb (0.20058760685133747, 0.07006457614998421, 0.21762721310371608) ,
  rgb (0.20290365333558247, 0.06943524858045896, 0.21833167885096033) ,
  rgb (0.20531725273301316, 0.06891959226639757, 0.21911516689288835) ,
  rgb (0.20785704662965598, 0.06848439879702528, 0.22000133917653536) ,
  rgb (0.21052882914958676, 0.06812195249816172, 0.22098759107715404) ,
  rgb (0.2133313859647627, 0.06783014842602667, 0.2220704321302429) ,
  rgb (0.21625279838647882, 0.06761633027051639, 0.22324568672294431) ,
  rgb (0.21930503925136402, 0.06746578636294004, 0.22451023616807558) ,
  rgb (0.22247308588973624, 0.06738821405309284, 0.22585960379408354) ,
  rgb (0.2257539681670791, 0.06738213230014747, 0.22728984778098055) ,
  rgb (0.2291562027859284, 0.06743473087115257, 0.22879681433956656) ,
  rgb (0.23266299920501882, 0.06755710438847978, 0.23037617493752832) ,
  rgb (0.23627495835774248, 0.06774359820987802, 0.23202360805926608) ,
  rgb (0.23999586188690308, 0.06798502996477995, 0.23373434258507808) ,
  rgb (0.2438114972024792, 0.06828985152901187, 0.23550427698321885) ,
  rgb (0.247720929905011, 0.06865333790948652, 0.2373288009471749) ,
  rgb (0.25172899728289466, 0.0690646308260355, 0.23920260612763083) ,
  rgb (0.2558213554748177, 0.06953231029187984, 0.24112190491594204) ,
  rgb (0.25999463887892144, 0.07005385560386188, 0.24308218808684579) ,
  rgb (0.2642551220706094, 0.07061659562299544, 0.24507758869355967) ,
  rgb (0.2685909594817286, 0.07122671627792246, 0.24710443563450618) ,
  rgb (0.272997015188973, 0.07188355544616351, 0.2491584709323293) ,
  rgb (0.277471508091428, 0.07258296989925478, 0.2512349399594277) ,
  rgb (0.2820174629736694, 0.07331569321404097, 0.25332800295084507) ,
  rgb (0.28662309235899847, 0.07408846082680887, 0.2554347867371703) ,
  rgb (0.29128515387578635, 0.0748990498474667, 0.25755101595750435) ,
  rgb (0.2960004726065818, 0.07574533600095842, 0.25967245030364566) ,
  rgb (0.3007727681291869, 0.07661782433616476, 0.2617929409781967) ,
  rgb (0.30559226007249934, 0.07752196310753731, 0.2639100669211966) ,
  rgb (0.31045520848595526, 0.07845687167618218, 0.2660200572779356) ,
  rgb (0.3153587000920581, 0.07942099731524319, 0.2681190407694196) ,
  rgb (0.3202998655799406, 0.08041299473755484, 0.2702032289303951) ,
  rgb (0.3252788886040126, 0.08142839007654609, 0.27226772884656186) ,
  rgb (0.3302917447118144, 0.08246763389003825, 0.27430929404579435) ,
  rgb (0.3353335322445545, 0.08353243411900396, 0.2763253435679004) ,
  rgb (0.34040164359597463, 0.08462223619170267, 0.27831254595259397) ,
  rgb (0.345493557138718, 0.08573665496512634, 0.28026769921081435) ,
  rgb (0.3506067824603248, 0.08687555176033529, 0.28218770540182386) ,
  rgb (0.35573889947341125, 0.08803897435024335, 0.2840695897279818) ,
  rgb (0.36088752387578377, 0.0892271943627452, 0.28591050458531014) ,
  rgb (0.36605031412464006, 0.0904406854276979, 0.2877077458811747) ,
  rgb (0.3712250843130934, 0.09167999748026273, 0.2894586539763317) ,
  rgb (0.3764103053221462, 0.09294519809377791, 0.2911602415731392) ,
  rgb (0.38160247377467543, 0.09423873126371218, 0.2928110750626949) ,
  rgb (0.3867993907954417, 0.09556181960083443, 0.29440901248173756) ,
  rgb (0.39199887556812907, 0.09691583650296684, 0.2959521200550908) ,
  rgb (0.39719876876325577, 0.09830232096827862, 0.2974385647628578) ,
  rgb (0.40239692379737496, 0.09972293031495055, 0.2988667436973397) ,
  rgb (0.4075912039268871, 0.10117945586419633, 0.300235195077286) ,
  rgb (0.41277985630360303, 0.1026734006932461, 0.3015422643746897) ,
  rgb (0.41796105205173684, 0.10420644885760968, 0.3027865203963184) ,
  rgb (0.42313214269556043, 0.10578120994917611, 0.3039675809469457) ,
  rgb (0.4282910131578975, 0.1073997763055258, 0.30508479060294547) ,
  rgb (0.4334355841041439, 0.1090642347484701, 0.3061376792828915) ,
  rgb (0.4385637818793154, 0.11077667828375456, 0.30712600062348083) ,
  rgb (0.44367358645071275, 0.11253912421257944, 0.3080497309546545) ,
  rgb (0.4487629917317482, 0.1143535557462255, 0.30890905921943196) ,
  rgb (0.4538300508699989, 0.11622183788331528, 0.3097044124984492) ,
  rgb (0.45887288947308297, 0.11814571137706886, 0.3104363697903881) ,
  rgb (0.46389102840284874, 0.12012561256850712, 0.31110343446582983) ,
  rgb (0.46888111384598413, 0.12216445576414045, 0.31170911458932665) ,
  rgb (0.473841437035254, 0.12426354237989065, 0.31225470169927194) ,
  rgb (0.47877034239726296, 0.12642401401409453, 0.3127417273582196) ,
  rgb (0.48366628618847957, 0.1286467902201389, 0.31317188565991266) ,
  rgb (0.48852847371852987, 0.13093210934893723, 0.31354553695453014) ,
  rgb (0.49335504375145617, 0.13328091630401023, 0.31386561956734976) ,
  rgb (0.4981443546207415, 0.13569380302451714, 0.314135190862664) ,
  rgb (0.5028952497497061, 0.13817086581280427, 0.3143566215383367) ,
  rgb (0.5076068118105369, 0.14071192654913128, 0.3145320012008257) ,
  rgb (0.5122783510532176, 0.14331656120063752, 0.3146630922831542) ,
  rgb (0.5169084880054446, 0.14598463068714407, 0.3147540759228004) ,
  rgb (0.5214965286322996, 0.14871544765633712, 0.3148076795453443) ,
  rgb (0.5260418962547748, 0.15150818660835483, 0.31482653406646727) ,
  rgb (0.5305442048985645, 0.15436183633886777, 0.3148129978918713) ,
  rgb (0.5350027976174474, 0.15727540775107324, 0.3147708520739653) ,
  rgb (0.5394173664919906, 0.16024769309971934, 0.31470295028655965) ,
  rgb (0.5437877131360856, 0.16327738551419116, 0.31461204226295625) ,
  rgb (0.5481137003346762, 0.1663630904279047, 0.3145010299091471) ,
  rgb (0.5523952157271191, 0.16950338809328983, 0.3143729155461537) ,
  rgb (0.5566322903496934, 0.17269677158182117, 0.31423043195101424) ,
  rgb (0.5608249903911717, 0.17594170887918095, 0.31407639883970623) ,
  rgb (0.564973435290177, 0.17923664950367169, 0.3139136046337036) ,
  rgb (0.5690778478401143, 0.18258004462335425, 0.3137444095679653) ,
  rgb (0.5731384575410787, 0.18597036007065024, 0.3135712686852) ,
  rgb (0.5771555081299204, 0.18940601489760422, 0.3133970433357208) ,
  rgb (0.5811293276158656, 0.19288548904692518, 0.3132239939418394) ,
  rgb (0.5850602439646688, 0.19640737049066315, 0.3130540116373273) ,
  rgb (0.5889486193554471, 0.19997020971775276, 0.31288922211590126) ,
  rgb (0.5927948053652026, 0.20357251410079796, 0.3127323483930494) ,
  rgb (0.5965991810912237, 0.207212956082026, 0.3125852303112123) ,
  rgb (0.6003621301041158, 0.21089030138947745, 0.3124493441041469) ,
  rgb (0.6040840169673274, 0.21460331490206347, 0.31232652641170694) ,
  rgb (0.6077652399481865, 0.21835070166659282, 0.312219032918702) ,
  rgb (0.6114062072731884, 0.22213124697023234, 0.3121288139643524) ,
  rgb (0.6150072323639137, 0.22594402043981826, 0.3120568068576574) ,
  rgb (0.6185686525887719, 0.2297879924917992, 0.3120046383872893) ,
  rgb (0.6220907982108261, 0.2336621873300741, 0.3119738327362739) ,
  rgb (0.6255741650043496, 0.23756535071152696, 0.3119669831491227) ,
  rgb (0.6290189201698587, 0.24149689191922535, 0.3119844719564572) ,
  rgb (0.6324253485421027, 0.24545598775548677, 0.3120276597462445) ,
  rgb (0.6357937104834237, 0.24944185818822678, 0.3120979395330059) ,
  rgb (0.6391243387840212, 0.2534536546198314, 0.3121968961206398) ,
  rgb (0.642417577481186, 0.257490519876798, 0.31232631707560987) ,
  rgb (0.6456734938264543, 0.2615520316161528, 0.31248673753935263) ,
  rgb (0.6488923016945825, 0.2656375533620908, 0.3126794181957019) ,
  rgb (0.652074172902773, 0.269746505252367, 0.3129056060581917) ,
  rgb (0.6552193260932713, 0.2738782665241015, 0.3131666792687211) ,
  rgb (0.6583280801134499, 0.2780321095766563, 0.3134643447952643) ,
  rgb (0.6614003753260178, 0.28220778870555907, 0.3137991292649849) ,
  rgb (0.6644363246987884, 0.2864048361425618, 0.31417223403606975) ,
  rgb (0.6674360376636913, 0.29062280081258873, 0.31458483752056837) ,
  rgb (0.670399595476762, 0.29486126309253047, 0.3150381395687221) ,
  rgb (0.6733272556481733, 0.29911962764489264, 0.3155337232398221) ,
  rgb (0.6762189792440975, 0.30339762792450425, 0.3160724937230589) ,
  rgb (0.6790747402815734, 0.30769497879760166, 0.31665545668946665) ,
  rgb (0.6818945715094452, 0.31201133280550686, 0.3172838048924495) ,
  rgb (0.6846785094249453, 0.3163463482122221, 0.31795870784057567) ,
  rgb (0.6874265643516962, 0.32069970535138104, 0.3186813762227769) ,
  rgb (0.6901389321505248, 0.32507091815606004, 0.319453323328983) ,
  rgb (0.6928154484676493, 0.32945984647042675, 0.3202754315314667) ,
  rgb (0.6954560834689112, 0.33386622163232865, 0.3211488430698579) ,
  rgb (0.6980608153581771, 0.3382897632604862, 0.3220747885521809) ,
  rgb (0.700629624772421, 0.34273019305341756, 0.32305449047765694) ,
  rgb (0.7031624945881415, 0.34718723719598, 0.32408913679491225) ,
  rgb (0.7056595112261009, 0.3516605297812094, 0.32518014084085567) ,
  rgb (0.7081205956842048, 0.356149855233803, 0.32632861885644465) ,
  rgb (0.7105456546582587, 0.36065500290840113, 0.3275357416278876) ,
  rgb (0.7129346683977347, 0.36517570519856757, 0.3288027427038317) ,
  rgb (0.7152876061484729, 0.3697117022522345, 0.3301308728723546) ,
  rgb (0.7176044490813385, 0.3742627271068619, 0.3315213862095893) ,
  rgb (0.7198852149054985, 0.37882848839337313, 0.332975552002454) ,
  rgb (0.7221299918421461, 0.3834086450896306, 0.33449469983585844) ,
  rgb (0.7243386564778159, 0.38800301593162145, 0.3360799596569183) ,
  rgb (0.7265112290022755, 0.3926113126792577, 0.3377325942005665) ,
  rgb (0.7286477385671655, 0.39723324476747235, 0.33945384341064017) ,
  rgb (0.7307482075484517, 0.401868526884681, 0.3412449533046818) ,
  rgb (0.7328127050626875, 0.4065168468778026, 0.3431071517341082) ,
  rgb (0.7348413359856494, 0.4111778700451951, 0.3450416947080907) ,
  rgb (0.7368342217358587, 0.4158512585029011, 0.347049785207584) ,
  rgb (0.7387914002459927, 0.4205367299231533, 0.34913260148542435) ,
  rgb (0.7407130161950609, 0.4252339389526239, 0.35129130890802607) ,
  rgb (0.7425992159973317, 0.42994254036133867, 0.3535270924537459) ,
  rgb (0.7444501867657067, 0.4346621718461711, 0.35584108091122535) ,
  rgb (0.7462661578916344, 0.439392450449735, 0.3582343914230064) ,
  rgb (0.7480473927555956, 0.44413297780351974, 0.36070813602540136) ,
  rgb (0.7497942054717047, 0.4488833348154881, 0.3632633755836028) ,
  rgb (0.7515068504589166, 0.45364314496866825, 0.36590112443835765) ,
  rgb (0.7531856636904657, 0.45841199172949604, 0.3686223664223477) ,
  rgb (0.7548310506695954, 0.46318942799460555, 0.3714280448394211) ,
  rgb (0.7564434157714071, 0.4679750143794846, 0.37431909037543515) ,
  rgb (0.7580232553845584, 0.4727682731566229, 0.3772963553109668) ,
  rgb (0.7595711110534006, 0.4775687122205708, 0.380360657784311) ,
  rgb (0.7610876378057071, 0.48237579130289127, 0.3835127572385229) ,
  rgb (0.7625733355405261, 0.48718906673415824, 0.38675335037837993) ,
  rgb (0.7640288560928866, 0.49200802533379656, 0.39008308392311997) ,
  rgb (0.7654549259333051, 0.4968321290972723, 0.3935025400011538) ,
  rgb (0.7668522895064389, 0.5016608471009063, 0.39701221751773474) ,
  rgb (0.768221765997353, 0.5064936237128791, 0.40061257089416885) ,
  rgb (0.7695642334401418, 0.5113298901696085, 0.4043039806968248) ,
  rgb (0.7708809196230247, 0.516168926434691, 0.40808667584648967) ,
  rgb (0.7721725722960555, 0.5210102658711383, 0.4119608998712287) ,
  rgb (0.7734402182988989, 0.5258533209345156, 0.41592679539764366) ,
  rgb (0.774684947460632, 0.5306974938477673, 0.4199844035696376) ,
  rgb (0.775907907306857, 0.5355421788246119, 0.42413367909988375) ,
  rgb (0.7771103295521099, 0.5403867491056124, 0.4283745037125848) ,
  rgb (0.7782934580763312, 0.545230594884266, 0.432706647838971) ,
  rgb (0.7794586273150664, 0.5500730841397727, 0.4371297985644476) ,
  rgb (0.7806077474948377, 0.5549133574489061, 0.4416433242636464) ,
  rgb (0.7817418047898184, 0.5597509805259486, 0.44624687186865436) ,
  rgb (0.7828622526444091, 0.5645853311116688, 0.45093985823706345) ,
  rgb (0.7839706083641448, 0.5694157832671042, 0.4557215474289206) ,
  rgb (0.7850684501960684, 0.5742417003617839, 0.46059116206904965) ,
  rgb (0.7861573713233296, 0.5790624629815756, 0.465547782819184) ,
  rgb (0.7872390410818835, 0.5838774374455721, 0.47059039582133383) ,
  rgb (0.7883151404562396, 0.5886860017356244, 0.4757179187907608) ,
  rgb (0.7893873776625194, 0.5934875421745599, 0.48092913815357724) ,
  rgb (0.7904577684772788, 0.5982813427706246, 0.48622257801969754) ,
  rgb (0.7915283284347561, 0.603066705931472, 0.49159667021646397) ,
  rgb (0.7926003430423745, 0.6078432208703702, 0.4970502062153201) ,
  rgb (0.7936755969866496, 0.6126102933407219, 0.5025816129126943) ,
  rgb (0.7947558597265404, 0.617367344002207, 0.5081892121310299) ,
  rgb (0.7958429237958377, 0.6221137880845115, 0.5138712409190979) ,
  rgb (0.7969385471995161, 0.626849056792967, 0.5196258425240281) ,
  rgb (0.7980444781513664, 0.6315725822508955, 0.5254510814483478) ,
  rgb (0.7991624518501963, 0.6362837937202919, 0.5313449594256143) ,
  rgb (0.8002941538975398, 0.6409821330674986, 0.5373053518514104) ,
  rgb (0.8014412429256005, 0.6456670345921877, 0.5433300863249918) ,
  rgb (0.8026053114611295, 0.6503379374810385, 0.5494169158460365) ,
  rgb (0.8037879253107763, 0.6549942654947263, 0.5555635086708381) ,
  rgb (0.804990547908103, 0.6596354502756416, 0.5617674511054698) ,
  rgb (0.8062146052692706, 0.6642608958528229, 0.5680262917864979) ,
  rgb (0.8074614045096935, 0.6688700095398864, 0.5743374637345958) ,
  rgb (0.8087321917008969, 0.6734621670219452, 0.5806983480557674) ,
  rgb (0.8100280946652069, 0.6780367267397182, 0.5871062690808275) ,
  rgb (0.8113501401176333, 0.6825930154624339, 0.5935584890905076) ,
  rgb (0.8126992203988149, 0.6871303371461888, 0.600052148204351) ,
  rgb (0.8140761104699334, 0.6916479479148213, 0.6065843782630862) ,
  rgb (0.8154814662727948, 0.6961450550830809, 0.6131522120932265) ,
  rgb (0.8169157577505589, 0.7006208301478398, 0.6197526063725792) ,
  rgb (0.8183793116449822, 0.705074381896351, 0.626382454789333) ,
  rgb (0.8198723065045529, 0.7095047497878748, 0.6330385704006711) ,
  rgb (0.8213947205565636, 0.7139109141951604, 0.6397176669767276) ,
  rgb (0.8229463511042843, 0.7182917733129006, 0.6464164243818421) ,
  rgb (0.8245268129450285, 0.7226461431208888, 0.653131379154226) ,
  rgb (0.8261354971058026, 0.7269727551823826, 0.659859001562165) ,
  rgb (0.8277716072353446, 0.7312702332407809, 0.6665957020468297) ,
  rgb (0.8294340781648147, 0.7355371221572935, 0.6733377200930191) ,
  rgb (0.8311216352909631, 0.7397718464763862, 0.6800812520363146) ,
  rgb (0.8328327718577798, 0.7439727181745988, 0.6868223587464855) ,
  rgb (0.8345656905566583, 0.7481379479992134, 0.6935569764986385) ,
  rgb (0.8363189884473793, 0.7522654895287526, 0.7002799902886496) ,
  rgb (0.8380912347613196, 0.7563531486080863, 0.7069856139021298) ,
  rgb (0.8398783988412087, 0.7603990719977968, 0.7136714781112923) ,
  rgb (0.8416775076684515, 0.7644010120098295, 0.7203329938728462) ,
  rgb (0.843485292229337, 0.7683566039987018, 0.7269653699897204) ,
  rgb (0.8452981073195511, 0.7722633860104472, 0.7335636824054149) ,
  rgb (0.847111955079651, 0.7761188023604716, 0.7401227576280706) ,
  rgb (0.8489224556311764, 0.7799202140765015, 0.7466371929366437) ,
  rgb (0.8507269702317879, 0.7836645734238389, 0.7530974636118285) ,
  rgb (0.8525190720770844, 0.7873493613354844, 0.7594994148789691) ,
  rgb (0.8542921961147046, 0.7909719677709199, 0.765838014779141) ,
  rgb (0.856040223147254, 0.7945296360155061, 0.7721061003767414) ,
  rgb (0.857756629435049, 0.7980196314271393, 0.778295716672475) ,
  rgb (0.8594346370300241, 0.8014392309950078, 0.7843978875138392) ,
  rgb (0.8610711702756552, 0.8047851790981223, 0.7903952966373629) ,
  rgb (0.8626560105112757, 0.8080552380426153, 0.796282666437655) ,
  rgb (0.8641834372394103, 0.8112464422465354, 0.8020461269686395) ,
  rgb (0.8656493432560532, 0.8143554406751491, 0.8076697232416455) ,
  rgb (0.867053149070485, 0.8173780404191124, 0.813134196269114) ,
  rgb (0.8683995469581863, 0.8203087551218152, 0.8184163896312899) ,
  rgb (0.8696913150261381, 0.8231415885956916, 0.8235047668317317) ,
  rgb (0.8709384671729751, 0.8258685788943851, 0.8283849726114961) ,
  rgb (0.8721533197845432, 0.8284805282370967, 0.8330486712880828) ,
  rgb (0.8733517136091627, 0.8309671525127262, 0.8374885100119709) ,
  rgb (0.8745379332026019, 0.8333197294864546, 0.8417192535806901) ,
  rgb (0.875714587099614, 0.8355302318472394, 0.8457553751902708) ,
  rgb (0.8768784845161469, 0.8375923807118654, 0.8496137354915025) ,
  rgb (0.8780229843664901, 0.8395016561854007, 0.8533064535245892) ,
  rgb (0.8791324424079277, 0.8412555488447591, 0.8568557229103964) ,
  rgb (0.8801929331569581, 0.8428522482477862, 0.8602739992715663) ,
  rgb (0.8811916987134195, 0.8442906671771735, 0.8635659516866988) ,
  rgb (0.8821154248940161, 0.8455700725455935, 0.8667376504623333) ,
  rgb (0.8829516859544853, 0.8466897027569927, 0.8697961704819097) ,
  rgb (0.8836912714589804, 0.8476489176151927, 0.8727414710144156) ,
  rgb (0.8843271305411354, 0.8484474157205542, 0.8755678522824297) ,
  rgb (0.8848513815990857, 0.849084264228938, 0.8782823528537247) ,
  rgb (0.8852589797263047, 0.8495589281098921, 0.8808841479402484) ,
  rgb (0.8855471481195238, 0.8498717428363158, 0.8833620612117095) ,
  rgb (0.8857115512284565, 0.8500218611585632, 0.8857253899008712)
});


list_data twilight_shifted = list_data(new pen[] {
  rgb (0.18739228342697645, 0.07710209689958833, 0.21618875376309582) ,
  rgb (0.18975853639094634, 0.07501986186214377, 0.2193005075652994) ,
  rgb (0.19199449184606268, 0.0731828306492733, 0.22243385243433622) ,
  rgb (0.19410351363791453, 0.07160830485689157, 0.22558727307410353) ,
  rgb (0.1960826032659409, 0.07032122724242362, 0.22874673279569585) ,
  rgb (0.19794834061899208, 0.06931406607166066, 0.23194647381302336) ,
  rgb (0.19971571438603364, 0.06859271055370472, 0.23517094067076993) ,
  rgb (0.2015613376412984, 0.06798327102620025, 0.23852974228695395) ,
  rgb (0.20352007949514977, 0.06744417961242422, 0.24205576625191821) ,
  rgb (0.20558051999470117, 0.06699046239786874, 0.24572497420147632) ,
  rgb (0.2077442377448806, 0.06661453200418091, 0.24954644291943817) ,
  rgb (0.21001214221188125, 0.06630857391894718, 0.2535289048041211) ,
  rgb (0.21237411048541005, 0.06608502466175845, 0.2576516509356954) ,
  rgb (0.21482843531473683, 0.06594038561377849, 0.26191675992376573) ,
  rgb (0.21737845072382705, 0.06585966123356204, 0.2663420934966951) ,
  rgb (0.22001251100779617, 0.0658579189189076, 0.2709027999432586) ,
  rgb (0.22272706739121817, 0.06593379067565194, 0.275597146520537) ,
  rgb (0.22552164337737857, 0.0660781731193956, 0.28043398847505197) ,
  rgb (0.2283917709422868, 0.06628997924139618, 0.28541074411068496) ,
  rgb (0.23132955273021344, 0.06657618693909059, 0.29051361067988485) ,
  rgb (0.23433055727563878, 0.0669355996616394, 0.295740099298676) ,
  rgb (0.23739062429054825, 0.06736588805055552, 0.3010892218406587) ,
  rgb (0.2405081333229594, 0.0678571038148556, 0.3065705449367832) ,
  rgb (0.24367372557466271, 0.06841985515092269, 0.3121652405088837) ,
  rgb (0.24688226237959, 0.06905363944920445, 0.31786993834254956) ,
  rgb (0.2501284530619938, 0.06975820642910699, 0.32368100685760637) ,
  rgb (0.25340685873736807, 0.07053358292685183, 0.3295945757321303) ,
  rgb (0.256711916510839, 0.07138010624208224, 0.3356064898460009) ,
  rgb (0.26003895187439957, 0.0722947810433622, 0.3417175669546984) ,
  rgb (0.26338121807151404, 0.07328165793989708, 0.34791888996380105) ,
  rgb (0.26673233211995284, 0.0743440180285462, 0.3542027606624096) ,
  rgb (0.2700863774911405, 0.07548367558427554, 0.36056376228111864) ,
  rgb (0.2734373934245081, 0.07670280023749607, 0.36699616136347685) ,
  rgb (0.2767793961581559, 0.07800394103378822, 0.37349382846504675) ,
  rgb (0.2801063857697547, 0.07938999480226153, 0.38005028528138707) ,
  rgb (0.28341239797185225, 0.08086415336549937, 0.38665868550105914) ,
  rgb (0.28669151388283165, 0.08242987384848069, 0.39331182532243375) ,
  rgb (0.2899379244517661, 0.08409078829085731, 0.40000214725256295) ,
  rgb (0.2931459309698525, 0.08585065688962071, 0.40672178082365834) ,
  rgb (0.2963100038890529, 0.08771325096046395, 0.41346259134143476) ,
  rgb (0.2994248396021477, 0.08968225371675004, 0.42021619665853854) ,
  rgb (0.3024853636457425, 0.0917611873973036, 0.42697404043749887) ,
  rgb (0.30548675819945936, 0.09395276484082374, 0.4337284999936111) ,
  rgb (0.30842444457210105, 0.09625938534057774, 0.44047194882050544) ,
  rgb (0.31129434479712365, 0.0986847719340522, 0.4471931371516162) ,
  rgb (0.3140927841475553, 0.10123077676403242, 0.45388335612058467) ,
  rgb (0.316816480890235, 0.10389861387653518, 0.46053414662739794) ,
  rgb (0.31946262395497965, 0.1066887988239266, 0.46713728801395243) ,
  rgb (0.3220288227606932, 0.10960114111258401, 0.4736849472572688) ,
  rgb (0.32451307931207785, 0.11263459791730848, 0.48017007211645196) ,
  rgb (0.3269137124539796, 0.1157873496841953, 0.4865864649569746) ,
  rgb (0.3292300520323141, 0.11905764321981127, 0.49292595612342666) ,
  rgb (0.33146154891145124, 0.12244245263391232, 0.4991827458843107) ,
  rgb (0.33360804901486, 0.1259381830100592, 0.505351647966549) ,
  rgb (0.33566978565015315, 0.12954074251271822, 0.5114280721516895) ,
  rgb (0.3376473209067111, 0.13324562282438077, 0.5174080757397947) ,
  rgb (0.33954168752669694, 0.1370480189671817, 0.5232879753508524) ,
  rgb (0.34135411074506483, 0.1409427920655632, 0.5290650094033675) ,
  rgb (0.34308600291572294, 0.14492465359918028, 0.534737042820671) ,
  rgb (0.34473901574536375, 0.1489882058982641, 0.5403024592040654) ,
  rgb (0.3463150717579309, 0.15312802296627787, 0.5457599924248665) ,
  rgb (0.3478165323877778, 0.1573386351115298, 0.5511085345270326) ,
  rgb (0.34924513554955644, 0.16161477763045118, 0.5563483747416378) ,
  rgb (0.3506030444193101, 0.1659512998472086, 0.5614796470399323) ,
  rgb (0.3518924860887379, 0.17034320478524959, 0.5665028491121665) ,
  rgb (0.35311574421123737, 0.17478570377561287, 0.5714187152355529) ,
  rgb (0.3542753496066376, 0.17927413271618647, 0.5762280966066872) ,
  rgb (0.3553741530690672, 0.18380392577704466, 0.580931914318328) ,
  rgb (0.35641381143126327, 0.18837119869242164, 0.5855320765920552) ,
  rgb (0.35739663292915563, 0.1929720819784246, 0.5900301125106313) ,
  rgb (0.3583248996661781, 0.197602942459778, 0.5944276851750107) ,
  rgb (0.35920088560930186, 0.20226037920758122, 0.5987265295935138) ,
  rgb (0.36002681809096376, 0.20694122817889948, 0.6029284543191687) ,
  rgb (0.36080493826624654, 0.21164251793458128, 0.6070353217206471) ,
  rgb (0.36153829010998356, 0.21636111429594002, 0.6110488067964093) ,
  rgb (0.36222809558295926, 0.22109488427338303, 0.6149711234609613) ,
  rgb (0.3628764356004103, 0.2258414929328703, 0.6188041741082302) ,
  rgb (0.36348537610385145, 0.2305987621839642, 0.6225498862239288) ,
  rgb (0.3640569302208851, 0.23536470386204195, 0.6262101345195302) ,
  rgb (0.3645930889012501, 0.24013747024823828, 0.629786801550261) ,
  rgb (0.3650957984588681, 0.24491536803550484, 0.6332817352005559) ,
  rgb (0.3655669837353898, 0.24969683475296395, 0.6366967518748858) ,
  rgb (0.36600853966739794, 0.25448043878086224, 0.6400336180336859) ,
  rgb (0.3664224325155063, 0.25926481158628106, 0.6432940914076554) ,
  rgb (0.3668108554010799, 0.26404857724525643, 0.6464799165290824) ,
  rgb (0.36717513650699446, 0.26883085667326956, 0.6495927229789225) ,
  rgb (0.36751707094367697, 0.2736106331709098, 0.6526341171161864) ,
  rgb (0.3678384369653108, 0.2783869718129776, 0.655605668384537) ,
  rgb (0.3681410147989972, 0.2831590122118299, 0.6585088880697231) ,
  rgb (0.36842655638020444, 0.2879259643777846, 0.661345269109446) ,
  rgb (0.3686968223189527, 0.292687098561501, 0.6641162529823691) ,
  rgb (0.3689535530659678, 0.29744175492366276, 0.6668232208982426) ,
  rgb (0.36919847837592484, 0.3021893217650707, 0.6694675433161452) ,
  rgb (0.3694333459127623, 0.3069292355186234, 0.6720505284912062) ,
  rgb (0.36965987626565955, 0.3116609876295197, 0.6745734474341955) ,
  rgb (0.3698798032902536, 0.31638410101153364, 0.6770375543809057) ,
  rgb (0.37009487130772695, 0.3210981375964933, 0.6794440539905685) ,
  rgb (0.37030682071842685, 0.32580269697872455, 0.6817941168448668) ,
  rgb (0.37051738634484427, 0.3304974124430785, 0.6840888878885721) ,
  rgb (0.3707283327942267, 0.33518193808489577, 0.6863294816960677) ,
  rgb (0.3709415155133733, 0.33985591488818123, 0.6885170337950512) ,
  rgb (0.37115856636209105, 0.3445191141023017, 0.6906525358646499) ,
  rgb (0.37138124223736607, 0.34917126878479027, 0.6927370347192883) ,
  rgb (0.3716113323440048, 0.3538121372967869, 0.6947714991938089) ,
  rgb (0.3718506155898596, 0.3584414828587522, 0.6967569581025654) ,
  rgb (0.3721008970244382, 0.3630590973698238, 0.6986943461507372) ,
  rgb (0.37236397858465387, 0.36766477862108266, 0.7005846349652077) ,
  rgb (0.37264166757849604, 0.3722583500483685, 0.7024287314570723) ,
  rgb (0.3729357864666503, 0.3768396383521984, 0.7042275578058994) ,
  rgb (0.37324816143326384, 0.38140848555753937, 0.7059820097480604) ,
  rgb (0.3735806215098284, 0.3859647438605754, 0.7076929760731058) ,
  rgb (0.3739349933047578, 0.3905082752937583, 0.7093613429347845) ,
  rgb (0.3743131319931234, 0.3950389482828331, 0.7109879652237746) ,
  rgb (0.3747168601930223, 0.3995566498711684, 0.7125736851650046) ,
  rgb (0.37514802505380473, 0.4040612609993941, 0.7141193695725726) ,
  rgb (0.375608469194424, 0.40855267638072096, 0.7156258509158755) ,
  rgb (0.37610001286385814, 0.4130307995247706, 0.7170939691992023) ,
  rgb (0.37662448930806297, 0.41749553747893614, 0.7185245473617611) ,
  rgb (0.3771837184425123, 0.4219468022345483, 0.7199184152447577) ,
  rgb (0.3777794975351373, 0.4263845142616835, 0.7212763999353023) ,
  rgb (0.378413635091359, 0.43080859411413064, 0.7225993199306104) ,
  rgb (0.3790878928311076, 0.43521897612544935, 0.7238879869132313) ,
  rgb (0.3798040374484875, 0.4396155882122263, 0.7251432377876109) ,
  rgb (0.3805638069656562, 0.4439983720863372, 0.7263658704513531) ,
  rgb (0.3813688793045416, 0.4483672766927786, 0.7275567131714135) ,
  rgb (0.38222094988570376, 0.45272225034283325, 0.7287165614400378) ,
  rgb (0.3831216808440275, 0.457063235814072, 0.7298462679135326) ,
  rgb (0.38407269378943537, 0.46139018782416635, 0.7309466543290268) ,
  rgb (0.3850755679365139, 0.46570306719930193, 0.732018540336905) ,
  rgb (0.386131841788921, 0.4700018340988123, 0.7330627749243106) ,
  rgb (0.3872430145933025, 0.4742864593363539, 0.7340801678785439) ,
  rgb (0.3884105330084243, 0.47855691131792805, 0.7350715764115726) ,
  rgb (0.38963580160340855, 0.48281316715123496, 0.7360378254693274) ,
  rgb (0.390920175719949, 0.4870552025122304, 0.7369797713388125) ,
  rgb (0.39226494876209317, 0.4912830033289926, 0.7378982478447508) ,
  rgb (0.39367135736822567, 0.4954965577745118, 0.738794102296364) ,
  rgb (0.3951405880820763, 0.4996958532637776, 0.7396682021171571) ,
  rgb (0.3966737490566561, 0.5038808905384797, 0.7405213858051674) ,
  rgb (0.3982719152586337, 0.5080516653927614, 0.7413545091809972) ,
  rgb (0.39993606933454834, 0.5122081814321852, 0.7421684457131799) ,
  rgb (0.40166714010896104, 0.5163504496968876, 0.7429640345324835) ,
  rgb (0.40346600333905397, 0.5204784765384003, 0.7437421522356709) ,
  rgb (0.4053334378930318, 0.5245922817498312, 0.7445036583670813) ,
  rgb (0.4072701869421907, 0.5286918801105741, 0.7452494263758127) ,
  rgb (0.4092768899914751, 0.5327773017713032, 0.7459803063570782) ,
  rgb (0.4113541469730457, 0.5368485776500593, 0.7466971285506578) ,
  rgb (0.4135024574331473, 0.5409057477109848, 0.7474007329754309) ,
  rgb (0.4157223260454232, 0.544948827153504, 0.7480920445900052) ,
  rgb (0.4180141407923363, 0.5489778542188889, 0.7487719316700112) ,
  rgb (0.42037822361396326, 0.5529928715810817, 0.7494412559451894) ,
  rgb (0.4228148567577266, 0.5569939212699658, 0.7501008698822764) ,
  rgb (0.42532423665011354, 0.560981049599503, 0.7507516498900512) ,
  rgb (0.42790652284925834, 0.5649543060909209, 0.7513944352191484) ,
  rgb (0.4305617907305757, 0.5689137457245718, 0.752030080993127) ,
  rgb (0.43329008867358393, 0.5728594162560667, 0.7526594653256667) ,
  rgb (0.4360913895835637, 0.5767913799818608, 0.7532834105961016) ,
  rgb (0.43896563958048396, 0.5807096924109849, 0.7539027620828594) ,
  rgb (0.441912717666964, 0.5846144110017557, 0.7545183888441067) ,
  rgb (0.4449324685421538, 0.5885055998308617, 0.7551311041857901) ,
  rgb (0.4480247093358917, 0.5923833145214658, 0.7557417647410792) ,
  rgb (0.45118918687617743, 0.5962476205135354, 0.7563512064324664) ,
  rgb (0.45442563977552913, 0.6000985950385866, 0.7569601366060649) ,
  rgb (0.45773377230160567, 0.6039363004658646, 0.7575693690185916) ,
  rgb (0.4611132664702388, 0.607760777169989, 0.7581798643680714) ,
  rgb (0.4645637671630393, 0.6115720882286415, 0.7587924262302581) ,
  rgb (0.46808490970531597, 0.6153702869579029, 0.7594078989109274) ,
  rgb (0.4716762951887709, 0.6191554324288464, 0.7600270922788305) ,
  rgb (0.47533752394906287, 0.6229275728383581, 0.7606508557181775) ,
  rgb (0.47906816236197386, 0.6266867625186013, 0.7612800037566242) ,
  rgb (0.4828677867260272, 0.6304330455306234, 0.761915371498953) ,
  rgb (0.4867359599430568, 0.6341664625110051, 0.7625578008592404) ,
  rgb (0.4906722493856122, 0.6378870485884708, 0.7632081276316384) ,
  rgb (0.4946761847863938, 0.6415948411950443, 0.7638671900213091) ,
  rgb (0.4987473366135607, 0.6452898684900934, 0.7645357873418008) ,
  rgb (0.5028853540415561, 0.6489721673409526, 0.7652144671817491) ,
  rgb (0.5070896957645166, 0.6526417240314645, 0.7659044566083585) ,
  rgb (0.5113599254160193, 0.6562985398183186, 0.7666066378064533) ,
  rgb (0.5156955988596057, 0.65994260812898, 0.7673219148959617) ,
  rgb (0.5200962739223556, 0.6635739143403039, 0.768051194033441) ,
  rgb (0.5245615147059358, 0.6671924299614223, 0.7687954171423095) ,
  rgb (0.529090861832473, 0.6707981230280622, 0.7695555229231313) ,
  rgb (0.533683891477284, 0.6743909370521273, 0.7703325054879735) ,
  rgb (0.5383401557517628, 0.677970811290954, 0.7711273443905772) ,
  rgb (0.5430593242401823, 0.6815376725306588, 0.7719407969783508) ,
  rgb (0.5478409815301863, 0.6850914221850988, 0.7727738647344087) ,
  rgb (0.5526845058934713, 0.6886319451516638, 0.7736279426902245) ,
  rgb (0.5575894041960517, 0.6921591145825405, 0.7745041337932782) ,
  rgb (0.5625551535721934, 0.6956727838162965, 0.7754035914230984) ,
  rgb (0.5675811785386197, 0.6991727930264627, 0.776327485342753) ,
  rgb (0.5726668948158774, 0.7026589535425779, 0.7772770268091199) ,
  rgb (0.5778116438998202, 0.7061310615715398, 0.7782534512102552) ,
  rgb (0.5830148703693241, 0.7095888767699747, 0.7792578182047659) ,
  rgb (0.588275797805555, 0.7130321464645814, 0.7802914140563652) ,
  rgb (0.5935933569683722, 0.7164606049658685, 0.781355882376401) ,
  rgb (0.5989665814482068, 0.7198739489224673, 0.7824525989934664) ,
  rgb (0.6043943420027486, 0.7232718614323369, 0.7835829559353559) ,
  rgb (0.6098754317609306, 0.7266539875975829, 0.7847483573269471) ,
  rgb (0.6154084641177048, 0.7300199523273969, 0.7859502270675048) ,
  rgb (0.6209919306481755, 0.733369347989232, 0.7871899462469658) ,
  rgb (0.6266240202260459, 0.7367017540369944, 0.7884690131633456) ,
  rgb (0.6323027138710603, 0.740016721601314, 0.7897889276264043) ,
  rgb (0.6380258682854598, 0.7433137671403319, 0.7911510045957317) ,
  rgb (0.6437910831099849, 0.7465923800833657, 0.7925565320130605) ,
  rgb (0.6495957300425348, 0.7498520122194177, 0.7940067402149911) ,
  rgb (0.6554369232645472, 0.7530920875676843, 0.7955027112903105) ,
  rgb (0.6613112930078745, 0.7563120270871903, 0.7970456043491897) ,
  rgb (0.6672147980375281, 0.7595112803730375, 0.7986367465453776) ,
  rgb (0.6731442255942621, 0.7626890873389048, 0.8002762854580953) ,
  rgb (0.6790955449988215, 0.765844721313959, 0.8019646617300199) ,
  rgb (0.6850644615439593, 0.7689774029354699, 0.8037020626717691) ,
  rgb (0.691046410093091, 0.7720862946067809, 0.8054884169067907) ,
  rgb (0.6970365443886174, 0.7751705000806606, 0.8073233538006345) ,
  rgb (0.7030297722540817, 0.7782290497335813, 0.8092061884805697) ,
  rgb (0.709020781345393, 0.7812608871607091, 0.8111359185511793) ,
  rgb (0.7150040307625213, 0.7842648709158119, 0.8131111655994991) ,
  rgb (0.7209728066553678, 0.7872399592345264, 0.8151307392087926) ,
  rgb (0.726921775128297, 0.7901846863592763, 0.8171921746672638) ,
  rgb (0.7328454364552346, 0.7930974646884407, 0.8192926338423038) ,
  rgb (0.7387377170049494, 0.7959766573503101, 0.8214292278043301) ,
  rgb (0.7445924777189017, 0.7988204771958325, 0.8235986758615652) ,
  rgb (0.750403467654067, 0.8016269900896532, 0.8257973785108242) ,
  rgb (0.7561644358438198, 0.8043940873347794, 0.8280213899472) ,
  rgb (0.7618690793798029, 0.8071194938764749, 0.830266486168872) ,
  rgb (0.7675110850441786, 0.8098007598713145, 0.8325281663805967) ,
  rgb (0.7730841659017094, 0.8124352473546601, 0.8348017295057968) ,
  rgb (0.7785789200822759, 0.8150208937874255, 0.8370834463093898) ,
  rgb (0.7839910104276492, 0.8175542640053343, 0.8393674746403673) ,
  rgb (0.7893144556460892, 0.8200321318870201, 0.8416486380471222) ,
  rgb (0.7945430508923111, 0.8224511622630462, 0.8439218478682798) ,
  rgb (0.79967075421268, 0.8248078181208093, 0.8461821002957853) ,
  rgb (0.8046916442981458, 0.8270983878056066, 0.8484244929697402) ,
  rgb (0.8095999819094809, 0.8293189667350546, 0.8506444160105037) ,
  rgb (0.8143898212114309, 0.8314655869419785, 0.8528375906214702) ,
  rgb (0.8190465467793753, 0.8335364929949034, 0.855002062870101) ,
  rgb (0.8235742968025285, 0.8355248776479544, 0.8571319132851495) ,
  rgb (0.8279689431601354, 0.837426007513952, 0.8592239945130679) ,
  rgb (0.8322270571293441, 0.8392349062775448, 0.8612756350042788) ,
  rgb (0.8363403180919118, 0.8409479651895194, 0.8632852800107016) ,
  rgb (0.8403042080595778, 0.8425605950855084, 0.865250882410458) ,
  rgb (0.8441261828674842, 0.8440648271103739, 0.8671697332269007) ,
  rgb (0.8478071070257792, 0.8454546229209523, 0.8690403678369444) ,
  rgb (0.8513371457085719, 0.8467273579611647, 0.8708608165735044) ,
  rgb (0.8547259318925698, 0.8478748812467282, 0.8726282980930582) ,
  rgb (0.8579825924567037, 0.8488893481028184, 0.8743403855344628) ,
  rgb (0.8611024543689985, 0.8497675485700126, 0.8759924292343957) ,
  rgb (0.86408985081464, 0.8505039116750779, 0.8775792578489263) ,
  rgb (0.8669601550533358, 0.8510896085314068, 0.8790976697717334) ,
  rgb (0.8697047485350982, 0.8515240300479789, 0.8805388339000336) ,
  rgb (0.8723313408512408, 0.8518016547808089, 0.8818970435500162) ,
  rgb (0.8748534750857597, 0.8519152612302319, 0.8831692696761383) ,
  rgb (0.8772488085896548, 0.8518702833887027, 0.8843412038131143) ,
  rgb (0.8795410528270573, 0.8516567540749572, 0.8854143767924102) ,
  rgb (0.8817223105928579, 0.8512759407765347, 0.8863805692551482) ,
  rgb (0.8837852019553906, 0.8507294054031063, 0.8872322209694989) ,
  rgb (0.8857501584075443, 0.8500092494306783, 0.8879736506427196) ,
  rgb (0.8857115512284565, 0.8500218611585632, 0.8857253899008712) ,
  rgb (0.8855471481195238, 0.8498717428363158, 0.8833620612117095) ,
  rgb (0.8852589797263047, 0.8495589281098921, 0.8808841479402484) ,
  rgb (0.8848513815990857, 0.849084264228938, 0.8782823528537247) ,
  rgb (0.8843271305411354, 0.8484474157205542, 0.8755678522824297) ,
  rgb (0.8836912714589804, 0.8476489176151927, 0.8727414710144156) ,
  rgb (0.8829516859544853, 0.8466897027569927, 0.8697961704819097) ,
  rgb (0.8821154248940161, 0.8455700725455935, 0.8667376504623333) ,
  rgb (0.8811916987134195, 0.8442906671771735, 0.8635659516866988) ,
  rgb (0.8801929331569581, 0.8428522482477862, 0.8602739992715663) ,
  rgb (0.8791324424079277, 0.8412555488447591, 0.8568557229103964) ,
  rgb (0.8780229843664901, 0.8395016561854007, 0.8533064535245892) ,
  rgb (0.8768784845161469, 0.8375923807118654, 0.8496137354915025) ,
  rgb (0.875714587099614, 0.8355302318472394, 0.8457553751902708) ,
  rgb (0.8745379332026019, 0.8333197294864546, 0.8417192535806901) ,
  rgb (0.8733517136091627, 0.8309671525127262, 0.8374885100119709) ,
  rgb (0.8721533197845432, 0.8284805282370967, 0.8330486712880828) ,
  rgb (0.8709384671729751, 0.8258685788943851, 0.8283849726114961) ,
  rgb (0.8696913150261381, 0.8231415885956916, 0.8235047668317317) ,
  rgb (0.8683995469581863, 0.8203087551218152, 0.8184163896312899) ,
  rgb (0.867053149070485, 0.8173780404191124, 0.813134196269114) ,
  rgb (0.8656493432560532, 0.8143554406751491, 0.8076697232416455) ,
  rgb (0.8641834372394103, 0.8112464422465354, 0.8020461269686395) ,
  rgb (0.8626560105112757, 0.8080552380426153, 0.796282666437655) ,
  rgb (0.8610711702756552, 0.8047851790981223, 0.7903952966373629) ,
  rgb (0.8594346370300241, 0.8014392309950078, 0.7843978875138392) ,
  rgb (0.857756629435049, 0.7980196314271393, 0.778295716672475) ,
  rgb (0.856040223147254, 0.7945296360155061, 0.7721061003767414) ,
  rgb (0.8542921961147046, 0.7909719677709199, 0.765838014779141) ,
  rgb (0.8525190720770844, 0.7873493613354844, 0.7594994148789691) ,
  rgb (0.8507269702317879, 0.7836645734238389, 0.7530974636118285) ,
  rgb (0.8489224556311764, 0.7799202140765015, 0.7466371929366437) ,
  rgb (0.847111955079651, 0.7761188023604716, 0.7401227576280706) ,
  rgb (0.8452981073195511, 0.7722633860104472, 0.7335636824054149) ,
  rgb (0.843485292229337, 0.7683566039987018, 0.7269653699897204) ,
  rgb (0.8416775076684515, 0.7644010120098295, 0.7203329938728462) ,
  rgb (0.8398783988412087, 0.7603990719977968, 0.7136714781112923) ,
  rgb (0.8380912347613196, 0.7563531486080863, 0.7069856139021298) ,
  rgb (0.8363189884473793, 0.7522654895287526, 0.7002799902886496) ,
  rgb (0.8345656905566583, 0.7481379479992134, 0.6935569764986385) ,
  rgb (0.8328327718577798, 0.7439727181745988, 0.6868223587464855) ,
  rgb (0.8311216352909631, 0.7397718464763862, 0.6800812520363146) ,
  rgb (0.8294340781648147, 0.7355371221572935, 0.6733377200930191) ,
  rgb (0.8277716072353446, 0.7312702332407809, 0.6665957020468297) ,
  rgb (0.8261354971058026, 0.7269727551823826, 0.659859001562165) ,
  rgb (0.8245268129450285, 0.7226461431208888, 0.653131379154226) ,
  rgb (0.8229463511042843, 0.7182917733129006, 0.6464164243818421) ,
  rgb (0.8213947205565636, 0.7139109141951604, 0.6397176669767276) ,
  rgb (0.8198723065045529, 0.7095047497878748, 0.6330385704006711) ,
  rgb (0.8183793116449822, 0.705074381896351, 0.626382454789333) ,
  rgb (0.8169157577505589, 0.7006208301478398, 0.6197526063725792) ,
  rgb (0.8154814662727948, 0.6961450550830809, 0.6131522120932265) ,
  rgb (0.8140761104699334, 0.6916479479148213, 0.6065843782630862) ,
  rgb (0.8126992203988149, 0.6871303371461888, 0.600052148204351) ,
  rgb (0.8113501401176333, 0.6825930154624339, 0.5935584890905076) ,
  rgb (0.8100280946652069, 0.6780367267397182, 0.5871062690808275) ,
  rgb (0.8087321917008969, 0.6734621670219452, 0.5806983480557674) ,
  rgb (0.8074614045096935, 0.6688700095398864, 0.5743374637345958) ,
  rgb (0.8062146052692706, 0.6642608958528229, 0.5680262917864979) ,
  rgb (0.804990547908103, 0.6596354502756416, 0.5617674511054698) ,
  rgb (0.8037879253107763, 0.6549942654947263, 0.5555635086708381) ,
  rgb (0.8026053114611295, 0.6503379374810385, 0.5494169158460365) ,
  rgb (0.8014412429256005, 0.6456670345921877, 0.5433300863249918) ,
  rgb (0.8002941538975398, 0.6409821330674986, 0.5373053518514104) ,
  rgb (0.7991624518501963, 0.6362837937202919, 0.5313449594256143) ,
  rgb (0.7980444781513664, 0.6315725822508955, 0.5254510814483478) ,
  rgb (0.7969385471995161, 0.626849056792967, 0.5196258425240281) ,
  rgb (0.7958429237958377, 0.6221137880845115, 0.5138712409190979) ,
  rgb (0.7947558597265404, 0.617367344002207, 0.5081892121310299) ,
  rgb (0.7936755969866496, 0.6126102933407219, 0.5025816129126943) ,
  rgb (0.7926003430423745, 0.6078432208703702, 0.4970502062153201) ,
  rgb (0.7915283284347561, 0.603066705931472, 0.49159667021646397) ,
  rgb (0.7904577684772788, 0.5982813427706246, 0.48622257801969754) ,
  rgb (0.7893873776625194, 0.5934875421745599, 0.48092913815357724) ,
  rgb (0.7883151404562396, 0.5886860017356244, 0.4757179187907608) ,
  rgb (0.7872390410818835, 0.5838774374455721, 0.47059039582133383) ,
  rgb (0.7861573713233296, 0.5790624629815756, 0.465547782819184) ,
  rgb (0.7850684501960684, 0.5742417003617839, 0.46059116206904965) ,
  rgb (0.7839706083641448, 0.5694157832671042, 0.4557215474289206) ,
  rgb (0.7828622526444091, 0.5645853311116688, 0.45093985823706345) ,
  rgb (0.7817418047898184, 0.5597509805259486, 0.44624687186865436) ,
  rgb (0.7806077474948377, 0.5549133574489061, 0.4416433242636464) ,
  rgb (0.7794586273150664, 0.5500730841397727, 0.4371297985644476) ,
  rgb (0.7782934580763312, 0.545230594884266, 0.432706647838971) ,
  rgb (0.7771103295521099, 0.5403867491056124, 0.4283745037125848) ,
  rgb (0.775907907306857, 0.5355421788246119, 0.42413367909988375) ,
  rgb (0.774684947460632, 0.5306974938477673, 0.4199844035696376) ,
  rgb (0.7734402182988989, 0.5258533209345156, 0.41592679539764366) ,
  rgb (0.7721725722960555, 0.5210102658711383, 0.4119608998712287) ,
  rgb (0.7708809196230247, 0.516168926434691, 0.40808667584648967) ,
  rgb (0.7695642334401418, 0.5113298901696085, 0.4043039806968248) ,
  rgb (0.768221765997353, 0.5064936237128791, 0.40061257089416885) ,
  rgb (0.7668522895064389, 0.5016608471009063, 0.39701221751773474) ,
  rgb (0.7654549259333051, 0.4968321290972723, 0.3935025400011538) ,
  rgb (0.7640288560928866, 0.49200802533379656, 0.39008308392311997) ,
  rgb (0.7625733355405261, 0.48718906673415824, 0.38675335037837993) ,
  rgb (0.7610876378057071, 0.48237579130289127, 0.3835127572385229) ,
  rgb (0.7595711110534006, 0.4775687122205708, 0.380360657784311) ,
  rgb (0.7580232553845584, 0.4727682731566229, 0.3772963553109668) ,
  rgb (0.7564434157714071, 0.4679750143794846, 0.37431909037543515) ,
  rgb (0.7548310506695954, 0.46318942799460555, 0.3714280448394211) ,
  rgb (0.7531856636904657, 0.45841199172949604, 0.3686223664223477) ,
  rgb (0.7515068504589166, 0.45364314496866825, 0.36590112443835765) ,
  rgb (0.7497942054717047, 0.4488833348154881, 0.3632633755836028) ,
  rgb (0.7480473927555956, 0.44413297780351974, 0.36070813602540136) ,
  rgb (0.7462661578916344, 0.439392450449735, 0.3582343914230064) ,
  rgb (0.7444501867657067, 0.4346621718461711, 0.35584108091122535) ,
  rgb (0.7425992159973317, 0.42994254036133867, 0.3535270924537459) ,
  rgb (0.7407130161950609, 0.4252339389526239, 0.35129130890802607) ,
  rgb (0.7387914002459927, 0.4205367299231533, 0.34913260148542435) ,
  rgb (0.7368342217358587, 0.4158512585029011, 0.347049785207584) ,
  rgb (0.7348413359856494, 0.4111778700451951, 0.3450416947080907) ,
  rgb (0.7328127050626875, 0.4065168468778026, 0.3431071517341082) ,
  rgb (0.7307482075484517, 0.401868526884681, 0.3412449533046818) ,
  rgb (0.7286477385671655, 0.39723324476747235, 0.33945384341064017) ,
  rgb (0.7265112290022755, 0.3926113126792577, 0.3377325942005665) ,
  rgb (0.7243386564778159, 0.38800301593162145, 0.3360799596569183) ,
  rgb (0.7221299918421461, 0.3834086450896306, 0.33449469983585844) ,
  rgb (0.7198852149054985, 0.37882848839337313, 0.332975552002454) ,
  rgb (0.7176044490813385, 0.3742627271068619, 0.3315213862095893) ,
  rgb (0.7152876061484729, 0.3697117022522345, 0.3301308728723546) ,
  rgb (0.7129346683977347, 0.36517570519856757, 0.3288027427038317) ,
  rgb (0.7105456546582587, 0.36065500290840113, 0.3275357416278876) ,
  rgb (0.7081205956842048, 0.356149855233803, 0.32632861885644465) ,
  rgb (0.7056595112261009, 0.3516605297812094, 0.32518014084085567) ,
  rgb (0.7031624945881415, 0.34718723719598, 0.32408913679491225) ,
  rgb (0.700629624772421, 0.34273019305341756, 0.32305449047765694) ,
  rgb (0.6980608153581771, 0.3382897632604862, 0.3220747885521809) ,
  rgb (0.6954560834689112, 0.33386622163232865, 0.3211488430698579) ,
  rgb (0.6928154484676493, 0.32945984647042675, 0.3202754315314667) ,
  rgb (0.6901389321505248, 0.32507091815606004, 0.319453323328983) ,
  rgb (0.6874265643516962, 0.32069970535138104, 0.3186813762227769) ,
  rgb (0.6846785094249453, 0.3163463482122221, 0.31795870784057567) ,
  rgb (0.6818945715094452, 0.31201133280550686, 0.3172838048924495) ,
  rgb (0.6790747402815734, 0.30769497879760166, 0.31665545668946665) ,
  rgb (0.6762189792440975, 0.30339762792450425, 0.3160724937230589) ,
  rgb (0.6733272556481733, 0.29911962764489264, 0.3155337232398221) ,
  rgb (0.670399595476762, 0.29486126309253047, 0.3150381395687221) ,
  rgb (0.6674360376636913, 0.29062280081258873, 0.31458483752056837) ,
  rgb (0.6644363246987884, 0.2864048361425618, 0.31417223403606975) ,
  rgb (0.6614003753260178, 0.28220778870555907, 0.3137991292649849) ,
  rgb (0.6583280801134499, 0.2780321095766563, 0.3134643447952643) ,
  rgb (0.6552193260932713, 0.2738782665241015, 0.3131666792687211) ,
  rgb (0.652074172902773, 0.269746505252367, 0.3129056060581917) ,
  rgb (0.6488923016945825, 0.2656375533620908, 0.3126794181957019) ,
  rgb (0.6456734938264543, 0.2615520316161528, 0.31248673753935263) ,
  rgb (0.642417577481186, 0.257490519876798, 0.31232631707560987) ,
  rgb (0.6391243387840212, 0.2534536546198314, 0.3121968961206398) ,
  rgb (0.6357937104834237, 0.24944185818822678, 0.3120979395330059) ,
  rgb (0.6324253485421027, 0.24545598775548677, 0.3120276597462445) ,
  rgb (0.6290189201698587, 0.24149689191922535, 0.3119844719564572) ,
  rgb (0.6255741650043496, 0.23756535071152696, 0.3119669831491227) ,
  rgb (0.6220907982108261, 0.2336621873300741, 0.3119738327362739) ,
  rgb (0.6185686525887719, 0.2297879924917992, 0.3120046383872893) ,
  rgb (0.6150072323639137, 0.22594402043981826, 0.3120568068576574) ,
  rgb (0.6114062072731884, 0.22213124697023234, 0.3121288139643524) ,
  rgb (0.6077652399481865, 0.21835070166659282, 0.312219032918702) ,
  rgb (0.6040840169673274, 0.21460331490206347, 0.31232652641170694) ,
  rgb (0.6003621301041158, 0.21089030138947745, 0.3124493441041469) ,
  rgb (0.5965991810912237, 0.207212956082026, 0.3125852303112123) ,
  rgb (0.5927948053652026, 0.20357251410079796, 0.3127323483930494) ,
  rgb (0.5889486193554471, 0.19997020971775276, 0.31288922211590126) ,
  rgb (0.5850602439646688, 0.19640737049066315, 0.3130540116373273) ,
  rgb (0.5811293276158656, 0.19288548904692518, 0.3132239939418394) ,
  rgb (0.5771555081299204, 0.18940601489760422, 0.3133970433357208) ,
  rgb (0.5731384575410787, 0.18597036007065024, 0.3135712686852) ,
  rgb (0.5690778478401143, 0.18258004462335425, 0.3137444095679653) ,
  rgb (0.564973435290177, 0.17923664950367169, 0.3139136046337036) ,
  rgb (0.5608249903911717, 0.17594170887918095, 0.31407639883970623) ,
  rgb (0.5566322903496934, 0.17269677158182117, 0.31423043195101424) ,
  rgb (0.5523952157271191, 0.16950338809328983, 0.3143729155461537) ,
  rgb (0.5481137003346762, 0.1663630904279047, 0.3145010299091471) ,
  rgb (0.5437877131360856, 0.16327738551419116, 0.31461204226295625) ,
  rgb (0.5394173664919906, 0.16024769309971934, 0.31470295028655965) ,
  rgb (0.5350027976174474, 0.15727540775107324, 0.3147708520739653) ,
  rgb (0.5305442048985645, 0.15436183633886777, 0.3148129978918713) ,
  rgb (0.5260418962547748, 0.15150818660835483, 0.31482653406646727) ,
  rgb (0.5214965286322996, 0.14871544765633712, 0.3148076795453443) ,
  rgb (0.5169084880054446, 0.14598463068714407, 0.3147540759228004) ,
  rgb (0.5122783510532176, 0.14331656120063752, 0.3146630922831542) ,
  rgb (0.5076068118105369, 0.14071192654913128, 0.3145320012008257) ,
  rgb (0.5028952497497061, 0.13817086581280427, 0.3143566215383367) ,
  rgb (0.4981443546207415, 0.13569380302451714, 0.314135190862664) ,
  rgb (0.49335504375145617, 0.13328091630401023, 0.31386561956734976) ,
  rgb (0.48852847371852987, 0.13093210934893723, 0.31354553695453014) ,
  rgb (0.48366628618847957, 0.1286467902201389, 0.31317188565991266) ,
  rgb (0.47877034239726296, 0.12642401401409453, 0.3127417273582196) ,
  rgb (0.473841437035254, 0.12426354237989065, 0.31225470169927194) ,
  rgb (0.46888111384598413, 0.12216445576414045, 0.31170911458932665) ,
  rgb (0.46389102840284874, 0.12012561256850712, 0.31110343446582983) ,
  rgb (0.45887288947308297, 0.11814571137706886, 0.3104363697903881) ,
  rgb (0.4538300508699989, 0.11622183788331528, 0.3097044124984492) ,
  rgb (0.4487629917317482, 0.1143535557462255, 0.30890905921943196) ,
  rgb (0.44367358645071275, 0.11253912421257944, 0.3080497309546545) ,
  rgb (0.4385637818793154, 0.11077667828375456, 0.30712600062348083) ,
  rgb (0.4334355841041439, 0.1090642347484701, 0.3061376792828915) ,
  rgb (0.4282910131578975, 0.1073997763055258, 0.30508479060294547) ,
  rgb (0.42313214269556043, 0.10578120994917611, 0.3039675809469457) ,
  rgb (0.41796105205173684, 0.10420644885760968, 0.3027865203963184) ,
  rgb (0.41277985630360303, 0.1026734006932461, 0.3015422643746897) ,
  rgb (0.4075912039268871, 0.10117945586419633, 0.300235195077286) ,
  rgb (0.40239692379737496, 0.09972293031495055, 0.2988667436973397) ,
  rgb (0.39719876876325577, 0.09830232096827862, 0.2974385647628578) ,
  rgb (0.39199887556812907, 0.09691583650296684, 0.2959521200550908) ,
  rgb (0.3867993907954417, 0.09556181960083443, 0.29440901248173756) ,
  rgb (0.38160247377467543, 0.09423873126371218, 0.2928110750626949) ,
  rgb (0.3764103053221462, 0.09294519809377791, 0.2911602415731392) ,
  rgb (0.3712250843130934, 0.09167999748026273, 0.2894586539763317) ,
  rgb (0.36605031412464006, 0.0904406854276979, 0.2877077458811747) ,
  rgb (0.36088752387578377, 0.0892271943627452, 0.28591050458531014) ,
  rgb (0.35573889947341125, 0.08803897435024335, 0.2840695897279818) ,
  rgb (0.3506067824603248, 0.08687555176033529, 0.28218770540182386) ,
  rgb (0.345493557138718, 0.08573665496512634, 0.28026769921081435) ,
  rgb (0.34040164359597463, 0.08462223619170267, 0.27831254595259397) ,
  rgb (0.3353335322445545, 0.08353243411900396, 0.2763253435679004) ,
  rgb (0.3302917447118144, 0.08246763389003825, 0.27430929404579435) ,
  rgb (0.3252788886040126, 0.08142839007654609, 0.27226772884656186) ,
  rgb (0.3202998655799406, 0.08041299473755484, 0.2702032289303951) ,
  rgb (0.3153587000920581, 0.07942099731524319, 0.2681190407694196) ,
  rgb (0.31045520848595526, 0.07845687167618218, 0.2660200572779356) ,
  rgb (0.30559226007249934, 0.07752196310753731, 0.2639100669211966) ,
  rgb (0.3007727681291869, 0.07661782433616476, 0.2617929409781967) ,
  rgb (0.2960004726065818, 0.07574533600095842, 0.25967245030364566) ,
  rgb (0.29128515387578635, 0.0748990498474667, 0.25755101595750435) ,
  rgb (0.28662309235899847, 0.07408846082680887, 0.2554347867371703) ,
  rgb (0.2820174629736694, 0.07331569321404097, 0.25332800295084507) ,
  rgb (0.277471508091428, 0.07258296989925478, 0.2512349399594277) ,
  rgb (0.272997015188973, 0.07188355544616351, 0.2491584709323293) ,
  rgb (0.2685909594817286, 0.07122671627792246, 0.24710443563450618) ,
  rgb (0.2642551220706094, 0.07061659562299544, 0.24507758869355967) ,
  rgb (0.25999463887892144, 0.07005385560386188, 0.24308218808684579) ,
  rgb (0.2558213554748177, 0.06953231029187984, 0.24112190491594204) ,
  rgb (0.25172899728289466, 0.0690646308260355, 0.23920260612763083) ,
  rgb (0.247720929905011, 0.06865333790948652, 0.2373288009471749) ,
  rgb (0.2438114972024792, 0.06828985152901187, 0.23550427698321885) ,
  rgb (0.23999586188690308, 0.06798502996477995, 0.23373434258507808) ,
  rgb (0.23627495835774248, 0.06774359820987802, 0.23202360805926608) ,
  rgb (0.23266299920501882, 0.06755710438847978, 0.23037617493752832) ,
  rgb (0.2291562027859284, 0.06743473087115257, 0.22879681433956656) ,
  rgb (0.2257539681670791, 0.06738213230014747, 0.22728984778098055) ,
  rgb (0.22247308588973624, 0.06738821405309284, 0.22585960379408354) ,
  rgb (0.21930503925136402, 0.06746578636294004, 0.22451023616807558) ,
  rgb (0.21625279838647882, 0.06761633027051639, 0.22324568672294431) ,
  rgb (0.2133313859647627, 0.06783014842602667, 0.2220704321302429) ,
  rgb (0.21052882914958676, 0.06812195249816172, 0.22098759107715404) ,
  rgb (0.20785704662965598, 0.06848439879702528, 0.22000133917653536) ,
  rgb (0.20531725273301316, 0.06891959226639757, 0.21911516689288835) ,
  rgb (0.20290365333558247, 0.06943524858045896, 0.21833167885096033) ,
  rgb (0.20058760685133747, 0.07006457614998421, 0.21762721310371608) ,
  rgb (0.19819343656336558, 0.07097462525273879, 0.21690975060032436) ,
  rgb (0.19571853588267552, 0.07215778103960274, 0.21617499187076789) ,
  rgb (0.1931548636579131, 0.07360681904011795, 0.21542362939081539) ,
  rgb (0.19049578401722037, 0.07531127841678764, 0.2146562337112265) ,
  rgb (0.18774482037046955, 0.07725158846803931, 0.21387448578597812) ,
  rgb (0.18488035509396164, 0.07942573027972388, 0.21307651648984993)
});


list_data viridis = list_data(new pen[] {
  rgb (0.267004, 0.004874, 0.329415) ,
  rgb (0.26851, 0.009605, 0.335427) ,
  rgb (0.269944, 0.014625, 0.341379) ,
  rgb (0.271305, 0.019942, 0.347269) ,
  rgb (0.272594, 0.025563, 0.353093) ,
  rgb (0.273809, 0.031497, 0.358853) ,
  rgb (0.274952, 0.037752, 0.364543) ,
  rgb (0.276022, 0.044167, 0.370164) ,
  rgb (0.277018, 0.050344, 0.375715) ,
  rgb (0.277941, 0.056324, 0.381191) ,
  rgb (0.278791, 0.062145, 0.386592) ,
  rgb (0.279566, 0.067836, 0.391917) ,
  rgb (0.280267, 0.073417, 0.397163) ,
  rgb (0.280894, 0.078907, 0.402329) ,
  rgb (0.281446, 0.08432, 0.407414) ,
  rgb (0.281924, 0.089666, 0.412415) ,
  rgb (0.282327, 0.094955, 0.417331) ,
  rgb (0.282656, 0.100196, 0.42216) ,
  rgb (0.28291, 0.105393, 0.426902) ,
  rgb (0.283091, 0.110553, 0.431554) ,
  rgb (0.283197, 0.11568, 0.436115) ,
  rgb (0.283229, 0.120777, 0.440584) ,
  rgb (0.283187, 0.125848, 0.44496) ,
  rgb (0.283072, 0.130895, 0.449241) ,
  rgb (0.282884, 0.13592, 0.453427) ,
  rgb (0.282623, 0.140926, 0.457517) ,
  rgb (0.28229, 0.145912, 0.46151) ,
  rgb (0.281887, 0.150881, 0.465405) ,
  rgb (0.281412, 0.155834, 0.469201) ,
  rgb (0.280868, 0.160771, 0.472899) ,
  rgb (0.280255, 0.165693, 0.476498) ,
  rgb (0.279574, 0.170599, 0.479997) ,
  rgb (0.278826, 0.17549, 0.483397) ,
  rgb (0.278012, 0.180367, 0.486697) ,
  rgb (0.277134, 0.185228, 0.489898) ,
  rgb (0.276194, 0.190074, 0.493001) ,
  rgb (0.275191, 0.194905, 0.496005) ,
  rgb (0.274128, 0.199721, 0.498911) ,
  rgb (0.273006, 0.20452, 0.501721) ,
  rgb (0.271828, 0.209303, 0.504434) ,
  rgb (0.270595, 0.214069, 0.507052) ,
  rgb (0.269308, 0.218818, 0.509577) ,
  rgb (0.267968, 0.223549, 0.512008) ,
  rgb (0.26658, 0.228262, 0.514349) ,
  rgb (0.265145, 0.232956, 0.516599) ,
  rgb (0.263663, 0.237631, 0.518762) ,
  rgb (0.262138, 0.242286, 0.520837) ,
  rgb (0.260571, 0.246922, 0.522828) ,
  rgb (0.258965, 0.251537, 0.524736) ,
  rgb (0.257322, 0.25613, 0.526563) ,
  rgb (0.255645, 0.260703, 0.528312) ,
  rgb (0.253935, 0.265254, 0.529983) ,
  rgb (0.252194, 0.269783, 0.531579) ,
  rgb (0.250425, 0.27429, 0.533103) ,
  rgb (0.248629, 0.278775, 0.534556) ,
  rgb (0.246811, 0.283237, 0.535941) ,
  rgb (0.244972, 0.287675, 0.53726) ,
  rgb (0.243113, 0.292092, 0.538516) ,
  rgb (0.241237, 0.296485, 0.539709) ,
  rgb (0.239346, 0.300855, 0.540844) ,
  rgb (0.237441, 0.305202, 0.541921) ,
  rgb (0.235526, 0.309527, 0.542944) ,
  rgb (0.233603, 0.313828, 0.543914) ,
  rgb (0.231674, 0.318106, 0.544834) ,
  rgb (0.229739, 0.322361, 0.545706) ,
  rgb (0.227802, 0.326594, 0.546532) ,
  rgb (0.225863, 0.330805, 0.547314) ,
  rgb (0.223925, 0.334994, 0.548053) ,
  rgb (0.221989, 0.339161, 0.548752) ,
  rgb (0.220057, 0.343307, 0.549413) ,
  rgb (0.21813, 0.347432, 0.550038) ,
  rgb (0.21621, 0.351535, 0.550627) ,
  rgb (0.214298, 0.355619, 0.551184) ,
  rgb (0.212395, 0.359683, 0.55171) ,
  rgb (0.210503, 0.363727, 0.552206) ,
  rgb (0.208623, 0.367752, 0.552675) ,
  rgb (0.206756, 0.371758, 0.553117) ,
  rgb (0.204903, 0.375746, 0.553533) ,
  rgb (0.203063, 0.379716, 0.553925) ,
  rgb (0.201239, 0.38367, 0.554294) ,
  rgb (0.19943, 0.387607, 0.554642) ,
  rgb (0.197636, 0.391528, 0.554969) ,
  rgb (0.19586, 0.395433, 0.555276) ,
  rgb (0.1941, 0.399323, 0.555565) ,
  rgb (0.192357, 0.403199, 0.555836) ,
  rgb (0.190631, 0.407061, 0.556089) ,
  rgb (0.188923, 0.41091, 0.556326) ,
  rgb (0.187231, 0.414746, 0.556547) ,
  rgb (0.185556, 0.41857, 0.556753) ,
  rgb (0.183898, 0.422383, 0.556944) ,
  rgb (0.182256, 0.426184, 0.55712) ,
  rgb (0.180629, 0.429975, 0.557282) ,
  rgb (0.179019, 0.433756, 0.55743) ,
  rgb (0.177423, 0.437527, 0.557565) ,
  rgb (0.175841, 0.44129, 0.557685) ,
  rgb (0.174274, 0.445044, 0.557792) ,
  rgb (0.172719, 0.448791, 0.557885) ,
  rgb (0.171176, 0.45253, 0.557965) ,
  rgb (0.169646, 0.456262, 0.55803) ,
  rgb (0.168126, 0.459988, 0.558082) ,
  rgb (0.166617, 0.463708, 0.558119) ,
  rgb (0.165117, 0.467423, 0.558141) ,
  rgb (0.163625, 0.471133, 0.558148) ,
  rgb (0.162142, 0.474838, 0.55814) ,
  rgb (0.160665, 0.47854, 0.558115) ,
  rgb (0.159194, 0.482237, 0.558073) ,
  rgb (0.157729, 0.485932, 0.558013) ,
  rgb (0.15627, 0.489624, 0.557936) ,
  rgb (0.154815, 0.493313, 0.55784) ,
  rgb (0.153364, 0.497, 0.557724) ,
  rgb (0.151918, 0.500685, 0.557587) ,
  rgb (0.150476, 0.504369, 0.55743) ,
  rgb (0.149039, 0.508051, 0.55725) ,
  rgb (0.147607, 0.511733, 0.557049) ,
  rgb (0.14618, 0.515413, 0.556823) ,
  rgb (0.144759, 0.519093, 0.556572) ,
  rgb (0.143343, 0.522773, 0.556295) ,
  rgb (0.141935, 0.526453, 0.555991) ,
  rgb (0.140536, 0.530132, 0.555659) ,
  rgb (0.139147, 0.533812, 0.555298) ,
  rgb (0.13777, 0.537492, 0.554906) ,
  rgb (0.136408, 0.541173, 0.554483) ,
  rgb (0.135066, 0.544853, 0.554029) ,
  rgb (0.133743, 0.548535, 0.553541) ,
  rgb (0.132444, 0.552216, 0.553018) ,
  rgb (0.131172, 0.555899, 0.552459) ,
  rgb (0.129933, 0.559582, 0.551864) ,
  rgb (0.128729, 0.563265, 0.551229) ,
  rgb (0.127568, 0.566949, 0.550556) ,
  rgb (0.126453, 0.570633, 0.549841) ,
  rgb (0.125394, 0.574318, 0.549086) ,
  rgb (0.124395, 0.578002, 0.548287) ,
  rgb (0.123463, 0.581687, 0.547445) ,
  rgb (0.122606, 0.585371, 0.546557) ,
  rgb (0.121831, 0.589055, 0.545623) ,
  rgb (0.121148, 0.592739, 0.544641) ,
  rgb (0.120565, 0.596422, 0.543611) ,
  rgb (0.120092, 0.600104, 0.54253) ,
  rgb (0.119738, 0.603785, 0.5414) ,
  rgb (0.119512, 0.607464, 0.540218) ,
  rgb (0.119423, 0.611141, 0.538982) ,
  rgb (0.119483, 0.614817, 0.537692) ,
  rgb (0.119699, 0.61849, 0.536347) ,
  rgb (0.120081, 0.622161, 0.534946) ,
  rgb (0.120638, 0.625828, 0.533488) ,
  rgb (0.12138, 0.629492, 0.531973) ,
  rgb (0.122312, 0.633153, 0.530398) ,
  rgb (0.123444, 0.636809, 0.528763) ,
  rgb (0.12478, 0.640461, 0.527068) ,
  rgb (0.126326, 0.644107, 0.525311) ,
  rgb (0.128087, 0.647749, 0.523491) ,
  rgb (0.130067, 0.651384, 0.521608) ,
  rgb (0.132268, 0.655014, 0.519661) ,
  rgb (0.134692, 0.658636, 0.517649) ,
  rgb (0.137339, 0.662252, 0.515571) ,
  rgb (0.14021, 0.665859, 0.513427) ,
  rgb (0.143303, 0.669459, 0.511215) ,
  rgb (0.146616, 0.67305, 0.508936) ,
  rgb (0.150148, 0.676631, 0.506589) ,
  rgb (0.153894, 0.680203, 0.504172) ,
  rgb (0.157851, 0.683765, 0.501686) ,
  rgb (0.162016, 0.687316, 0.499129) ,
  rgb (0.166383, 0.690856, 0.496502) ,
  rgb (0.170948, 0.694384, 0.493803) ,
  rgb (0.175707, 0.6979, 0.491033) ,
  rgb (0.180653, 0.701402, 0.488189) ,
  rgb (0.185783, 0.704891, 0.485273) ,
  rgb (0.19109, 0.708366, 0.482284) ,
  rgb (0.196571, 0.711827, 0.479221) ,
  rgb (0.202219, 0.715272, 0.476084) ,
  rgb (0.20803, 0.718701, 0.472873) ,
  rgb (0.214, 0.722114, 0.469588) ,
  rgb (0.220124, 0.725509, 0.466226) ,
  rgb (0.226397, 0.728888, 0.462789) ,
  rgb (0.232815, 0.732247, 0.459277) ,
  rgb (0.239374, 0.735588, 0.455688) ,
  rgb (0.24607, 0.73891, 0.452024) ,
  rgb (0.252899, 0.742211, 0.448284) ,
  rgb (0.259857, 0.745492, 0.444467) ,
  rgb (0.266941, 0.748751, 0.440573) ,
  rgb (0.274149, 0.751988, 0.436601) ,
  rgb (0.281477, 0.755203, 0.432552) ,
  rgb (0.288921, 0.758394, 0.428426) ,
  rgb (0.296479, 0.761561, 0.424223) ,
  rgb (0.304148, 0.764704, 0.419943) ,
  rgb (0.311925, 0.767822, 0.415586) ,
  rgb (0.319809, 0.770914, 0.411152) ,
  rgb (0.327796, 0.77398, 0.40664) ,
  rgb (0.335885, 0.777018, 0.402049) ,
  rgb (0.344074, 0.780029, 0.397381) ,
  rgb (0.35236, 0.783011, 0.392636) ,
  rgb (0.360741, 0.785964, 0.387814) ,
  rgb (0.369214, 0.788888, 0.382914) ,
  rgb (0.377779, 0.791781, 0.377939) ,
  rgb (0.386433, 0.794644, 0.372886) ,
  rgb (0.395174, 0.797475, 0.367757) ,
  rgb (0.404001, 0.800275, 0.362552) ,
  rgb (0.412913, 0.803041, 0.357269) ,
  rgb (0.421908, 0.805774, 0.35191) ,
  rgb (0.430983, 0.808473, 0.346476) ,
  rgb (0.440137, 0.811138, 0.340967) ,
  rgb (0.449368, 0.813768, 0.335384) ,
  rgb (0.458674, 0.816363, 0.329727) ,
  rgb (0.468053, 0.818921, 0.323998) ,
  rgb (0.477504, 0.821444, 0.318195) ,
  rgb (0.487026, 0.823929, 0.312321) ,
  rgb (0.496615, 0.826376, 0.306377) ,
  rgb (0.506271, 0.828786, 0.300362) ,
  rgb (0.515992, 0.831158, 0.294279) ,
  rgb (0.525776, 0.833491, 0.288127) ,
  rgb (0.535621, 0.835785, 0.281908) ,
  rgb (0.545524, 0.838039, 0.275626) ,
  rgb (0.555484, 0.840254, 0.269281) ,
  rgb (0.565498, 0.84243, 0.262877) ,
  rgb (0.575563, 0.844566, 0.256415) ,
  rgb (0.585678, 0.846661, 0.249897) ,
  rgb (0.595839, 0.848717, 0.243329) ,
  rgb (0.606045, 0.850733, 0.236712) ,
  rgb (0.616293, 0.852709, 0.230052) ,
  rgb (0.626579, 0.854645, 0.223353) ,
  rgb (0.636902, 0.856542, 0.21662) ,
  rgb (0.647257, 0.8584, 0.209861) ,
  rgb (0.657642, 0.860219, 0.203082) ,
  rgb (0.668054, 0.861999, 0.196293) ,
  rgb (0.678489, 0.863742, 0.189503) ,
  rgb (0.688944, 0.865448, 0.182725) ,
  rgb (0.699415, 0.867117, 0.175971) ,
  rgb (0.709898, 0.868751, 0.169257) ,
  rgb (0.720391, 0.87035, 0.162603) ,
  rgb (0.730889, 0.871916, 0.156029) ,
  rgb (0.741388, 0.873449, 0.149561) ,
  rgb (0.751884, 0.874951, 0.143228) ,
  rgb (0.762373, 0.876424, 0.137064) ,
  rgb (0.772852, 0.877868, 0.131109) ,
  rgb (0.783315, 0.879285, 0.125405) ,
  rgb (0.79376, 0.880678, 0.120005) ,
  rgb (0.804182, 0.882046, 0.114965) ,
  rgb (0.814576, 0.883393, 0.110347) ,
  rgb (0.82494, 0.88472, 0.106217) ,
  rgb (0.83527, 0.886029, 0.102646) ,
  rgb (0.845561, 0.887322, 0.099702) ,
  rgb (0.85581, 0.888601, 0.097452) ,
  rgb (0.866013, 0.889868, 0.095953) ,
  rgb (0.876168, 0.891125, 0.09525) ,
  rgb (0.886271, 0.892374, 0.095374) ,
  rgb (0.89632, 0.893616, 0.096335) ,
  rgb (0.906311, 0.894855, 0.098125) ,
  rgb (0.916242, 0.896091, 0.100717) ,
  rgb (0.926106, 0.89733, 0.104071) ,
  rgb (0.935904, 0.89857, 0.108131) ,
  rgb (0.945636, 0.899815, 0.112838) ,
  rgb (0.9553, 0.901065, 0.118128) ,
  rgb (0.964894, 0.902323, 0.123941) ,
  rgb (0.974417, 0.90359, 0.130215) ,
  rgb (0.983868, 0.904867, 0.136897) ,
  rgb (0.993248, 0.906157, 0.143936)
});


