// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

// AspectC++ includes
#include "Transformer.h"
#include "JPAdvice.h"
#include "PointCutEvaluator.h"
#include "PointCut.h"
#include "PointCutContext.h"
#include "OrderInfo.h"
#include "AdviceInfo.h"
#include "AspectInfo.h"
#include "IntroductionInfo.h"
#include "JoinPointPlan.h"
#include "Plan.h"
#include "Repository.h"
#include "PointCutContext.h"
#include "CFlow.h"
#include "BackEndProblems.h"
#include "ACConfig.h"
#include "Introducer.h"
#include "IncludeGraph.h"
#include "PointCutExpr.h"
#include "ModelBuilder.h"
#include "IntroductionUnit.h"
#include "version.h"
#ifdef ACMODEL
#include "XmlModelWriter.h"
#include "XmlModelReader.h"
#endif

// PUMA includes
#include "Puma/CProject.h"
#include "Puma/CCSemVisitor.h"
#include "Puma/VerboseMgr.h"
#include "Puma/CFunctionInfo.h"
#include "Puma/CUnionInfo.h"
#include "Puma/CArgumentInfo.h"
#include "Puma/CTranslationUnit.h"
#include "Puma/CSemDatabase.h"
#include "Puma/ACAspectInfo.h"
#include "Puma/ACAdviceInfo.h"
#include "Puma/ACIntroductionInfo.h"
#include "Puma/ACPointcutInfo.h"
#include "Puma/ACTree.h"
#include "Puma/CPrintVisitor.h"
#include "Puma/SysCall.h"
#ifdef PROFILING
#include "Puma/GenericProfiler.ah"
#endif

// C++ includes
#include <sstream>
using std::stringstream;
using std::endl;
#include <stdlib.h> // for getenv
#ifdef _MSC_VER
#include <io.h>
#else
#include <unistd.h> // for access()!
#endif // _MSC_VER
#include <fcntl.h>

void Transformer::work (Unit *unit, Token *primary_start, Token *primary_end) {

  // set the start and end token of the unit
  _code_weaver.init (primary_start, primary_end);
  
  // determine back-end compiler problems and setup code weaver
  BackEndProblems back_end_problems;
  back_end_problems._local_class       = _conf.problem_local_class ();
  back_end_problems._spec_scope        = _conf.problem_spec_scope ();
  back_end_problems._use_always_inline = !_conf.problem_force_inline ();
  back_end_problems._size_type         = _conf.size_type ();
  back_end_problems._warn_macro        = _conf.warn_macro();
  back_end_problems._warn_deprecated   = _conf.warn_deprecated();
  _code_weaver.problems (back_end_problems);
  
  // perform the transformation
  CTranslationUnit *tunit1 = 0;
  ModelBuilder jpm (_vm, _err, _conf);
  
  bool ok = (phase1 (unit, tunit1, jpm) &&
             phase2 (unit, tunit1, jpm));
  if (tunit1)
    delete tunit1;
  if (!ok)
    _vm << "Aborting" << endvm;
}

bool Transformer::phase1 (Unit *unit, CTranslationUnit *&tunit,
                          ModelBuilder &jpm) {

#ifdef PROFILING
  static Profiler::Data data (__PRETTY_FUNCTION__, NULL);
  Profiler::ClockTicks start = Profiler::rdtsc ();
#endif

  _vm << "Inserting namespace AC" << endvm;
  // in front of start!
  _code_weaver.insert_namespace_ac_before ((Token*)unit->first ());

  if (!_conf.iterate () && !_conf.ifiles () && _conf.file_in ()) {
    // STU mode and not generating transformed headers!
    string forced_includes = "";
    list<const ConfOption*> forced_include_opts;
    for (unsigned int i = 0; i < _conf.project().config ().Options (); i++) {
      const ConfOption *o = _conf.project().config().Option (i);
      if (! strcmp (o->Name (), "--include")) {
        if (o->Arguments () != 1) continue;
        const char *filename = o->Argument (0);
        forced_includes += "#include ";
        if (filename[0] == '\"' || filename[0] == '<')
          forced_includes += filename;
        else {
          forced_includes += "\"";
          forced_includes += filename;
          forced_includes += "\"";
        }
        forced_includes += "\n";
        forced_include_opts.push_back (o); // collect for removed
      }
    }
    if (forced_includes != "") {
      _vm << "Inserting forced includes" << endvm;
      _code_weaver.paste (_code_weaver.weave_pos ((Token*)unit->first (), WeavePos::WP_BEFORE),
        forced_includes);
      for (list<const ConfOption*>::iterator i = forced_include_opts.begin ();
        i != forced_include_opts.end (); ++i)
        _conf.project().config().Remove(*i);
    }
  }
  _code_weaver.commit ();
  
  _vm << "Parsing ..." << endvm;
  // parse the translation unit, but ignore function bodies
  unsigned options = _parser.Options ();
  _parser.Options (options | CCParser::SKIP_FCT_BODY);
  tunit = _parser.parse (*unit, _project);
  _parser.Options (options);
  // do semantic analysis of expressions
  _sem_visitor.run (tunit->tree ());
  
  if (_err.severity () >= sev_error)
    return false;

//  tunit->unit ()->print(cout);
//  CPrintVisitor printer;
//  printer.print (tunit->tree (), cout);

  //    Some debugging code:
  //    tunit->unit ()->print(cout);
  //    CPrintVisitor printer;
  //    printer.print (tunit->tree (), cout);
  //    tunit->db ().Dump (cout, 2);
  
  // create the phase 1 join point model
  _vm << "Setting up join point model 1 ..." << endvm;
  _vm++;
  int tunit_len = _code_weaver.primary_end ()->location().line() -
      _code_weaver.primary_start()->location().line() + 1;
  jpm.setup_phase1 (*tunit, tunit_len);
  _vm--;
  
  // set up the project repository
  _repo.setup (jpm.tunit_file());

  _vm << "Setting Aspect Access Privileges ..." << endvm;
  aspect_privileges (jpm);
  
  _vm << "Weaving Introductions ..." << endvm;
  introductions (*tunit, jpm);
      
  _vm << "Weaving Advice Declarations ..." << endvm;
  advice (*tunit);
  
  _vm << "Weaving Singleton Aspects ..." << endvm;
  singleton_aspects (*tunit);

//  _vm << "Slice/Intro Includes ..." << endvm;
//  _code_weaver.slice_includes (_project, _primary_start);

  _vm << "Committing" << endvm;
  _code_weaver.commit ();

//    _vm << "Stage1 save!" << endvm;
//    _project.save();
//    _vm << " done." << endvm;
//	exit(1);

#ifdef PROFILING
  Profiler::ClockTicks end = Profiler::rdtsc ();
  data._time += Profiler::duration (start, end);
  data._calls = 1;
#endif

  return (_err.severity () < sev_error);
}

bool Transformer::phase2 (Unit *unit, CTranslationUnit *tunit1,
                          ModelBuilder &jpm) {

#ifdef PROFILING
  static Profiler::Data data (__PRETTY_FUNCTION__, NULL);
  Profiler::ClockTicks start = Profiler::rdtsc ();
#endif

  IncludeGraph ig (_project);
  ig.init (*tunit1);
//  ig.dump ();
  
  _vm << "Preparing introductions ..." << endvm;
  Plan plan (_err, jpm);
  PointCutContext context (jpm);
  PointCutEvaluator eval (_err, context);

  // get all aspects from the plan (in some order)
  const Plan::AspectContainer &aspects = plan.aspect_infos ();
  for (Plan::AspectContainer::const_iterator i = aspects.begin ();
    i != aspects.end (); ++i) {
    
#ifdef ACMODEL
    ACM_Aspect &jpl_aspect = i->loc ();
    context.concrete_aspect (jpl_aspect);

    // add introductions to plan
    list<ACM_Introduction*> intros;
    collect_intros (jpl_aspect, intros);
    
    for (list<ACM_Introduction*>::iterator i = intros.begin ();
      i != intros.end (); ++i) {
      IntroductionInfo *ii = plan.addIntroduction (jpl_aspect, **i);
      ii->pointcut_expr () =
        eval.create (TI_PointcutExpr::of(*(*i)->get_expr ())->tree(), JPT_Name);
      if (!(ii->pointcut_expr()->possible_types() & JPT_Class)) {
        _err  << sev_warning << TI_Introduction::of(ii->intro ())->loc()
             << "pointcut expression for introduction can't match class"
             << endMessage;
      }
    }

    // add order advices to plan
    list<ACM_Order*> orders;
    collect_orders (jpl_aspect, orders);
    
    for (list<ACM_Order*>::iterator i = orders.begin ();
      i != orders.end (); ++i) {
      OrderInfo *oi = plan.addOrder (jpl_aspect, **i);
      oi->analyze_exprs (eval);
      if (!getenv ("ACNEWREPO")) {
        _repo.update (*oi);
      }
    }
#else
    JPL_Aspect &jpl_aspect = i->loc ();
    context.concrete_aspect (jpl_aspect);

    // add introductions to plan
    list<JPL_Introduction*> intros;
    jpl_aspect.collect_intros (intros);
    
    for (list<JPL_Introduction*>::iterator i = intros.begin ();
      i != intros.end (); ++i) {
      IntroductionInfo *ii = plan.addIntroduction (jpl_aspect, **i);
      ii->pointcut_expr() = eval.create ((*i)->expr (), JoinPointLoc::Name);
      if (!(ii->pointcut_expr()->possible_types() & JoinPointLoc::Class)) {
        _err  << sev_warning << TI_Introduction::of(ii->intro ())->loc()
             << "pointcut expression for introduction can't match class"
             << endMessage;
      }
    }

    // add order advices to plan
    list<JPL_Order*> orders;
    jpl_aspect.collect_orders (orders);
    
    for (list<JPL_Order*>::iterator i = orders.begin ();
      i != orders.end (); ++i) {
      OrderInfo *oi = plan.addOrder (jpl_aspect, **i);
      oi->analyze_exprs (eval);
      if (!getenv ("ACNEWREPO")) {
        _repo.update (*oi);
      }
    }
#endif
  }

  _vm << "Parsing again ..." << endvm;
  
  // parse the translation unit, but ignore function bodies
  Introducer introducer (plan, _code_weaver, _parser, jpm, ig, _conf);
  _parser.introducer (&introducer);
  CTranslationUnit *tunit = _parser.parse (*unit, _project);
  _parser.introducer (0);

  // do semantic analysis of expressions
  _sem_visitor.run (tunit->tree ());
  list<CTree*> &ah_trees = introducer.ah_trees ();
  for (list<CTree*>::iterator i = ah_trees.begin (); i != ah_trees.end (); ++i)
    if (*i)
      _sem_visitor.run (*i);

  if (_err.severity () >= sev_error) {
    // TODO: delete takes too much time and has no real use for ac++
    // so we skip it for now 
    // delete tunit;
    return false;
  }

//#ifdef TRY_INTRODUCER
//  cout << "Printing semantic database..." << endl;
//  tunit->db ().Dump (cout, 10);
//#endif // TRY_INTRODUCER
  
  //    CPrintVisitor printer;
  //    printer.print (tunit->tree (), cout);
  
  _vm << "Updating repository intro part" << endvm;
  if (!getenv ("ACNEWREPO")) {
    update_intros_in_repo (jpm);
  }
  
  jpm.setup_phase2 (*tunit, ah_trees);
  
  _vm << "Weaving access control bypass classes ..." << endvm;
  insert_bypass_class (jpm);

  _vm << "Weaving Join Points ..." << endvm;
  Plan plan2 (_err, jpm);
  join_points (*tunit, jpm, plan2);
  if (_err.severity () >= sev_error)
    return false;
      
  _vm << "Final cleanup" << endvm;
  cleanup (*tunit);

  if (_conf.dynamic ()) {
    _vm << "Preparing for dynamic weaving" << endvm;
    prepare_dynamic_weaving (jpm);  
  }

  // generate a string with aspect header include directives
  determine_aspect_includes (ig);
  
  _vm << "Committing" << endvm;
  _code_weaver.commit ();

  // TODO: delete takes too much time and has no real use for ac++
  // so we skip it for now 
  // delete tunit;
  
  _repo.cleanup ();

#ifdef PROFILING
  Profiler::ClockTicks end = Profiler::rdtsc ();
  data._time += Profiler::duration (start, end);
  data._calls = 1;
#endif

  return (_err.severity () < sev_error);
}

void Transformer::determine_aspect_includes (const IncludeGraph &ig) {
  // find all files that are included by aspect headers
  if (_conf.iterate_aspects ()) {
    // collect the names of aspect header files,
    // generate a unit with include statements for these files,
    PathIterator ah_iter (".*\\.ah$");
    while (_project.iterate (ah_iter))
      _aspect_includes += aspect_include_cluster (ah_iter.file (), ig);
  }
  else {
    // Get the names from the configuration object (-a options)
    for (int i = 0; i < _conf.aspect_headers (); i++)
      _aspect_includes += aspect_include_cluster (_conf.aspect_header (i), ig);
  }
}

string Transformer::aspect_include_cluster (const char* ah_file,
  const IncludeGraph &ig) {

  // find the corresponding unit object for the aspect header file name
  const Unit *unit = _project.unitManager ().getUnit (ah_file, true);
  if (!unit || !unit->isFile ()) {
    _err << sev_fatal << "Unit for \"" << ah_file << "\" not found or no file."
         << endMessage;
    return "";
  }

  // determine the aspect header cluster for this aspect header unit
  set<const Unit*> cluster_units;
  determine_aspect_cluster (unit, ig, cluster_units);
  
  // generate the code
  stringstream cluster;
  cluster << "#ifdef __ac_need_";
  Naming::mangle_file (cluster, (FileUnit*)unit);
  cluster << endl;

  for (set<const Unit*>::iterator i = cluster_units.begin ();
    i != cluster_units.end (); ++i) {
    const Unit *aspect_unit = *i;  
    Filename incname = _project.getInclString (aspect_unit->name ());
    cluster << "#ifndef __ac_have_";
    Naming::mangle_file (cluster, (FileUnit*)aspect_unit);
    cluster << endl;
    cluster << "#define __ac_have_";
    Naming::mangle_file (cluster, (FileUnit*)aspect_unit);
    cluster << endl;
    cluster << "#include \"" << incname << "\"" << endl;
    cluster << "#endif" << endl;
  }  
  
  cluster << "#endif" << endl;
  return cluster.str ();
}

void Transformer::determine_aspect_cluster (const Unit* ah_unit,
  const IncludeGraph &ig, set<const Unit*> &cluster) {
  
  // if the ah file is already a cluster member, we return immediately
  if (cluster.find (ah_unit) != cluster.end ())
    return;

  // otherwise the unit will be inserted
  cluster.insert (ah_unit);
      
  // find all header files that are included by this aspect header
  set<const Unit*> inc_units;
  ig.included_files (ah_unit, inc_units);
  
  // include all aspect headers that affect join points in these headers
  AspectIncludes &ais = _code_weaver.aspect_includes ();
  for (set<const Unit*>::iterator i = inc_units.begin ();
    i != inc_units.end (); ++i) {
    Unit *inc_unit = (Unit*)*i;
    AspectIncludes::const_iterator aii = ais.find (inc_unit);
    if (aii != ais.end ()) {
      const set<AspectRef> &aspect_refs = aii->second;
      for (set<AspectRef>::const_iterator ari = aspect_refs.begin ();
        ari != aspect_refs.end (); ++ari) {
#ifdef ACMODEL
        ACM_Aspect &jpl_aspect = ari->_aspect->loc ();
#else
        JPL_Aspect &jpl_aspect = ari->_aspect->loc ();
#endif
        Unit *aspect_unit = TI_Aspect::of (jpl_aspect)->unit ();
        // recursively analyze the cluster of this aspect header unit
        determine_aspect_cluster (aspect_unit, ig, cluster);
      }
    }
  }
}

void Transformer::prepare_dynamic_weaving (ModelBuilder &jpm) {

  // mark all operations that access introduced attributes
  const list<AccessInfo> &access_infos = jpm.access_infos ();
  for (list<AccessInfo>::const_iterator i = access_infos.begin ();
    i != access_infos.end (); ++i) {
    if (i->_info->isStatic () || i->_info->isAnonymous ())
      continue;
    Unit *unit = (Unit*)i->_info->Tree ()->token ()->belonging_to ();
    while (unit->isMacroExp ())
      unit = ((MacroUnit*)unit)->CallingUnit ();
    if (IntroductionUnit::cast (unit)) {
      Token *tok_before = i->_tree->token ();
      const WeavePos &before = _code_weaver.weave_pos (tok_before, WeavePos::WP_BEFORE);
      _code_weaver.paste (before, string ("/*** +access ") +
        string (i->_info->QualName ()) + string (" ***/\n"));      
      Token *tok_after = i->_tree->end_token ();
      const WeavePos &after = _code_weaver.weave_pos (tok_after, WeavePos::WP_AFTER);
      _code_weaver.paste (after, string ("/*** -access ") +
        string (i->_info->QualName ()) + string (" ***/\n"));      
      if (i->_tree->NodeName () == CT_MembPtrExpr::NodeId () ||
          i->_tree->NodeName () == CT_MembRefExpr::NodeId ()) {
        CTree *op = i->_tree->Son (1);
        Token *tok_before = op->token ();
        const WeavePos &before = _code_weaver.weave_pos (tok_before, WeavePos::WP_BEFORE);
        _code_weaver.paste (before, string ("/*** op ") +
          string (i->_info->QualName ()) + string (" ***/\n"));      
      }
    }
  }
  
  // mark all classes that contain dynamically introduced attributes
#ifdef ACMODEL
  ProjectModel::Selection classes;
  jpm.select ((JoinPointType)(JPT_Class|JPT_Aspect), classes);
  for (ProjectModel::Selection::iterator i = classes.begin ();
    i != classes.end (); ++i) {
    ACM_Class *cls = (ACM_Class*)*i;
    if (!cls->get_intro_target ())
#else
      JoinPointModel::Selection classes;
      jpm.select ((JoinPointLoc::join_point_type)(JoinPointLoc::Class|
        JoinPointLoc::Aspect), classes);
      for (JoinPointModel::Selection::iterator i = classes.begin ();
        i != classes.end (); ++i) {
        JPL_Class *cls = (JPL_Class*)*i;
        if (!cls->intro_target ())
#endif
      continue;
    bool mark_class = false;
    const TI_Class *ti = TI_Class::of (*cls);
    CClassInfo *ci = ti->class_info ();
    for (unsigned i = 0; i < ci->Attributes (); i++) {
      CAttributeInfo *attr = ci->Attribute (i);
      if (attr->isStatic () || attr->isAnonymous ())
        continue;
      Unit *unit = (Unit*)attr->Tree ()->token ()->belonging_to ();
      IntroductionUnit *iunit = IntroductionUnit::cast (unit);
      if (iunit) {
        mark_class = true;
#ifdef ACMODEL
        ACM_Name *jpl_aspect = (ACM_Name*)iunit->intro ()->get_parent ();
#else
        JPL_Name *jpl_aspect = iunit->intro ()->parent ();
#endif
        Token *tok_before = attr->Tree ()->ObjDecl ()->token ();
        const WeavePos &before = _code_weaver.weave_pos (tok_before, WeavePos::WP_BEFORE);
        _code_weaver.paste (before, string ("/*** +intro ") +
#ifdef ACMODEL
        signature (*jpl_aspect) + string (" ***/\n"));      
#else
        string (jpl_aspect->signature ()) + string (" ***/\n"));      
#endif
        Token *tok_after = attr->Tree ()->ObjDecl ()->end_token ();
        const WeavePos &after = _code_weaver.weave_pos (tok_after, WeavePos::WP_AFTER);
        _code_weaver.paste (after, string ("/*** -intro ") +
#ifdef ACMODEL
        signature (*jpl_aspect) + string (" ***/\n"));      
#else
        string (jpl_aspect->signature ()) + string (" ***/\n"));      
#endif
      }
    }
    if (mark_class) {
      Token *tok_before = ((CT_ClassDef*)ci->Tree ())->ObjDecl ()->token ();
      const WeavePos &before = _code_weaver.weave_pos (tok_before, WeavePos::WP_BEFORE);
      _code_weaver.paste (before, string ("/*** +class ") +
#ifdef ACMODEL
      signature (*cls) + string (" ***/\n"));      
#else
      string (cls->signature ()) + string (" ***/\n"));      
#endif
      
      Token *tok_after = ((CT_ClassDef*)ci->Tree ())->Members ()->end_token ();
      const WeavePos &after = _code_weaver.weave_pos (tok_after, WeavePos::WP_BEFORE);
      _code_weaver.paste (after, string ("/*** -class ") +
#ifdef ACMODEL
      signature (*cls) + string (" ***/\n"));      
#else
      string (cls->signature ()) + string (" ***/\n"));
#endif
    }
  }
}


void Transformer::update_intros_in_repo (ModelBuilder &jpm1) {
#ifdef ACMODEL
  ProjectModel::Selection intros;
  jpm1.select (JPT_Introduction, intros);
  for (ProjectModel::Selection::iterator i = intros.begin ();
       i != intros.end (); ++i) {
    ACM_Introduction &intro = *(ACM_Introduction*)*i;
    _repo.consider (intro);
  }
  ProjectModel::Selection classes;
  jpm1.select ((JoinPointType)(JPT_Class|JPT_Aspect), classes);
  for (ProjectModel::Selection::iterator i = classes.begin ();
       i != classes.end (); ++i) {
    ACM_Class &cls = *(ACM_Class*)*i;
    // get the weaving plan of this intro
    JPP_Class *plan = (JPP_Class*)cls.plan ();
    // collect the intros into this class
    if (plan) {
      set<ACM_Introduction*> intros;
      for (int b = 0; b < plan->baseIntros (); b++)
        intros.insert (plan->baseIntro (b));
      for (int o = 0; o < plan->otherIntros (); o++)
        intros.insert (plan->otherIntro (o));
      // now update the repository with the intro information
      for (set<ACM_Introduction*>::iterator intro_iter = intros.begin ();
           intro_iter != intros.end (); ++intro_iter) {
        _repo.update (**intro_iter, cls);
      }
    }
  }
#else
  ModelBuilder::Selection intros;
  jpm1.select (JoinPointLoc::Introduction, intros);
  for (JoinPointModel::Selection::iterator i = intros.begin ();
       i != intros.end (); ++i) {
    JPL_Introduction &intro = *(JPL_Introduction*)*i;
    _repo.consider (intro);
  }
  ModelBuilder::Selection classes;
  jpm1.select ((JoinPointLoc::join_point_type)(JoinPointLoc::Class|
    JoinPointLoc::Aspect), classes);
  for (JoinPointModel::Selection::iterator i = classes.begin ();
       i != classes.end (); ++i) {
    JPL_Class &cls = *(JPL_Class*)*i;
    // get the weaving plan of this intro
    JPP_Class *plan = (JPP_Class*)cls.plan ();
    // collect the intros into this class
    if (plan) {
      set<JPL_Introduction*> intros;
      for (int b = 0; b < plan->baseIntros (); b++)
        intros.insert (plan->baseIntro (b));
      for (int o = 0; o < plan->otherIntros (); o++)
        intros.insert (plan->otherIntro (o));
      // now update the repository with the intro information
      for (set<JPL_Introduction*>::iterator intro_iter = intros.begin ();
           intro_iter != intros.end (); ++intro_iter) {
        _repo.update (**intro_iter, cls);
      }
    }
  }
#endif
}


static bool needs_friend_injection (ModelBuilder &jpm, CRecord *rec) {
  static Filename anon("<anonymous unit>");
  CSemDatabase *semdb = jpm.get_db();

  // no need for template instances; the templates already get the friend injection
  if (rec->isTemplateInstance())
    return false;
  // only defined classes can get an injection
  if (!rec->isDefined ())
    return false;
  // code introduced by ac++ (in an anonymous unit) is not modified here
  if (rec->Tree()->token ()->location().filename() == anon)
    return false;
  // nested classes in template instances should also not be modified to avoid double injection
  if (jpm.inside_template_instance(rec))
    return false;
  // the class has to belong to the project
  Unit *unit = (Unit*)rec->Tree ()->token ()->belonging_to ();
  if (!semdb->Project ()->isBelow (unit))
    return false;
  return true;
}


void Transformer::aspect_privileges (ModelBuilder &jpm) {
  // TODO: move this into phase 2 parse; introduce classes don't
  //       get the friend injection yet!

  // get all aspects from the join point model
#ifdef ACMODEL
  ProjectModel::Selection all_aspects;
  jpm.select (JPT_Aspect, all_aspects);

  // remember that these aspects should become friend of all classes
  list<CClassInfo*> friends;
  for (ProjectModel::Selection::iterator iter = all_aspects.begin ();
       iter != all_aspects.end (); ++iter) {
    ACM_Aspect &jpl = (ACM_Aspect&)**iter;
    friends.push_back (((TI_Aspect*)jpl.transform_info ())->ClassInfo());
  }
#else
  JoinPointModel::Selection all_aspects;
  jpm.select (JoinPointLoc::Aspect, all_aspects);

  // remember that these aspects should become friend of all classes
  list<CClassInfo*> friends;
  for (JoinPointModel::Selection::iterator iter = all_aspects.begin ();
       iter != all_aspects.end (); ++iter) {
    JPL_Aspect &jpl = (JPL_Aspect&)**iter;
    friends.push_back (((TI_Aspect*)jpl.transform_info ())->ClassInfo());
  }
#endif
  
  if (!friends.empty ()) {
    CSemDatabase *semdb = jpm.get_db();
    // handle struct and classes
    for (unsigned i = 0; i < semdb->ClassInfos(); i++) {
      CClassInfo *ci = semdb->ClassInfo(i);

      if (needs_friend_injection (jpm, ci)) {
        // perform the code transformation
        _code_weaver.declare_friends (ci, friends);
      }
    }
    // handle unions
    for (unsigned i = 0; i < semdb->UnionInfos(); i++) {
      CUnionInfo *ui = semdb->UnionInfo(i);

      if (needs_friend_injection (jpm, ui)) {
        // perform the code transformation
        _code_weaver.declare_friends (ui, friends);
      }
    }
  }
}
      

void Transformer::cleanup (CTranslationUnit &tunit) {
  CSemDatabase &db = tunit.db ();

  // replace keyword "aspect" with "class" and remove advice
  for (int a = 0; a < db.AspectInfos (); a++) {
    ACAspectInfo *acai = db.AspectInfo (a);

    // remove advice declarations
    for (int adv = 0; adv < acai->AdviceNodes (); adv++)
      _code_weaver.kill (acai->AdviceNode (adv));

    _code_weaver.to_class (acai);
  }
      
  // delete all pointcut definitions that are left
  for (int p = 0; p < db.PointcutInfos (); p++)
    _code_weaver.kill (db.PointcutInfo (p)->def_node ());
}


void Transformer::introductions (CTranslationUnit &tunit,
  ModelBuilder &jpm) {

  // remove the introductions from the code
#ifdef ACMODEL
  ProjectModel::Selection intros;
  jpm.select (JPT_Introduction, intros);
  for (ProjectModel::Selection::iterator i = intros.begin ();
       i != intros.end (); ++i) {
    ACM_Introduction &intro = *(ACM_Introduction*)*i;
    _code_weaver.kill (TI_Introduction::of (intro)->tree ());
  }
#else
  ModelBuilder::Selection intros;
  jpm.select (JoinPointLoc::Introduction, intros);
  for (JoinPointModel::Selection::iterator i = intros.begin ();
       i != intros.end (); ++i) {
    JPL_Introduction &intro = *(JPL_Introduction*)*i;
    _code_weaver.kill (TI_Introduction::of (intro)->tree ());
  }
#endif
  
  // now delete all slice definitions
  CSemDatabase &db = tunit.db ();
  for (int s = 0; s < db.SliceInfos (); s++) {
    ACSliceInfo *acsi = db.SliceInfo (s);
    // first all members
    for (int sm = 0; sm < acsi->members (); sm++) {
      CT_Intro *member = acsi->member (sm);
      _code_weaver.kill (member);
    }
    // now the main definition
    if (!acsi->in_advice ())
      _code_weaver.kill (acsi->def_node ());
  }
}

void Transformer::orderings (CTranslationUnit &tunit, Plan& plan,
                             ModelBuilder &jpm, bool purge) {

  // let the plan object calculate the right order
  plan.order ();

  // remove the introductions from the code
  if (purge) {
    // Iterate through order advice declarations
#ifdef ACMODEL
    ProjectModel::Selection orders;
    jpm.select (JPT_Order, orders);
    for (ProjectModel::Selection::iterator i = orders.begin ();
       i != orders.end (); ++i) {
      ACM_Order &order = *(ACM_Order*)*i;
#else
      ModelBuilder::Selection orders;
      jpm.select (JoinPointLoc::Order, orders);
      for (JoinPointModel::Selection::iterator i = orders.begin ();
         i != orders.end (); ++i) {
        JPL_Order &order = *(JPL_Order*)*i;
#endif
      TI_Order  &ti   = *TI_Order::of (order);
      _code_weaver.kill (ti.tree ());
    }
  }
}


void Transformer::advice (CTranslationUnit &tunit)
 {
   CSemDatabase &db = tunit.db ();
   for (int a = 0; a < db.AspectInfos (); a++)
    {
      ACAspectInfo *ai = db.AspectInfo (a);

      _vm++;
      // handle all advice of this aspect
      for (int i = 0; i < ai->AdviceNodes (); i++) {
	CT_AdviceDecl *ad = ai->AdviceNode (i);
	CFunctionInfo *advice_func = ((CT_FctDef*)ad->Decl ())->Object ()->
	  FunctionInfo ();
	
	// handle only real advice here
	if (strncmp (advice_func->Name (), "%a", 2) != 0)
	  continue;
	
	// don't handle inherited advice here. This is done in the base class
	if (advice_func->BaseObject ())
	  continue;
	
	_vm << ai->ClassInfo ()->QualName () << "::" 
	    << advice_func->Name () << endvm;
	
	_code_weaver.declare_function (advice_func, ad); // first phase
      }
      _vm--;
    }

   return;
 }


void Transformer::singleton_aspects (CTranslationUnit &tunit) {
  CSemDatabase &db = tunit.db ();
  for (int a = 0; a < db.AspectInfos (); a++) {
    ACAspectInfo *ai = db.AspectInfo (a);

    if (ai->is_abstract ())
      continue;

    _code_weaver.singleton (ai);
  }
}

void Transformer::insert_bypass_class (ModelBuilder &jpm) {
  set<Token*> blacklist; // see comments below
  // Iterate through classes and structs, what about unions?
#ifdef ACMODEL
  ProjectModel::Selection all_classes;
  jpm.select ((JoinPointType)(JPT_Class|JPT_Aspect), all_classes);
  for (ProjectModel::Selection::iterator i = all_classes.begin ();
       i != all_classes.end (); ++i) {
    ACM_Class *cls = (ACM_Class*)*i;
#else
  ModelBuilder::Selection all_classes;
  jpm.select ((JoinPointLoc::join_point_type)(JoinPointLoc::Class|
      JoinPointLoc::Aspect), all_classes);
  for (JoinPointModel::Selection::iterator i = all_classes.begin ();
       i != all_classes.end (); ++i) {
    JPL_Class *cls = (JPL_Class*)*i;
#endif
    CClassInfo *ci = TI_Class::of (*cls)->class_info();
    Token *token = ci->DefObject ()->Tree ()->token();
    if (blacklist.find (token) == blacklist.end ()) {
      // handle all classes that are not on the blacklist
      _code_weaver.insert_bypass_class (ci);

      // put a class in the backlist if is defined inside a template instance
      // -> it can appear more than once in the model!
      if (jpm.inside_template_instance (ci)) {
        blacklist.insert (token);
      }
    }
  }
}


void Transformer::join_points (CTranslationUnit &tunit,
                                   ModelBuilder &jpm, Plan &plan) {

#ifdef PROFILING
  static Profiler::Data data (__PRETTY_FUNCTION__, NULL);
  Profiler::ClockTicks start = Profiler::rdtsc ();
#endif
  _vm++;

  _vm << "Advicecode manipulation" << endvm;
  // Iterate through advice
#ifdef ACMODEL
  ProjectModel::Selection advice_codes;
  jpm.select (JPT_AdviceCode, advice_codes);
  for (ProjectModel::Selection::iterator i = advice_codes.begin ();
       i != advice_codes.end (); ++i) {
    ACM_AdviceCode &code = *(ACM_AdviceCode*)*i;
#else
    ModelBuilder::Selection advice_codes;
    jpm.select (JoinPointLoc::AdviceCode, advice_codes);
    for (JoinPointModel::Selection::iterator i = advice_codes.begin ();
         i != advice_codes.end (); ++i) {
      JPL_AdviceCode &code = *(JPL_AdviceCode*)*i;
#endif
    TI_AdviceCode  &ti   = *TI_AdviceCode::of (code);
    // setup ThisJoinPoint object of this advice code
    ti.this_join_point ().setup (ti.function ());
    _code_weaver.provide_tjp (ti.function (), ti.this_join_point ());
  }

  _vm << "Collecting Advice" << endvm;
  // Iterate through advice
  _vm++;

  // Create a pointcut evaluator
  PointCutContext context (jpm);
  PointCutEvaluator eva (_err, context);

  // Create a data structure for collecting advice per join point type
  typedef list<AdviceInfo*> AdviceInfoList;
#ifdef ACMODEL
  typedef JoinPointType JPT;
#else
  typedef JoinPointLoc::join_point_type JPT;
#endif
  typedef map<JPT, AdviceInfoList> TypeAdviceMap;
  TypeAdviceMap advice_map;

  Plan::AspectContainer &aspects = plan.aspect_infos ();
  for (Plan::AspectContainer::iterator i = aspects.begin ();
    i != aspects.end (); ++i) {
    AspectInfo &aspect_info = (AspectInfo&)*i;
#ifdef ACMODEL
    ACM_Aspect &jpl_aspect = aspect_info.loc ();
#else
    JPL_Aspect &jpl_aspect = aspect_info.loc ();
#endif
    context.concrete_aspect (jpl_aspect);

    // setup thisJoinPoint for aspectOf function, if there is one
    CFunctionInfo *aspect_of_func = TI_Aspect::of (jpl_aspect)->aspectof();
    if (aspect_of_func) {
      _vm << "Setting up thisJoinPoint for aspectof" << endvm;
      aspect_info.aspectof_this_join_point ().setup (aspect_of_func);
      _vm << "Supplying aspectof() with JoinPoint and tjp if needed" << endvm;
      _code_weaver.provide_tjp (aspect_of_func,
        aspect_info.aspectof_this_join_point ());
    }

    // handle the advice for the current aspect
    list<AdviceInfo*> &advices = aspect_info.advice_infos ();
    for (list<AdviceInfo*>::const_iterator i = advices.begin ();
      i != advices.end (); ++i) {
      AdviceInfo *advice_info = *i;
#ifdef ACMODEL
      ACM_AdviceCode *code = &advice_info->code ();
#else
      JPL_AdviceCode *code = &advice_info->code ();
#endif
      CFunctionInfo *adfunc = TI_AdviceCode::of (*code)->function ();

#ifdef ACMODEL
      _vm << signature (jpl_aspect) << ": "  << adfunc->Name () << endvm;
#else
      _vm << jpl_aspect.signature () << ": "  << adfunc->Name () << endvm;
#endif
      _vm++;

      // let the pointcut evaluator create the pointcut expression object
      _vm << "Create pointcut expression tree" << endvm;
      context.func (adfunc);
      advice_info->pointcut_expr () =
#ifdef ACMODEL
      eva.create (TI_PointcutExpr::of(*code->get_expr ())->tree(), JPT_Code);
#else
      eva.create (code->expr (), JoinPointLoc::Code);
#endif

      if (advice_info->pointcut_expr()) {
        // set the pointcut type before destroy the expression(!)
        advice_info->pointcut().type (advice_info->pointcut_expr()->type() == PCE_CODE ? PointCut::PCT_CODE : PointCut::PCT_CLASS);

        // remember the advice for each joinpoint type that might match
        int mask = 1;
        while (mask) {
          if (mask &
#ifdef ACMODEL
              JPT_Code
#else
              JoinPointLoc::Code
#endif
              ) {
            pair<TypeAdviceMap::iterator, bool> result =
              advice_map.insert (TypeAdviceMap::value_type ((TypeAdviceMap::key_type)mask, AdviceInfoList()));
            result.first->second.push_back (advice_info);
          }
          mask <<= 1;
        }
      }

      // copy the cflow trigger pointcut from the expressions to the pointcut
      for (set<PointCutExpr*>::const_iterator iter = context.cflows ().begin ();
           iter != context.cflows ().end (); ++iter) {
        advice_info->pointcut().cflow_triggers(((PCE_CFlow*)*iter)->arg_pointcut ());
      }
      _vm--;
    }
    context.cflow_reset ();
  }
  _vm--;

  // now iterate over all join points and check whether they match the
  // pointcut expressions
  _vm << "Matching joinpoints" << endvm;
  _vm++;
  for (TypeAdviceMap::iterator mi = advice_map.begin (); mi != advice_map.end ();
      ++mi) {
    JPT jp_type = mi->first;
    AdviceInfoList &advice_info_list = mi->second;
#ifdef ACMODEL
    ProjectModel::Selection all;
    jpm.select (jp_type, all);
    for (ProjectModel::Selection::iterator iter = all.begin ();
        iter != all.end (); ++iter) {
      ACM_Any &jpl = (ACM_Any&)**iter;
#else
    JoinPointModel::Selection all;
    jpm.select (jp_type, all);
    for (JoinPointModel::Selection::iterator iter = all.begin ();
        iter != all.end (); ++iter) {
      JoinPointLoc &jpl = **iter;
#endif
      for (AdviceInfoList::iterator li = advice_info_list.begin ();
          li != advice_info_list.end (); ++li) {
        AdviceInfo *advice_info = *li;
        context.pseudo_true (false);

        // now match
        Binding binding;
        Condition condition;
        PointCutExpr *expr = advice_info->pointcut_expr();
        if (expr->match (jpl, context, binding, condition)) {
          JoinPoint &jp = *new JoinPoint (&jpl, condition);
          advice_info->pointcut().append (jp);

          // consider this joinpoint in the big plan
          plan.consider (jp.location (), jp.condition (), advice_info);

          // remember units for inclusion of aspect headers
          _code_weaver.add_aspect_include (jp.location (), advice_info->aspect_info(),
                                           AspectRef::AR_ADVICE);

    #ifdef ACMODEL
          if (jpl.type_val () == JPT_Call && !jpl.get_parent()) // TODO: pseud
    #else
          if (jpl.type () == JoinPointLoc::MethodCall &&
              ((JPL_MethodCall&)jpl).is_pseudo ())
    #endif
            continue;
          // check if the context variable binding is the same for all
          // non-pseudo join points
          if (advice_info->binding () != binding) {
            if (!advice_info->binding ()._used) {
              advice_info->binding () = binding;
            }
            else {
              _err << sev_error;
              if (expr->node () && expr->node ()->token ())
                _err << expr->node ()->token ()->location ();
              _err << "incompatible argument bindings in pointcut expression"
                   << endMessage;
              // remove this erroneous advice from all lists
              for (TypeAdviceMap::iterator i = advice_map.begin (); i != advice_map.end ();
                  ++i) {
                i->second.remove (advice_info);
              }
              break;
            }
          }
        }
      }
    }
  }

  // again iterate over all aspects
  for (Plan::AspectContainer::iterator i = aspects.begin ();
    i != aspects.end (); ++i) {
    AspectInfo &aspect_info = (AspectInfo&)*i;
#ifdef ACMODEL
    ACM_Aspect &jpl_aspect = aspect_info.loc ();
#else
    JPL_Aspect &jpl_aspect = aspect_info.loc ();
#endif
    context.concrete_aspect (jpl_aspect);
    int index = 0; // CFlow index (each CFlow has a unique index per aspect)

    // and now over all advice
    list<AdviceInfo*> advices = aspect_info.advice_infos ();
    for (list<AdviceInfo*>::const_iterator i = advices.begin ();
      i != advices.end (); ++i) {
      AdviceInfo *advice_info = *i;
      PointCut &pc = advice_info->pointcut();
      // consider the cflow trigger needed for this advice in the plan
      const list<PointCut*> &trigger_pcs = pc.cflow_triggers();
      for (list<PointCut*>::const_iterator iter = trigger_pcs.begin ();
           iter != trigger_pcs.end (); ++iter, ++index) {
        PointCut *trigger_pc = *iter;
        // Consider a cflow trigger for every joinpoint is pointcut
        for (PointCut::iterator iter = trigger_pc->begin ();
             iter != trigger_pc->end (); ++iter) {
          const JoinPoint &jp = *iter;

          // consider this joinpoint in the big plan
          plan.consider (jp.location (), CFlow (advice_info, index));

          // remember units for inclusion of aspect headers
          _code_weaver.add_aspect_include (jp.location (), aspect_info,
                                           AspectRef::AR_DECL);

        }
      }

      // update the project repository
      if (getenv ("ACNEWREPO") == 0)
        _repo.update (*advice_info, pc);
    }

    // add order advices to plan
#ifdef ACMODEL
    list<ACM_Order*> orders;
    collect_orders (jpl_aspect, orders);
    for (list<ACM_Order*>::iterator i = orders.begin ();
#else
    list<JPL_Order*> orders;
    jpl_aspect.collect_orders (orders);
    for (list<JPL_Order*>::iterator i = orders.begin ();
#endif

      i != orders.end (); ++i) {
      OrderInfo *oi = plan.addOrder (jpl_aspect, **i);
      oi->analyze_exprs (eva);
    }

    ACAspectInfo *acai = TI_Aspect::of (jpl_aspect)->aspect_info();
    _code_weaver.invocation_functions (acai,
      aspect_info.ifct_decls (_code_weaver.problems ()),
      aspect_info.ifct_defs (_code_weaver.problems ()));
  }
  _vm--;

  _vm << "Aspect ordering ..." << endvm;
  orderings(tunit, plan, jpm, true);

  // now do the final checks on the accumulated plan
  _vm << "Final checks before weaving code join points" << endvm;
  plan.check ();

  // Don't weave if there were errors in the planning phase
  if (_err.severity () >= sev_error)
    return;

  char *repo_file = getenv ("ACNEWREPO");
  if (repo_file) {
#ifdef _MSC_VER
    if (!_access (repo_file, 04)) {
#else
    if (!access (repo_file, R_OK)) {
#endif // _MSC_VER

      _vm << "Opening new project repository '" << repo_file << "'" << endvm;
      int fd = SysCall::open_excl (repo_file, O_RDWR, &_err);
#ifdef ACMODEL
      ProjectModel project_model;
      XmlModelReader reader;
      if (!reader.read (project_model, repo_file, fd)) {
        _err << sev_error << "project repository '" << repo_file << "' cannot be opened"
        " or is invalid" << endMessage;
        return;
      }
      if (project_model.get_version () != ac_version ())
        _err << sev_warning << "project file version differs from ac++ version"
            << endMessage;

      // merge jpm and project_mode here
      project_model.merge (jpm);

      // make project model file empty
      lseek (fd, 0, SEEK_SET);
      if (ftruncate (fd, 0) != 0)
        perror ("truncate");

      // save the merged model
      XmlModelWriter writer;
      if (!writer.write (project_model, repo_file, fd)) {
      _err << sev_error << "saving merged project file '" << repo_file <<
        "'failed" << endMessage;
      return;
      }
#else
      JoinPointModel ljpm;
      JoinPointModel::FileOpResult res = ljpm.load (fd, repo_file);
      if (res == JoinPointModel::JPM_LOAD_ERR) {
        _err << sev_error << "project repository '" << repo_file << "' cannot be opened"
        " or is invalid" << endMessage;
        return;
      }
      if (res == JoinPointModel::JPM_VERSION) {
        _err << sev_warning << "project file version differs from ac++ version"
            << endMessage;
      }
      ljpm.reconcile (jpm);
      lseek (fd, 0, SEEK_SET);
      if (ftruncate (fd, 0) != 0)
        perror ("truncate");
      ljpm.save (fd, repo_file);
#endif
      SysCall::close_excl (fd, &_err);
    }
    else {
      int fd = SysCall::create_excl (repo_file, 0600, &_err);
#ifdef ACMODEL
      jpm.set_version(ac_version ());
      XmlModelWriter writer;
      writer.write (jpm, repo_file, fd);
#else
      jpm.save (fd, repo_file);
#endif
      SysCall::close_excl  (fd, &_err);
    }
  }

  _vm << "Type Check Functions" << endvm;
  _vm++;
  const TypeCheckSet &checks_false = plan.type_checks_false ();
  for (TypeCheckSet::const_iterator iter = checks_false.begin ();
       iter != checks_false.end (); ++iter) {
    _vm << "check for " << iter->second << " in "
        << iter->first->QualName () << " is false" << endvm;
    _code_weaver.type_check (iter->first, iter->second, false);
  }
  const TypeCheckSet &checks_true = plan.type_checks_true ();
  for (TypeCheckSet::const_iterator iter = checks_true.begin ();
       iter != checks_true.end (); ++iter) {
    _vm << "check for " << iter->second << " in "
        << iter->first->QualName () << " is true" << endvm;
    _code_weaver.type_check (iter->first, iter->second, true);
  }
  _vm--;

  _vm << "Call Join Points" << endvm;
  _vm++;
  for (int i = 0; i < plan.call_jp_plans (); i++) {
    JPP_Code &jp_plan = plan.call_jp_plan (i);
#ifdef ACMODEL
    ACM_Call &jp_loc = plan.call_jp_loc (i);
    _vm << signature (jp_loc) << endvm;
#else
    JPL_MethodCall &jp_loc = plan.call_jp_loc (i);
    _vm << jp_loc.signature () << endvm;
#endif

    // handle call joinpoint itself
    _code_weaver.call_join_point (&jp_loc, jp_plan);
  }
  _vm--;

  _vm << "Execution Join Points" << endvm;
  _vm++;
  for (int i = 0; i < plan.exec_jp_plans (); i++) {
    JPP_Code &jp_plan = plan.exec_jp_plan (i);
#ifdef ACMODEL
    ACM_Execution &jp_loc = plan.exec_jp_loc (i);
    _vm << signature (jp_loc) << endvm;
#else
    JPL_Method &jp_loc = plan.exec_jp_loc (i);
    _vm << jp_loc.signature () << endvm;
#endif

    // handle exec joinpoint itself
    _code_weaver.exec_join_point (&jp_loc, jp_plan);
  }
  _vm--;

  _vm << "Construction Join Points" << endvm;
  _vm++;
  for (int i = 0; i < plan.cons_jp_plans (); i++) {
    JPP_Code &jp_plan = plan.cons_jp_plan (i);
#ifdef ACMODEL
    ACM_Construction &jp_loc = plan.cons_jp_loc (i);
    _vm << signature (jp_loc) << endvm;
#else
    JPL_Construction &jp_loc = plan.cons_jp_loc (i);
    _vm << jp_loc.signature () << endvm;
#endif

    // handle construction joinpoint itself
    _code_weaver.cons_join_point (&jp_loc, jp_plan);
  }
  _vm--;

  _vm << "Destruction Join Points" << endvm;
  _vm++;
  for (int i = 0; i < plan.dest_jp_plans (); i++) {
    JPP_Code &jp_plan = plan.dest_jp_plan (i);
#ifdef ACMODEL
    ACM_Destruction &jp_loc = plan.dest_jp_loc (i);
    _vm << signature (jp_loc) << endvm;
#else
    JPL_Destruction &jp_loc = plan.dest_jp_loc (i);
    _vm << jp_loc.signature () << endvm;
#endif

    // handle destruction joinpoint itself
    _code_weaver.dest_join_point (&jp_loc, jp_plan);
  }
  _vm--;

  _vm--;

  _vm << "Aspect Includes ..." << endvm;
  _code_weaver.aspect_includes (_project);

#ifdef PROFILING
  Profiler::ClockTicks end = Profiler::rdtsc ();
  data._time += Profiler::duration (start, end);
  data._calls = 1;
#endif
}
