/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
/*
 * ramon - argus RMON group support.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <ramon.h>

#include <math.h>
#include <signal.h>

void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
int RaInitialized = 0;
int RaTimeout     = 0x1FFFFFFF;
int RaIdleTimeout = 0;
int RaTopNModel = 0;
int RaPrintMACAddress = 0;

char *RaMonTopNModelString    = "Model 200  ip %s  0.0.0.0  no  no  no";
char *RaMonMatrixModelString  = "Model 200  ip %s %s  no  no  no";
char *RaMonHostPModelString   = "Model 200  ip %s 0.0.0.0  yes  no  no";
char *RaMonHostSvcModelString = "Model 200  ip %s 0.0.0.0  yes  no  yes";


char *RaMonMatrixFlowModelFile [] = {
"Flow  100  ip        *                   *             *       *        *  200 3153600 3153600",
"Model 200  ip 255.255.255.255     255.255.255.255      no      no       no",
NULL,
};

char *RaMonTopNFlowModelFile [] = {
"Flow  100 ip         *            *      *   *   *   200  3153600 3153600",
"Model 200 ip  255.255.255.255  0.0.0.0   no  no  no",
NULL,
};

char *RaMonServicesFlowModelFile [] = {
"Flow   10 ip     *        *     tcp   20        *    100  3153600 3153600",
"Flow  100 ip     *        *     *      *        *    200  3153600 3153600",
"Model 100 ip  0.0.0.0  0.0.0.0  yes   yes      yes",
"Model 200 ip  0.0.0.0  0.0.0.0  yes   no       yes",
NULL,
};

char *RaMonHostProtoFlowModelFile [] = {
"Flow  100 ip     *                *     *     *    *    200  0 0",
"Model 200 ip  255.255.255.255  0.0.0.0  yes   no   no",
NULL,
};

char *RaMonHostSvcFlowModelFile [] = {
"Flow  100 ip     *                *     *     *    *    200  0 0",
"Model 200 ip  255.255.255.255  0.0.0.0  yes   no   yes",
NULL,
};


struct RaFlowModelStruct *RaFlowModel = NULL;
int RaNetMode = 0;
int RaMonMode = 0;
int RaMonMask = 0;

#define RAMON_TOPN	1
#define RAMON_MATRIX	2
#define RAMON_SVCS	3
#define RAMON_HOSTP	4
#define RAMON_HOSTSVC	5

extern int RaFlowMajorModified;

void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode;
   char buf[MAXSTRLEN], *ptr = NULL;
   int i, x;

   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;
      hfield = 17;
      pfield = 8;

      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);
   
      bzero ((char *) buf, sizeof(buf));
      bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));
   
      if ((mode = ArgusModeList) == NULL)
         usage();
      
      while (mode) {
         if (!(strcasecmp (mode->mode, "TopN"))) {
            ArgusFlowModelFile = *RaMonTopNFlowModelFile;
            RaTopNModel++;
            RaMonMode = RAMON_TOPN;
            
            bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));
            RaPrintAlgorithms[0] = ArgusPrintStartDate;
            RaPrintAlgorithms[1] = ArgusPrintSrcAddr;
            RaPrintAlgorithms[2] = ArgusPrintCounts;
            RaPrintAlgorithms[3] = ArgusPrintBytes;

         } else 
         if (!(strcasecmp (mode->mode, "Matrix"))) {
            RaMonMode = RAMON_MATRIX;
            ArgusFlowModelFile = *RaMonMatrixFlowModelFile;
            RaPrintAlgorithms[0] = ArgusPrintStartDate;
            RaPrintAlgorithms[1] = ArgusPrintSrcAddr;
            RaPrintAlgorithms[2] = ArgusPrintDstAddr;
            RaPrintAlgorithms[3] = ArgusPrintCounts;
            RaPrintAlgorithms[4] = ArgusPrintBytes;

         } else
         if (!(strcasecmp (mode->mode, "Svc"))) {
            RaMonMode = RAMON_SVCS;
            ArgusFlowModelFile = *RaMonMatrixFlowModelFile;
            RaPrintAlgorithms[0] = ArgusPrintStartDate;
            RaPrintAlgorithms[1] = ArgusPrintProto;
            RaPrintAlgorithms[2] = ArgusPrintDstPort;
            RaPrintAlgorithms[3] = ArgusPrintCounts;
            RaPrintAlgorithms[4] = ArgusPrintBytes;
            pfield = 16;
         } else
         if (!(strcasecmp (mode->mode, "HostProto"))) {
            RaTopNModel++;  
            RaMonMode = RAMON_HOSTP;
            ArgusFlowModelFile = *RaMonHostProtoFlowModelFile;
            RaPrintAlgorithms[0] = ArgusPrintStartDate;
            RaPrintAlgorithms[1] = ArgusPrintSrcAddr;
            RaPrintAlgorithms[2] = ArgusPrintProto;
            RaPrintAlgorithms[3] = ArgusPrintCounts;
            RaPrintAlgorithms[4] = ArgusPrintBytes;
         } else
         if (!(strcasecmp (mode->mode, "HostSvc"))) {
            RaTopNModel++;
            RaMonMode = RAMON_HOSTSVC;
            ArgusFlowModelFile = *RaMonHostSvcFlowModelFile;
            RaPrintAlgorithms[0] = ArgusPrintStartDate;
            RaPrintAlgorithms[1] = ArgusPrintSrcAddr;
            RaPrintAlgorithms[2] = ArgusPrintProto;
            RaPrintAlgorithms[3] = ArgusPrintDstPort;
            RaPrintAlgorithms[4] = ArgusPrintCounts;
            RaPrintAlgorithms[5] = ArgusPrintBytes;
         }

         if (!(strncasecmp (mode->mode, "Net", 3))) {
            if ((ptr = strchr (mode->mode, '/')) != NULL)
               if (sscanf(&ptr[1], "%d", &RaMonMask) != 1)
                  usage();
            RaNetMode++;
         }

         mode = mode->nxt;
      }

      if (!RaNetMode)
         RaMonMask = 32;

      switch (RaMonMode) {
         case RAMON_TOPN: {
            char addrstr[128];

            if (RaMonMask != 0) {
               sprintf (addrstr, "255.255.255.255/%d", RaMonMask);
            } else
               sprintf (addrstr, "class");

            sprintf (buf, RaMonTopNModelString, addrstr);
            RaMonTopNFlowModelFile [1] = strdup(buf);
            RaFlowModel = RaReadFlowModelFile (RaMonTopNFlowModelFile);
            break;
         }

         case RAMON_MATRIX: {
            char addrstr[128];

            if (RaMonMask != 0) {
               sprintf (addrstr, "255.255.255.255/%d", RaMonMask);
            } else
               sprintf (addrstr, "class");

            sprintf (buf, RaMonMatrixModelString, addrstr, addrstr);
            RaMonTopNFlowModelFile [1] = strdup(buf);
            RaFlowModel = RaReadFlowModelFile (RaMonMatrixFlowModelFile);
            break;
         }

         case RAMON_SVCS:
            RaFlowModel = RaReadFlowModelFile (RaMonServicesFlowModelFile);
            break;

         case RAMON_HOSTP: {
            char addrstr[128];
            if (RaMonMask != 0) {
               sprintf (addrstr, "255.255.255.255/%d", RaMonMask);
            } else
               sprintf (addrstr, "class");
            sprintf (buf, RaMonHostPModelString, addrstr);
            RaMonHostProtoFlowModelFile [1] = strdup(buf);
            RaFlowModel = RaReadFlowModelFile (RaMonHostProtoFlowModelFile);
            RaFlowModel->preserve = 0;
            break;
         }

         case RAMON_HOSTSVC: {
            char addrstr[128];
            if (RaMonMask != 0) {
               sprintf (addrstr, "255.255.255.255/%d", RaMonMask);
            } else
               sprintf (addrstr, "class");
            sprintf (buf, RaMonHostSvcModelString, addrstr);
            RaMonHostSvcFlowModelFile [1] = strdup(buf);
            RaFlowModel = RaReadFlowModelFile (RaMonHostSvcFlowModelFile);
            RaFlowModel->preserve = 0;
            break;
         }

         default:
            break;
      }

      if (RaFlowModel == NULL)
         usage();
   
      if ((RaModelerQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL)
         RaHashTable.size = RA_HASHTABLESIZE;

      if (Hflag) {
         RaHistoTimeSeries = ARGUS_TIMESERIES;
      }

      if (Bflag) {
         if ((RaHoldingBuffer = (struct ArgusHoldingBuffer *) ArgusCalloc (1, sizeof(*RaHoldingBuffer))) != NULL) {
            if ((RaHoldingBuffer->array = (struct ArgusListStruct **) ArgusCalloc (Bflag, sizeof(void *))) != NULL) {
               RaHoldingBuffer->size = Bflag;
            } else {
               ArgusFree (RaHoldingBuffer);
               RaHoldingBuffer = NULL;
               exit (-1);
            }
         } else
            exit (-1);
      }

      bzero ((char *) RaSortAlgorithms, sizeof(RaSortAlgorithms));

      if (RaSOptionIndex > 0) {
         for (i = 0; i < ARGUS_MAX_S_OPTIONS; i++) {
            if (RaSOptionStrings[i] != NULL) {
               for (x = 0; x < MAX_SORT_ALG_TYPES; x++) {
                  if (!strncmp (RaSortKeyWords[x], RaSOptionStrings[i], strlen(RaSortKeyWords[x]))) {
                     RaSortAlgorithms[i] = RaSortAlgorithmTable[x];
                     break;
                  }
               }
               if (x == MAX_SORT_ALG_TYPES)
                  ArgusLog (LOG_ERR, "sort syntax error. \'%s\' not supported", RaSOptionStrings[i]);
            } else
               break;
         }
      } else
         RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTPKTSCOUNT];

      RaInitialized++;
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   int i;

   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;

      if (ArgusInput != NULL) {
         ArgusInput->ArgusManStart.ahdr.type |= ARGUS_RMON;

         if (Nflag == 0)
            Nflag = RaModelerQueue->count;

         if (Nflag > 0) {
            RaSortQueue (RaModelerQueue);

            Nflag = (Nflag > RaModelerQueue->count) ? RaModelerQueue->count : Nflag;
            for (i = 0; i < Nflag; i++) {
               if (RaModelerQueue->array[i] != NULL)
                  RaSendArgusRecord ((struct ArgusRecordStore *) RaModelerQueue->array[i]);
            }
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ramon Version %s\n", version);
   fprintf (stderr, "usage:  %s -M mode [-f flowfile]\n", ArgusProgramName);
   fprintf (stderr, "usage:  %s -M mode [-f flowfile] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -M <mode>         specify TopN, Matrix, Svc, HostProto, or HostSvc mode, modify with Net.\n");
   fprintf (stderr, "            -f <flowfile>     read flow model from <flowfile>.\n");
   fprintf (stderr, "ra-options: -a                print record summaries on termination.\n");
   fprintf (stderr, "            -A                print application bytes.\n");
   fprintf (stderr, "            -b                dump packet-matching code.\n");
   fprintf (stderr, "            -C                treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>        specify debug level\n");
   fprintf (stderr, "            -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "            -h                print help.\n");
   fprintf (stderr, "            -n                don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>       print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -q                quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R                print out response data when availabile.\n");
   fprintf (stderr, "            -S <host[:port]>  specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "            -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                     format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                              timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                  mm/dd[/yy]\n");
   fprintf (stderr, "                                                  -%%d{yMhdms}\n");
   fprintf (stderr, "            -T <secs>         attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u                print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth>    specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>         write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z                print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>        print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}

void
RaProcessRecord (struct ArgusRecord *argus)
{
   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }
}

struct ArgusRecord * RaRmonArgusRecord (struct ArgusRecord *);
void RaProcessThisSrvRecord (struct ArgusServiceRecord *);
void RaCreateEtherFlow (struct ArgusRecord *);
void RaModifyEtherFlow(struct ArgusRecord *);

void
RaModifyEtherFlow (struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   bzero ((char *) &flow->mac_flow.ehdr.ether_dhost, 6);
}


void
RaCreateEtherFlow (struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   struct ArgusMacStruct *mac = NULL;
   struct ArgusETHERObject *ether = NULL;
   
   if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
      if ((mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX]) != NULL) {
         ether = &mac->ether_mac;
         bzero ((char *) flow, sizeof(*flow));
         bcopy ((char *) ether, (char *) &flow->mac_flow.ehdr, 12);
         argus->ahdr.status &= ~0xFFFF;
      }

   } else {
      ArgusLog (LOG_ERR, "ether addrs not present");
   }
}

struct ArgusRecord *
RaRmonArgusRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *retn = NULL;
   struct ArgusMeter meter;
   struct ArgusFlow *flow1, *flow2;
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0;

   if (RaPrintMACAddress)
      RaCreateEtherFlow (argus);

   if ((retn = RaCopyArgusRecord (argus)) != NULL) {

      a1DSRStatus = ArgusIndexRecord (argus, a1farhdr);
      a2DSRStatus = ArgusIndexRecord (retn,  a2farhdr);
   
      flow1 = &argus->argus_far.flow;
      flow2 = &retn->argus_far.flow;

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            flow2->ip_flow.ip_src = flow1->ip_flow.ip_dst;
            flow2->ip_flow.ip_dst = flow1->ip_flow.ip_src;

            switch (flow1->ip_flow.ip_p) {
               case IPPROTO_UDP:
               case IPPROTO_TCP:
                  flow2->ip_flow.sport = flow1->ip_flow.dport;
                  flow2->ip_flow.dport = flow1->ip_flow.sport;
                  break;
            }

            meter = argus->argus_far.src;
            argus->argus_far.src = argus->argus_far.dst;
            argus->argus_far.dst = meter;

            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            break;

         default:
            bcopy ((char *)&flow1->mac_flow.ehdr.ether_dhost,
                   (char *)&flow2->mac_flow.ehdr.ether_shost, 6);
            bzero ((char *)&flow1->mac_flow.ehdr.ether_dhost, 6);
            bzero ((char *)&flow2->mac_flow.ehdr.ether_dhost, 6);

            meter = retn->argus_far.src;
            retn->argus_far.src = retn->argus_far.dst;
            retn->argus_far.dst = meter;
            break;
      }

      if ((a1DSRStatus & ARGUS_MAC_DSR_STATUS) && (a2DSRStatus & ARGUS_MAC_DSR_STATUS)) {
         struct ArgusMacStruct *mac1 = NULL, *mac2 = NULL;

         mac1 = (struct ArgusMacStruct *) a1farhdr[ARGUS_MAC_DSR_INDEX];
         mac2 = (struct ArgusMacStruct *) a2farhdr[ARGUS_MAC_DSR_INDEX];

         bcopy ((char *)&mac2->ether_mac.etherdst, (char *)&mac1->ether_mac.ethersrc, 6);
         bzero ((char *)&mac1->ether_mac.etherdst, 6);
         bzero ((char *)&mac2->ether_mac.etherdst, 6);
      }

       retn->ahdr.type |= ARGUS_RMON;
      argus->ahdr.type |= ARGUS_RMON;
   }

   return (retn);
}


void
RaProcessThisSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *arg2 = NULL;

   if (RaTopNModel) {
      arg2 = RaRmonArgusRecord(srv->argus);
      RaProcessSrvRecord(srv);
      srv->argus = arg2;
      RaProcessSrvRecord(srv);
      ArgusFree(arg2);

   } else
      RaProcessSrvRecord(srv);
}


void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;
   struct ArgusFlow *flow = &argus->argus_far.flow;

   RaThisActiveDuration = RaGetActiveDuration(argus);

   if (Hflag) {
      RaThisActiveIndex = (RaThisActiveDuration < 100)        ? 0 :
                          (RaThisActiveDuration < 500)        ? 1 :
                          (RaThisActiveDuration < 1000)       ? 2 :
                          (RaThisActiveDuration < 5000)       ? 3 :
                          (RaThisActiveDuration < 10000)      ? 4 :
                          (RaThisActiveDuration < 50000)      ? 4 :
                          (RaThisActiveDuration < 100000)     ? 5 :
                          (RaThisActiveDuration < 500000)     ? 6 :
                          (RaThisActiveDuration < 100000)     ? 7 :
                          (RaThisActiveDuration < 250000)     ? 8 :
                          (RaThisActiveDuration < 500000)     ? 9 :
                          (RaThisActiveDuration < 750000)     ? 10:
                          (RaThisActiveDuration < 1000000)    ? 11:
                          (RaThisActiveDuration < 5000000)    ? 12:
                          (RaThisActiveDuration < 10000000)   ? 13:
                          (RaThisActiveDuration < 50000000)   ? 14: 15;
   } else
      RaThisActiveIndex = 0;

   bcopy((char *) flow, (char *) RaArgusFlow, sizeof(struct ArgusFlow));

   if (!(RaPrintMACAddress)) {
      if ((rap = RaFlowModelOverRides(argus, RaFlowModel)) != NULL) {
         RaModifyFlow(rap, argus, RaFlowModel);
         RaTimeout = rap->ArgusTimeout;
         RaIdleTimeout = rap->ArgusIdleTimeout;
      } else {
         RaModifyDefaultFlow(argus);
      }
   } else
      RaModifyEtherFlow(argus);

   if ((store = RaFindArgusRecord(&RaHashTable, argus)) == NULL) {
   }

   if (store) {
      RaThisArgusStore = store;

      RaCheckTimeout (store, argus);

      if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
         RaSendArgusRecord(store);
         store->status &= ~RA_SVCTEST;
         store->status |= (srv->status & RA_SVCTEST);
      }

      if (!(store->data[RaThisActiveIndex])) {
         struct ArgusRecordData *data = NULL;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.meanval * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.meanval * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else
         RaMergeArgusRecord(argus, store, RaThisActiveIndex);

      RaUpdateArgusStore(argus, store);

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         struct ArgusRecordData *data = NULL;

         store->ArgusTimeout     = RaTimeout;
         store->ArgusIdleTimeout = RaIdleTimeout;

         RaThisArgusStore = store;

         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.meanval * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.meanval * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;

            if ((store->rahtblhdr = RaAddHashEntry (&RaHashTable, store)) != NULL)
               RaAddToQueue(RaModelerQueue, &store->qhdr);
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

   RaIdleTimeout = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaProcessSrvRecord: done\n");
#endif
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{
   unsigned int status;
   struct ArgusRecord *initCon = &ArgusInput->ArgusInitCon;
   
   initCon->ahdr.type |= ARGUS_RMON;
   status = ntohl(initCon->ahdr.status);

   if (RaTopNModel)
      status |= ARGUS_TOPN;
   else
      status |= ARGUS_MATRIX;

   initCon->ahdr.status = htonl(status);
}


struct ArgusServiceRecord ArgusThisSrv;

void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (tcp->state & ARGUS_DST_RESET)
                     if (argus->argus_far.src.count && argus->argus_far.dst.count)
                        if (!(argus->argus_far.src.bytes && argus->argus_far.dst.bytes))
                           ArgusThisSrv.status = RA_SVCFAILED;
       
                  if (tcp->state & ARGUS_SRC_RESET)
                     if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                        ArgusThisSrv.status = RA_SVCFAILED;
               }
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      }
   }

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
RaProcessUDPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) 
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
RaProcessARPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessThisSrvRecord (&ArgusThisSrv);
}


#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  25000
#define RA_MAXQSIZE  250000
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
            RaTimeoutArgusStore(obj);

         break;

      default:
         while (queue->count > RA_MAXQSIZE) {
            obj = (struct ArgusRecordStore *) RaRemoveFromQueue(RaModelerQueue, RaModelerQueue->start->prv);
            RaTimeoutArgusStore(obj);
         }

         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


extern struct ArgusCanonicalRecord *RaThisCanon;
int RaLabelCounter = 0;

int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   struct ArgusAGRStruct *agr = NULL;
   int retn = 0, i;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (!aflag) {
#ifdef _LITTLE_ENDIAN
            ArgusHtoN(argus);
#endif
            ArgusGenerateCanonicalRecord (argus, RaThisCanon);
 
#ifdef _LITTLE_ENDIAN
            ArgusNtoH(argus);
#endif
            if ((retn = argus_filter (ArgusFilterCode.bf_insns, RaThisCanon)) == 0)
               return (retn);
         }

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.meanval = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL)
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);

            bcopy ((char *) argus, buf, argus->ahdr.length);
            argus = (struct ArgusRecord *) buf;

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) == NULL) {
               if (ArgusFlowModelFile && (data->agr.count > 1)) {
                  bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                  argus->ahdr.length += data->agr.length;
                  argus->ahdr.status |= ARGUS_MERGED;
                  ArgusFree (data->argus);
                  data->argus = RaCopyArgusRecord(argus);
                  ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);

               } else {
                  argus->ahdr.status &= ~(ARGUS_MERGED);
               }
            }
      
            if ((ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusWfileList)))) {
               struct ArgusWfileStruct *wfile = NULL, *start = NULL;
 
               if ((wfile = ArgusFrontList(ArgusWfileList)) != NULL) { 
                  start = wfile; 
                  do { 
                     if ((exceptfile == NULL) || strcmp(wfile->filename, exceptfile)) { 
#ifdef _LITTLE_ENDIAN
                        ArgusHtoN(argus); 
#endif
                        ArgusWriteNewLogfile (wfile, argus); 
 
#ifdef _LITTLE_ENDIAN
                        ArgusNtoH(argus); 
#endif
                     }   
                     ArgusPopFrontList(ArgusWfileList); 
                     ArgusPushBackList(ArgusWfileList, wfile); 
                     wfile = ArgusFrontList(ArgusWfileList); 
 
                  } while (wfile != start); 
               }   

            } else {
      
               if (argus->ahdr.type & ARGUS_MAR)
                  printf ("%s\n", get_man_string (argus));
                  
               else {

                  if (Lflag && !(qflag)) {
                     if (RaLabel == NULL)
                        RaLabel = RaGenerateLabel(argus);

                     if (!(RaLabelCounter++ % Lflag)) {
                        printf ("%s\n", RaLabel);
                        fflush (stdout);
                     }

                     if (Lflag < 0)
                        Lflag = 0;
                  }

                  ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                  switch (argus->ahdr.status & 0xFFFF) {
                     case ETHERTYPE_IP:
                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP: { 
                              int srcportind = 0;
                              if (argus->argus_far.flow.ip_flow.sport == 20) {
                                 if (RaPrintAlgorithms[3] == ArgusPrintDstPort) {
                                    RaPrintAlgorithms[3] = ArgusPrintSrcPort;
                                    srcportind++;
                                 }
                              }
                              
                              printf ("%s", get_tcp_string (argus));

                              if (srcportind)
                                 RaPrintAlgorithms[3] = ArgusPrintDstPort;
                              break;
                           }
            
                           case IPPROTO_ICMP:              
                              printf ("%s", get_icmp_string (argus));
                              break;
            
                           default:
                              printf ("%s", get_ip_string (argus));
                              break;
                        }
                        break;
            
                     case ETHERTYPE_ARP:
                     case ETHERTYPE_REVARP:
                        printf ("%s", get_arp_string (argus));
                        break;
            
                     default:
                        printf ("%s", get_nonip_string (argus));
                        break;
                  }

                  printf ("\n");
               }
               fflush (stdout);
            }
      
            argus = data->argus;

            if (argus->ahdr.type & ARGUS_FAR) {
               int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);
               farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
      
               while (length > 0) {
                  switch (farhdr->type) {
                     case ARGUS_FAR: {
                        struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                        far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                        far->time.last.tv_sec = 0; far->time.last.tv_usec = 0;
                        far->src.count = 0; far->src.bytes = 0;
                        far->dst.count = 0; far->dst.bytes = 0;
                        break;
                     }
      
                     case ARGUS_TCP_DSR: {
                        struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
      
                        tcp->state = 0;
                        tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                        tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                        tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                        tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
      
                        break;
                     }
      
                     case ARGUS_TIME_DSR: {
                        struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) farhdr;

                        time->src.act.n = 0;
                        time->src.act.meanval = 0;
                        time->src.act.stdev = 0;
                        time->src.act.maxval = 0;
                        time->src.act.minval = 0x7FFFFFFF;
                        time->src.idle.n = 0;
                        time->src.idle.meanval = 0;
                        time->src.idle.stdev = 0;
                        time->src.idle.maxval = 0;
                        time->src.idle.minval = 0x7FFFFFFF;

                        time->dst.act.n = 0;
                        time->dst.act.meanval = 0;
                        time->dst.act.stdev = 0;
                        time->dst.act.maxval = 0;
                        time->dst.act.minval = 0x7FFFFFFF;
                        time->dst.idle.n = 0;
                        time->dst.idle.meanval = 0;
                        time->dst.idle.stdev = 0;
                        time->dst.idle.maxval = 0;
                        time->dst.idle.minval = 0x7FFFFFFF;

                        break;
                     }

                     case ARGUS_VLAN_DSR: {
                        struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
                        vlan->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_MPLS_DSR: {
                        struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
                        mpls->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_AGR_DSR: {
                        struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
      
                        agr->count = 0;
                        agr->act.n = 0;
                        agr->act.minval = 0x7FFFFFFF;   agr->act.meanval = 0;
                        agr->act.stdev = 0;  agr->act.maxval = 0;
                        agr->idle.n = 0;
                        agr->idle.minval = 0x7FFFFFFF; agr->idle.meanval = 0;
                        agr->idle.stdev = 0; agr->idle.maxval = 0;
                        break;
                     }
                  }

                  if ((farlen = farhdr->length) == 0)
                     break;
                  if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                      (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                     farlen = farlen * 4;
                  length -= farlen;
                  farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
               }
            }

            data->agr.count = 0;
            data->agr.act.n = 0;
            data->agr.act.meanval = 0;
            data->agr.act.stdev = 0;
            data->agr.act.maxval = 0;
            data->agr.act.minval = 0x7FFFFFFF;
            data->agr.idle.n = 0;
            data->agr.idle.meanval = 0;
            data->agr.idle.stdev = 0;
            data->agr.idle.maxval = 0;
            data->agr.idle.minval = 0x7FFFFFFF;
      
            data->act.n = 0;
            data->act.sumtime = 0;
            data->act.sumsqrd = 0;
            data->idle.n = 0;
            data->idle.sumtime = 0;
            data->idle.sumtime = 0;
      
            data->argus->ahdr.status &= ~ARGUS_MERGED;
            data->status &= ~RA_MODIFIED;
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif

   return (retn);
}

