/*
 * ifp supporting functions.. "compound" ones
 * $Id: comp.c,v 1.19.2.1 2005/08/22 03:03:56 oakhamg Exp $
 *
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 */

#include "ifp.h"
#include "ifp_os.h"
#include "prim.h"

static char * battery_status_art [] = {
    "=[    ]",
    "=[#   ]",
    "=[##  ]",
    "=[### ]",
    "=[####]",
};

/** \brief Creates a human readable status string.
 *
 * Creates a human readable status string similar to
 * "model IFP-007T, firmware 1.14, battery =[####], delta 1.8.4.42".
 */
int ifp_device_info(struct ifp_device * dev, char * s, int n) {
    char * p = dev->b2;
    char * battery_info = NULL;
    int i = 0;
    int version, battery;
    int deltas[4];

    i = ifp_model(dev, p, IFP_BUFFER_SIZE);
    if (i) {
        ifp_err_i(i, "error getting model number.");
        p[0] = '-';
        p[1] = '\0';
    }

    version = ifp_firmware_version(dev);
    if (version < 0) {
        ifp_err_i(version, "error getting firmware version.");
        i = version;
    }

    battery = ifp_battery(dev);
    if (battery < 0) {
        ifp_err_i(battery, "error getting firmware version.");
        battery_info = "=[fubr]";
        i = battery;
    } else if (battery > 4) {
        battery_info = "=[????]";
    } else {
        battery_info = battery_status_art[battery];
    }

    i = ifp_delta(dev, deltas);
    if (i) {
        ifp_wrn("problem getting delta value (error code %d).", i);
        //experimental feature, don't return the error
        i = 0;
    }

    snprintf(s, n, "model %s, firmware %x.%02x, battery %s"
            ", delta %d.%d.%d.%d"
            , p, version/0x100, version % 0x100, battery_info
            , deltas[0], deltas[1], deltas[2], deltas[3]
            );

    return i;
}
IFP_EXPORT(ifp_device_info);

static int noop_dir_callbk(void * p, int t, const char * s, int fs)
{
    int * pn = p;
    if (*pn == 5) {
        //return early--save time.
        return 1;
    }
    (*pn)++;
    return 0;
}

/** \brief Tests communications with the device.
 *
 * This is done automatically on startup by ::ifp_init, so normal shouldn't
 * need to call this.. unless they really want to.
 */
int ifp_selftest(struct ifp_device * dev) {
    int i = 0;
    int n = 0;

    i = ifp_firmware_version(dev);
    if (i < 0) {
        ifp_err_i(i, "couldn't get firmware version.");
        return i;
    }

    i = ifp_battery(dev);
    if (i < 0) {
        ifp_err_i(i, "couldn't get battery status.");
        return i;
    }

    i = ifp_list_dirs(dev, "\\", noop_dir_callbk, &n);
    if (i) {
        ifp_err_i(i, "couldn't get basic directory listing.");
        return i;
    }

    if (n < 3) {
        ifp_wrn("only %d items could be found in the root directory.  Either there's a problem, or the device is empty.", n);
    }

    return i;
}
IFP_EXPORT(ifp_selftest);

/** returns -ENOENT, -EEXIST */
static int touch(struct ifp_device * dev, const char * dir, const char * file)
{
	int i = 0;

	//ifp_dbg("[ifp touch] touching %s\n", (char *)file);
	//ifp_dbg("[ifp touch] in %s\n", (char *)dir);

	i = ifp_dir_open(dev, dir);
        ifp_err_expect(i, i==-ENOENT, out, "couldn't open dir.");
	i = ifp_file_open_new(dev, file, 0);
        ifp_err_expect(i, i==-EEXIST||i==-IFP_ERR_BAD_FILENAME, out, "couldn't create file.");
	i = ifp_file_close(dev);
        ifp_err_jump(i, out, "couldn't close file.");
	i = ifp_dir_close(dev);
        ifp_err_jump(i, out, "couldn't close dir.");

out:
	return i;
}

int _ifp_list_dirs_debug(
	struct ifp_device * dev,
	const char * name, int n,
	int(*callbk)(void *, int, const char *, int, int, int, int),
	void * context)
{
	//unsigned char utf8_buf[IFP_BULK_MAXPATHLEN];
	uint8_t * utf8_buf = dev->b2;
	int i = 0;

	i = ifp_dir_open(dev, name);
	ifp_err_jump(i, out, "dir.open failed.");

	while(1) {
		int dir, slot, entrysize;
		int ret;
		ret = ifp_dir_next_debug(dev, utf8_buf, IFP_BUFFER_SIZE,
			IFP_FILE|IFP_DIR, &dir, &slot, &entrysize);
		if (ret < 0) {
			ifp_err_i(ret, "error sending dir.next.");
			i = ret; goto out;
		} else if (ret == 0) {
			//ifp_dbg("Done.");
			break;
		} else {
			i = callbk(context, ret, utf8_buf,
				strlen(utf8_buf),
				dir, slot, entrysize);
			if (i < 0) {
				ifp_err_i(i, "callback returned error.");
				break;
			} else if (i == 1) {
				//ifp_dbg("callback requested early break.");
				break;
			} else if (i != 0) {
				ifp_err_i(i, "callback returned a wierd value.");
				i = -1;
				goto out;
			} else {
				//ifp_dbg("callback returned %d for '%s'.",
				//	i, utf8_buf);
			}
		}
	}

	i = ifp_dir_close(dev);
	ifp_err_jump(i, out, "dir.close failed.");

	i = 0;
out:
	return i;
}

static int get_id_callback(void * context, int type, const char * name, int n,
	int dir, int slot, int entrysize)
{
	void ** pp = (void **) context;
	char * target = pp[0];
	int * target_width = pp[1];
	int * pdir = pp[2];
	int * pslot = pp[3];
	int * pentrysize = pp[4];
	int * found = pp[5];

#if 0
	if (entrysize>1) {
		ifp_dbg("%10.10s: dir=%#x %d slots end at %2d, so you can find them in %d-%d",
			name, dir, entrysize, slot, slot-entrysize, slot-1);
	} else {
		ifp_dbg("%10.10s: dir=%#x  a slot ends at %2d, it's just %d",
			name, dir, slot, slot-1);
	}
#endif
	if ( *target_width == n) {
		if (strncmp(target, name, n)==0)
		{
			//printk("[get_id_callback] '%s' matches '%s'.\n",
			//	target, name);
			*pdir = dir;
			*pslot = slot;
			*pentrysize = entrysize;
			*found = 1;
			return 1;
		}
	}

	return 0;
}

static int get_file_id(struct ifp_device * dev, const char * dir, const char * file,
	int * id_dir, int * id_slot, int * id_size)
{
	int i = 0;
	int filelen;
	int dirlen;
	int found = 0;
	void ** context[6] = {(void*)file, (void*)&filelen, (void*)id_dir,
		(void*)id_slot, (void*)id_size, (void*)&found};
	dirlen = strlen(dir);
	//one extra for the path separator
	filelen = strlen(file) - dirlen;
	file += dirlen;
	if (dirlen > 1) {
		filelen--;
		file++;
	}
	//ifp_dbg("file='%s', filelen=%d, dir='%s', dirlen=%d", file, filelen,
	//	dir, dirlen);
	if (filelen < 0) {
		ifp_err_i(i, "bailing");
		return -1;
	}
	IFP_BUG_ON(filelen < 0);
	context[0] = (void *)(file);

	//FIXME: return an error on not finding a matching entry.
	i = _ifp_list_dirs_debug(dev, dir, strlen(dir), get_id_callback, (void *)context);
	ifp_err_jump(i, out, "list failed.");
	if (found == 0) {
		ifp_err("entry not found for dir='%s' file='%s'", dir, file);
		i = -1;
	}
	//ifp_dbg("returning dir=%#x, page=%#x, off=%#x", *id_dir, *id_slot, *id_size);

out:
	return i;
}

/*
 * need to convert "slot number" -> "page" and "offset"
 *
 *
 */

/* Calculates the address of 
 *
 * Given that an entry ends on (base_page, slot_number
 * */
static inline int calc_slot_address(
	int slot_number,
	int record_offset,
	int * page,
	int * offset
	)
{
	const int slot_width = IFP_FAT_SLOT_WIDTH;
	int abs_offset;
	IFP_BUG_ON(page == NULL);
	IFP_BUG_ON(offset == NULL);

	abs_offset = slot_width * slot_number + record_offset;
	*page      = abs_offset / IFP_FAT_PAGE_SIZE;
	*offset    = abs_offset % IFP_FAT_PAGE_SIZE;
	
	return 0;
}

static int swap_fat_entries(struct ifp_device * dev,
	int id_dirA, int id_slotA, int id_sizeA,
	int id_dirB, int id_slotB, int id_sizeB)
{
	const int data_ptr_size = 8;
	const int page_size  = IFP_FAT_PAGE_SIZE;
	uint8_t tmp[data_ptr_size];

	uint8_t * _buffA = dev->b2, * _buffB = dev->b3;
	uint8_t * buffA = _buffA, * buffB = _buffB;

	int i = 0;
	int id_pageA, id_pageB, id_offA, id_offB;

//	+2*IFP_FAT_PAGE_SIZE/IFP_FAT_SLOT_WIDTH == +2*16 == +32 == +0x20
	i = calc_slot_address(id_slotA+id_sizeA - 1, IFP_FAT_SLOT_WIDTH - data_ptr_size,
		&id_pageA, &id_offA);
	ifp_err_jump(i, out, "cal failed. (a)");

	i = calc_slot_address(id_slotB+id_sizeB-1, IFP_FAT_SLOT_WIDTH - data_ptr_size,
		&id_pageB, &id_offB);
	ifp_err_jump(i, out, "cal failed. (b)");

	i = ifp_get_fat_page(dev, id_dirA, id_pageA, _buffA, page_size);
	ifp_err_jump(i, out, "read failed for page=%#x, dir=%#x (a)",
		id_pageA, id_dirA);

	if (id_dirA == id_dirB && id_pageA == id_pageB) {
		//ifp_dbg("same page number, using same buffers.");
		buffB = _buffA;
	} else {
		i = ifp_get_fat_page(dev, id_dirB, id_pageB, _buffB, page_size);
		ifp_err_jump(i, out, "read failed for page=%#x, dir=%#x (b)",
			id_pageB, id_dirB);
	}

	buffA += id_offA;
	buffB += id_offB;

#if 0
	{
		uint8_t *slot;

		slot = buffA - (IFP_FAT_SLOT_WIDTH - data_ptr_size);
		ifp_dbg("slot %d is %02x ('%s')", id_slotA, (int)slot[0], slot);
		ifp_dbg("page %d, offset %d", id_pageA, id_offA);

		slot = buffB - (IFP_FAT_SLOT_WIDTH - data_ptr_size);
		ifp_dbg("slot %d is %02x ('%s')", id_slotB, (int)slot[0], slot);
		ifp_dbg("page %d, offset %d", id_pageB, id_offB);
	}
#endif

	memcpy(tmp,   buffA, data_ptr_size);
	memcpy(buffA, buffB, data_ptr_size);
	memcpy(buffB, tmp,   data_ptr_size);

	i = ifp_set_fat_page(dev, id_dirA, id_pageA, _buffA, page_size);
	ifp_err_jump(i, out, "write failed. (a)");

	if (id_dirA != id_dirB || id_pageA != id_pageB) {
		i = ifp_set_fat_page(dev, id_dirB, id_pageB, _buffB, page_size);
		ifp_err_jump(i, out, "write failed. (b)");
	}

out:
	return i;
}

static int swap_filenames(struct ifp_device * dev, const char * oldpath, const char * newpath)
{
	int id_dirA, id_dirB, id_slotA, id_slotB, id_sizeA, id_sizeB;
	int i;
	char * b = dev->b3;

	i = ifp_copy_parent_string(b, oldpath, IFP_BUFFER_SIZE);
	ifp_err_jump(i, out, "getting parent directory of %s failed.",oldpath);
	i = get_file_id(dev, b, oldpath, &id_dirA, &id_slotA, &id_sizeA);
	ifp_err_jump(i, out, "file_id failed. (a)");

	i = ifp_copy_parent_string(b, newpath, IFP_BUFFER_SIZE);
	ifp_err_jump(i, out, "getting parent directory of %s failed.",newpath);
	i = get_file_id(dev, b, newpath, &id_dirB, &id_slotB, &id_sizeB);
	ifp_err_jump(i, out, "file_id failed. (b)");
	b = NULL; //release b3

	//uses b1,b2,b3.
	i = swap_fat_entries(dev, id_dirA, id_slotA, id_sizeA,
				id_dirB, id_slotB, id_sizeB);
	ifp_err_jump(i, out, "swap failed. "
		"dirA=%#x, pageA=%#x, offA=%#x, dirB=%#x, pageB=%#x, offB=%#x",
		id_dirA, id_slotA, id_sizeA, id_dirB, id_slotB, id_sizeB);

out:
	return i;
}

/** returns -ENOENT, -EEXIST, IFP_ERR_BAD_FILENAME */
int ifp_rename_file(struct ifp_device * dev, const char * old_file, const char * new_file)
{
	int i;
	char * b = dev->b3;

	i = ifp_copy_parent_string(b, new_file, strlen(new_file));
	ifp_err_jump(i, out, "getting parent directory of %s failed.",new_file);
	i = touch(dev, b, new_file);
	ifp_err_expect(i, i==-ENOENT || i==-EEXIST || i==IFP_ERR_BAD_FILENAME, out, "Touch failed.");
	b = NULL; //release b3

	i = swap_filenames(dev, old_file, new_file);
	ifp_err_jump(i, out, "filename swap failed.");

	i = ifp_delete(dev, old_file);
	ifp_err_jump(i, out, "delete failed.");

out:
	return i;
}

/** returns -ENOENT, -EEXIST, -EACCES, IFP_ERR_BAD_FILENAME */
int ifp_rename_dir(struct ifp_device * dev, const char * old_dir, const char * new_dir)
{
	int i;

	if (strcmp(old_dir, "\\VOICE") == 0 || strcmp(old_dir, "\\RECORD") == 0) {
		return -EACCES;
	}

	i = ifp_mkdir(dev, new_dir);
	ifp_err_expect(i, i==-ENOENT || i==-EEXIST || i==IFP_ERR_BAD_FILENAME, out, "mkdir failed.");

	i = swap_filenames(dev, old_dir, new_dir);
	ifp_err_jump(i, out, "filename swap failed.");

	i = ifp_rmdir(dev, old_dir);
	ifp_err_jump(i, out, "rmdir failed.");

out:
	return i;
}

/** \brief Renames a file or directory.
 
 Renames or moves the object 'old_path' to 'new_path'.

 \param old_path an existing file or directory.
 \param new_path an available path for a new file or directory.
 	(Ie, the path's parent directory exists and the path isn't in use
	by another object in that directory.)

 Returns 0 on success and  -ENOENT, -EEXIST, -EACCES, IFP_ERR_BAD_FILENAME on failure,
 as appropriate.
 */
int ifp_rename(struct ifp_device * dev, const char * old_path, const char * new_path)
{
	int i;

	i = ifp_is_dir(dev, old_path);
	if (i < 0) {
		ifp_err_jump(i, out, "ifp_is_dir failed");
	}
	if (i) {
		//directory
		i = ifp_rename_dir(dev, old_path, new_path);
		ifp_err_expect(i, i==-ENOENT || i==-EEXIST || i==-EACCES, out,
			"ifp_rename_dir failed");
	} else {
		i = ifp_rename_file(dev, old_path, new_path);
		ifp_err_expect(i, i==-ENOENT || i==-EEXIST, out,
			"ifp_rename_file failed");
	}

out:
	return i;
}
IFP_EXPORT(ifp_rename);

static int get_file_size(struct ifp_device * dev,
	const char * dir, int dsize,
	const char * f, int fsize)
{
	int size = 0;
	int i;
	uint8_t * buf = dev->b3;
	uint8_t * p = buf;
	int dlen, flen;

	dlen = strlen(dir);
	flen = strlen(f);
	IFP_BUG_ON(dlen + flen + 2 >= IFP_BULK_MAXPATHLEN);
	memcpy(p, dir, dlen); p += dlen;
	if (dlen > 1) {
		//if dir ls simply \ we don't want to make it \\.
		*p = '\\'; p++;
	}
	memcpy(p, f, flen); p += flen;
	*p = 0; p++;


	i = ifp_file_open(dev, buf);
	if (i == 1) {
        	ifp_err("The file '%s' doesn't exist!", buf);
		return -ENOENT;
	} else if (i) {
        	ifp_err_i(i, "Error opening file '%s'.", buf);
		return i < 0 ? i : -EIO;
	}

	size = ifp_file_size(dev);
	if (size < 0) {
        	ifp_err_i(size, "Error getting size of '%s'.", buf);
		//fallthrough
	}
	i = ifp_file_close(dev);
	if (i) {
        	ifp_err_i(i, "Error closing file '%s'.", buf);
		return i < 0 ? i : -EIO;
	}

	return size;
}

int _ifp_list_dirs(
	struct ifp_device * dev,
	const char * dirname, int dsize,
	int type,
	int(*callbk)(void *, int, const char *, int),
	void * context)
{
	uint8_t * utf8_buf = dev->b2;
	int i = 0;

	IFP_BUG_ON(sizeof(dev->b2) < IFP_BULK_MAXPATHLEN);

	while(1) {
		int ret;
		ret = ifp_dir_next(dev, utf8_buf, IFP_BULK_MAXPATHLEN, type);
		if (ret < 0) {
        		ifp_err_i(ret, "Error sending control message dir.next.");
			i = ret;
			return i;
		} else if (ret == 0) {
			//ifp_dbg("Done.");
			break;
		} else {
			int filesize = 0;
			if (ret == IFP_FILE) {
				filesize = get_file_size(dev,
					dirname, dsize, utf8_buf,
					IFP_BULK_MAXPATHLEN);
			}
			if (filesize < 0) {
        			ifp_err_i(filesize, "Error getting filesize.");
				return filesize;
			}

			i = callbk(context, ret, utf8_buf, filesize);
			if (i < 0) {
        			ifp_err_i(i, "Callback returned error.");
				return i;
			} else if (i == 1) {
				//ifp_dbg("[_ifp_list_dirs] callback requested early break.\n");
				break;
			} else if (i != 0) {
        			ifp_err_i(i, "Callback returned a wierd value.");
				return -EIO;
			}
		}
	}

	i = 0;
	return i;
}

/** \brief Reads directory contents.
 
  Passes the contents of 'dirname' to a callback function, one entry at
  a time.  The parameters given to the callback function are:
    - void * context is the same context passed to list_dirs
    - int type is either ::IFP_FILE or ::IFP_DIR
    - char * name is the entry name without a full path.  (Ie, no '\\' chars)
    - int filesize is the number of bytes in a file (undefied for directories)
    .
  The callback can return '0' on success, '1' to "break" (leave early
  without error) or <0 on error.

  Returns 0 on success or -ENOENT if the directory doesn't exist.
 */
int ifp_list_dirs(
	struct ifp_device * dev,
	const char * filename, 
	int(*callbk)(void *, int, const char *, int),
	void * context)
{
	int i = 0;

	i = ifp_dir_open(dev, filename);
	ifp_err_expect(i, i==-ENOENT, out, "dir.open failed.");

	i = _ifp_list_dirs(dev, filename, strlen(filename), IFP_FILE|IFP_DIR,
		callbk, context);
	ifp_err_jump(i, out, "_list_dirs failed.");

	i = ifp_dir_close(dev);
	ifp_err_jump(i, out, "dir.close failed.");

out:
	return i;
}
IFP_EXPORT(ifp_list_dirs);

static int _subdir_counter(void * context, int type, const char * name, int size)
{
	int * pn = context;
	(*pn) += 1;

	return 0;
}

//NOTE: this is should be removed
int ifp_count_subdirs(struct ifp_device * dev, const char * dirname)
{
	int i = 0;
	int n = 0;

	i = ifp_dir_open(dev, dirname);
	if (i) {
		ifp_err_i(i, "dir_open failed");
		return i;
	}

	i = _ifp_list_dirs(dev, dirname, strlen(dirname), IFP_DIR, _subdir_counter, &n);
	if (i) {
		ifp_err_i(i, "ifp_list_dirs failed");
		return i;
	} else {
		IFP_BUG_ON(n < 0);
	}

	i = ifp_dir_close(dev);
	if (i) {
		ifp_err_i(i, "dir_close failed");
		return i;
	}

	return n;
}

/** \brief Tests if f is a file.
 *
 * Returns 1 if it is, and 0 if it doesn't exist or isn't a file.  */
int ifp_is_file(struct ifp_device * dev, const char * f) {
	int i = 0, r = 0;
	char * b = dev->b2;

	i = ifp_copy_parent_string(b, f, IFP_BULK_MAXPATHLEN);
	ifp_err_jump(i, out, "parent directory copy failed");

	i = ifp_dir_open(dev, b);
	if (i == -ENOENT) {
		//no such dir, therefore no such file. qed
		return 0;
	} else if (i != 0) {
		ifp_err_i(i, "dir.open failed");
		goto out;
	}

	i = ifp_file_open(dev, f);
	if (i == -ENOENT) {
		//no such file
		r = 0;
	} else {
		r = 1;
		i = ifp_file_close(dev);
		ifp_err_jump(i, out, "file.close failed.");
	}
	i = ifp_dir_close(dev);
	ifp_err_jump(i, out, "dir.close failed.");

	return r;
out:
	if (i > 0) {
		i = -1;
	}
	return i;
}
IFP_EXPORT(ifp_is_file);

/** \brief Tests if f is a directory.
 *
 * Returns 1 if it is, and 0 if it doesn't exist or isn't a dir.  */
int ifp_is_dir (struct ifp_device * dev, const char * f) {
	int i = 0;
	i = ifp_dir_open(dev, f);
	if (i == 0) {
		//does exist
		i = ifp_dir_close(dev);
		ifp_err_jump(i, out, "dir.close failed.");
		//ifp_dbg("here, dir ifp:\\%s exists", f);
		return 1;
	} else if (i == -ENOENT) {
		//does not exist
		//ifp_dbg("here, dir ifp:\\%s does not exist", f);
		return 0;
	} else {
		//chipmunk
		ifp_err_i(i, "dir.open failed");
	}
out:
	i = i < 0 ? i : -1;
	return i;
}
IFP_EXPORT(ifp_is_dir);

/** \brief Tests for the existance of f.
 *
 * \return ::IFP_FILE if f is a file
 * \return ::IFP_DIR  if f is a directory
 * \return 0        if f doesn't exist
 * \return <0       error.
 */
int ifp_exists (struct ifp_device * dev, const char * f)
{
	int i = 0;

	i = ifp_is_dir(dev, f);
	//ifp_dbg("is_dir returned %d for %s.", i, f);
	if (i == 1) {
		return IFP_DIR;
	} else if (i < 0) {
		ifp_err_i(i, "dir checking failed");
		return i;
	} else if (i != 0) {
		ifp_err_i(i, "unexpected result checking dir");
		return -1;
	}


	i = ifp_is_file(dev, f);
	//ifp_dbg("is_file returned %d for %s / %s", i, d, f);
	if (i == 1) {
		return IFP_FILE;
	} else if (i == 0) {
		return 0;
	} else if (i < 0) {
		ifp_err_i(i, "dir checking failed");
		return i;
	} else {
		ifp_err_i(i, "unexpected result checking file");
		return -1;
	}

	return i;
}
IFP_EXPORT(ifp_exists);

static int _empty_dir_checker(void * context, int type,
	const char * name, int size)
{
	int * pn = context;
	(*pn) += 1;

	//return early
	return 1;
}

//NOTE: this is should be removed
//returns -ENOTEMPTY
static int check_dir_is_empty(struct ifp_device * dev, const char * dirname)
{
	int i = 0;
	int n = 0;

	i = ifp_dir_open(dev, dirname);
	ifp_err_expect(i, i == -ENOENT, err, "dir_open failed");

	i = _ifp_list_dirs(dev, dirname, strlen(dirname), IFP_FILE|IFP_DIR,
		_empty_dir_checker, &n);
	ifp_err_jump(i, err, "ifp_list_dirs failed");
	IFP_BUG_ON(n < 0);

	i = ifp_dir_close(dev);
	ifp_err_jump(i, err, "dir_close failed");

	if (n != 0) {
		i = -ENOTEMPTY;
	}

err:
	return i;
}

/** \brief Deletes the directory f.
 *
 * Returns 0 on success or:
 * 	-ENOENT
 * 	-ENOTEMPTY
 * 	-EACCES
 */
int ifp_rmdir(struct ifp_device * dev, const char * d)
{
	int i = 0;

	if (strcmp(d, "\\VOICE") == 0 || strcmp(d, "\\RECORD") == 0) {
		//ifp_err("Can't delete ifp:\\%s", d);
		return -EACCES;
	}

	//will catch if the directory doesn't exist.
	i = check_dir_is_empty(dev, d);
	ifp_err_expect(i, i==-ENOENT || i==-ENOTEMPTY, err, "error checking dir");

	i = ifp_rmdir_nocheck(dev, d);
	ifp_err_jump(i, err, "error removing dir");

err:
	return i;
}
IFP_EXPORT(ifp_rmdir);

