#   Copyright (C) 2013 Canonical Ltd.
#
#   Author: Scott Moser <scott.moser@canonical.com>
#
#   Simplestreams is free software: you can redistribute it and/or modify it
#   under the terms of the GNU Affero General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or (at your
#   option) any later version.
#
#   Simplestreams is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
#   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
#   License for more details.
#
#   You should have received a copy of the GNU Affero General Public License
#   along with Simplestreams.  If not, see <http://www.gnu.org/licenses/>.

import errno
import tempfile

import boto3

import simplestreams.objectstores as objectstores
import simplestreams.contentsource as cs


class S3ObjectStore(objectstores.ObjectStore):

    def __init__(self, prefix):
        # expect 's3://bucket/path_prefix'
        self.prefix = prefix
        if prefix.startswith("s3://"):
            path = prefix[5:]
        else:
            path = prefix

        (self.bucketname, self.path_prefix) = path.split("/", 1)
        self._client = boto3.client("s3")

    def insert(self, path, reader, checksums=None, mutable=True, size=None):
        # store content from reader.read() into path, expecting result checksum
        try:
            tfile = tempfile.TemporaryFile()
            with reader(path) as rfp:
                while True:
                    buf = rfp.read(self.read_size)
                    tfile.write(buf)
                    if len(buf) != self.read_size:
                        break
            tfile.seek(0)
            self.insert_content(
                path, tfile, checksums=checksums, mutable=mutable
            )
        finally:
            tfile.close()

    def insert_content(self, path, content, checksums=None, mutable=True):
        self._client.put_object(
            Body=content, Bucket=self.bucketname, Key=self.path_prefix + path
        )

    def remove(self, path):
        # remove path from store
        self._client.delete_object(
            Bucket=self.bucketname, Key=self.path_prefix + path
        )

    def source(self, path):
        # essentially return an 'open(path, r)'
        try:
            obj_resp = self._client.get_object(
                Bucket=self.bucketname, Key=self.path_prefix + path
            )
        except self._client.exceptions.NoSuchKey:
            fd = None
        else:
            fd = obj_resp.get("Body")
        if not fd:
            myerr = IOError("Unable to open %s" % path)
            myerr.errno = errno.ENOENT
            raise myerr

        return cs.FdContentSource(fd=fd, url=self.path_prefix + path)

    def exists_with_checksum(self, path, checksums=None):
        try:
            obj_resp = self._client.get_object(
                Bucket=self.bucketname, Key=self.path_prefix + path
            )
        except self._client.exceptions.NoSuchKey:
            return False

        if "md5" in checksums:
            md5 = obj_resp["ResponseMetadata"]["HTTPHeaders"]["etag"].replace(
                '"', ""
            )
            return checksums["md5"] == md5

        return False


# vi: ts=4 expandtab
