#!/bin/sh
set -eux

download_openssl () {
    if [ ! -f "download-cache/openssl-${OPENSSL_VERSION}.tar.gz" ]; then
        wget -P download-cache/ \
            "https://www.openssl.org/source/openssl-${OPENSSL_VERSION}.tar.gz"
    fi
}

# recent openssl versions support parallel builds and skipping the docs,
# while older ones require to build everything sequentially.
build_openssl_linux () {
    (
        cd "openssl-${OPENSSL_VERSION}/"
        ./config shared --prefix="${HOME}/opt" --openssldir="${HOME}/opt" -DPURIFY
        if [ -z "${OPENSSL_VERSION##1.*}" ]; then
            make all
        else
            make -j$(nproc) build_sw
        fi
        make install_sw
    )
}

build_openssl_osx () {
    (
        cd "openssl-${OPENSSL_VERSION}/"
        ./Configure darwin64-x86_64-cc shared \
            --prefix="${HOME}/opt" --openssldir="${HOME}/opt" -DPURIFY
        make depend build_sw install_sw
    )
}

build_openssl () {
    if [ "$(cat ${HOME}/opt/.openssl-version)" != "${OPENSSL_VERSION}" ]; then
        tar zxf "download-cache/openssl-${OPENSSL_VERSION}.tar.gz"
	case `uname` in
		'Darwin')
			build_openssl_osx
			;;
		'Linux')
			build_openssl_linux
			;;
	esac
        echo "${OPENSSL_VERSION}" > "${HOME}/opt/.openssl-version"
    fi
}

download_libressl () {
    if [ ! -f "download-cache/libressl-${LIBRESSL_VERSION}.tar.gz" ]; then
        wget -P download-cache/ \
	    "https://ftp.openbsd.org/pub/OpenBSD/LibreSSL/libressl-${LIBRESSL_VERSION}.tar.gz"
    fi
}

build_libressl () {
    if [ "$(cat ${HOME}/opt/.libressl-version)" != "${LIBRESSL_VERSION}" ]; then
        tar zxf "download-cache/libressl-${LIBRESSL_VERSION}.tar.gz"
        (
           cd "libressl-${LIBRESSL_VERSION}/"
           ./configure --prefix="${HOME}/opt"
            make all install
        )
        echo "${LIBRESSL_VERSION}" > "${HOME}/opt/.libressl-version"
    fi
}

download_boringssl () {
    if [ ! -d "download-cache/boringssl" ]; then
        git clone --depth=1 https://boringssl.googlesource.com/boringssl download-cache/boringssl
    else
       (
        cd download-cache/boringssl
        git pull
       )
    fi
}

if [ ! -z ${LIBRESSL_VERSION+x} ]; then
	download_libressl
	build_libressl
fi

if [ ! -z ${OPENSSL_VERSION+x} ]; then
	download_openssl
	build_openssl
fi

if [ ! -z ${BORINGSSL+x} ]; then
	(
        download_boringssl
	cd download-cache/boringssl
        if [ -d build ]; then rm -rf build; fi
	mkdir build
	cd build
	cmake -DCMAKE_BUILD_TYPE=release -DBUILD_SHARED_LIBS=1 ..
	make

	rm -rf ${SSL_LIB} || exit 0
	rm -rf ${SSL_INC} || exit 0

	mkdir -p ${SSL_LIB}
	cp crypto/libcrypto.so ssl/libssl.so ${SSL_LIB}

	mkdir -p ${SSL_INC}
	cp -r ../include/* ${SSL_INC}
	)
fi

