﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the MIT license.  See License.txt in the project root for license information.

using System;
using System.Text;
using System.Threading.Tasks;
using Microsoft.CodeAnalysis.ResxSourceGenerator.CSharp;
using Microsoft.CodeAnalysis.ResxSourceGenerator.VisualBasic;
using Microsoft.CodeAnalysis.Text;
using Xunit;
using CSharpLanguageVersion = Microsoft.CodeAnalysis.CSharp.LanguageVersion;
using VerifyCS = Microsoft.CodeAnalysis.ResxSourceGenerator.Test.CSharpSourceGeneratorVerifier<Microsoft.CodeAnalysis.ResxSourceGenerator.CSharp.CSharpResxGenerator>;
using VerifyVB = Microsoft.CodeAnalysis.ResxSourceGenerator.Test.VisualBasicSourceGeneratorVerifier<Microsoft.CodeAnalysis.ResxSourceGenerator.VisualBasic.VisualBasicResxGenerator>;

namespace Microsoft.CodeAnalysis.ResxSourceGenerator.Test
{
    public class ResxGeneratorTests
    {
        private const string ResxHeader = @"<?xml version=""1.0"" encoding=""utf-8""?>
<root>
  <xsd:schema id=""root"" xmlns="""" xmlns:xsd=""http://www.w3.org/2001/XMLSchema"" xmlns:msdata=""urn:schemas-microsoft-com:xml-msdata"">
    <xsd:import namespace=""http://www.w3.org/XML/1998/namespace"" />
    <xsd:element name=""root"" msdata:IsDataSet=""true"">
      <xsd:complexType>
        <xsd:choice maxOccurs=""unbounded"">
          <xsd:element name=""metadata"">
            <xsd:complexType>
              <xsd:sequence>
                <xsd:element name=""value"" type=""xsd:string"" minOccurs=""0"" />
              </xsd:sequence>
              <xsd:attribute name=""name"" use=""required"" type=""xsd:string"" />
              <xsd:attribute name=""type"" type=""xsd:string"" />
              <xsd:attribute name=""mimetype"" type=""xsd:string"" />
              <xsd:attribute ref=""xml:space"" />
            </xsd:complexType>
          </xsd:element>
          <xsd:element name=""assembly"">
            <xsd:complexType>
              <xsd:attribute name=""alias"" type=""xsd:string"" />
              <xsd:attribute name=""name"" type=""xsd:string"" />
            </xsd:complexType>
          </xsd:element>
          <xsd:element name=""data"">
            <xsd:complexType>
              <xsd:sequence>
                <xsd:element name=""value"" type=""xsd:string"" minOccurs=""0"" msdata:Ordinal=""1"" />
                <xsd:element name=""comment"" type=""xsd:string"" minOccurs=""0"" msdata:Ordinal=""2"" />
              </xsd:sequence>
              <xsd:attribute name=""name"" type=""xsd:string"" use=""required"" msdata:Ordinal=""1"" />
              <xsd:attribute name=""type"" type=""xsd:string"" msdata:Ordinal=""3"" />
              <xsd:attribute name=""mimetype"" type=""xsd:string"" msdata:Ordinal=""4"" />
              <xsd:attribute ref=""xml:space"" />
            </xsd:complexType>
          </xsd:element>
          <xsd:element name=""resheader"">
            <xsd:complexType>
              <xsd:sequence>
                <xsd:element name=""value"" type=""xsd:string"" minOccurs=""0"" msdata:Ordinal=""1"" />
              </xsd:sequence>
              <xsd:attribute name=""name"" type=""xsd:string"" use=""required"" />
            </xsd:complexType>
          </xsd:element>
        </xsd:choice>
      </xsd:complexType>
    </xsd:element>
  </xsd:schema>
  <resheader name=""resmimetype"">
    <value>text/microsoft-resx</value>
  </resheader>
  <resheader name=""version"">
    <value>2.0</value>
  </resheader>
  <resheader name=""reader"">
    <value>System.Resources.ResXResourceReader, System.Windows.Forms, Version=4.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089</value>
  </resheader>
  <resheader name=""writer"">
    <value>System.Resources.ResXResourceWriter, System.Windows.Forms, Version=4.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089</value>
  </resheader>
";
        private const string ResxFooter = @"
</root>";

        [Theory]
        [InlineData(CSharpLanguageVersion.CSharp5, Skip = "Expression-bodied members are not supported in C# 5")]
        [InlineData(CSharpLanguageVersion.CSharp6)]
        [InlineData(CSharpLanguageVersion.CSharp7)]
        [InlineData(CSharpLanguageVersion.CSharp8)]
        [InlineData(CSharpLanguageVersion.CSharp9)]
        public async Task SingleString_DefaultCSharpAsync(CSharpLanguageVersion languageVersion)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;
            var generated = languageVersion switch
            {
                CSharpLanguageVersion.CSharp5 or CSharpLanguageVersion.CSharp6 or CSharpLanguageVersion.CSharp7 => @"// <auto-generated/>


using System.Reflection;


namespace TestProject
{
    internal static partial class Resources
    {
        private static global::System.Resources.ResourceManager s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof(Resources)));
        internal static global::System.Globalization.CultureInfo Culture { get; set; }
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string GetResourceString(string resourceKey, string defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;
        /// <summary>value</summary>
        internal static string @Name => GetResourceString(""Name"");

    }
}
",
                CSharpLanguageVersion.CSharp8 or CSharpLanguageVersion.CSharp9 => @"// <auto-generated/>

#nullable enable
using System.Reflection;


namespace TestProject
{
    internal static partial class Resources
    {
        private static global::System.Resources.ResourceManager? s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof(Resources)));
        internal static global::System.Globalization.CultureInfo? Culture { get; set; }
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;
        /// <summary>value</summary>
        internal static string? @Name => GetResourceString(""Name"");

    }
}
",
                _ => throw new NotSupportedException(),
            };

            await new VerifyCS.Test
            {
                LanguageVersion = languageVersion,
                TestState =
                {
                    Sources = { "" },
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    GeneratedSources =
                    {
                        (typeof(CSharpResxGenerator), "TestProject.Resources.Designer.cs", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Fact]
        public async Task SingleString_DefaultVisualBasicAsync()
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;
            var generated = @"' <auto-generated/>

Imports System.Reflection


Namespace Global.TestProject
    Friend Partial Class Resources
        Private Sub New
        End Sub
        
        Private Shared s_resourceManager As Global.System.Resources.ResourceManager
        Friend Shared ReadOnly Property ResourceManager As Global.System.Resources.ResourceManager
            Get
                If s_resourceManager Is Nothing Then
                    s_resourceManager = New Global.System.Resources.ResourceManager(GetType(Resources))
                End If
                Return s_resourceManager
            End Get
        End Property
        Friend Shared Property Culture As Global.System.Globalization.CultureInfo
        <Global.System.Runtime.CompilerServices.MethodImpl(Global.System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)>
        Friend Shared Function GetResourceString(ByVal resourceKey As String, Optional ByVal defaultValue As String = Nothing) As String
            Return ResourceManager.GetString(resourceKey, Culture)
        End Function
        ''' <summary>value</summary>
        Friend Shared ReadOnly Property [Name] As String
          Get
            Return GetResourceString(""Name"")
          End Get
        End Property

    End Class
End Namespace
";

            await new VerifyVB.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    GeneratedSources =
                    {
                        (typeof(VisualBasicResxGenerator), "TestProject.Resources.Designer.vb", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Fact]
        public async Task SingleString_DisableCodeGenAsync()
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            await new VerifyCS.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", @"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.GenerateSource = false
"),
                    },
                },
            }.RunAsync();

            await new VerifyVB.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", @"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.GenerateSource = false
"),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [InlineData("", Skip = "Empty root namespaces are not supported")]
        [InlineData("NS")]
        [InlineData("NS1.NS2")]
        public async Task SingleString_RootNamespaceCSharpAsync(string rootNamespace)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;
            var generated = $@"// <auto-generated/>

#nullable enable
using System.Reflection;


namespace {rootNamespace}
{{
    internal static partial class Resources
    {{
        private static global::System.Resources.ResourceManager? s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof(Resources)));
        internal static global::System.Globalization.CultureInfo? Culture {{ get; set; }}
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;
        /// <summary>value</summary>
        internal static string? @Name => GetResourceString(""Name"");

    }}
}}
";

            await new VerifyCS.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

build_property.RootNamespace = {rootNamespace}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(CSharpResxGenerator), $"{rootNamespace}.Resources.Designer.cs", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [InlineData("", Skip = "Empty root namespaces are not supported")]
        [InlineData("NS")]
        [InlineData("NS1.NS2")]
        public async Task SingleString_RootNamespaceVisualBasicAsync(string rootNamespace)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;
            var generated = $@"' <auto-generated/>

Imports System.Reflection


Namespace Global.{rootNamespace}
    Friend Partial Class Resources
        Private Sub New
        End Sub
        
        Private Shared s_resourceManager As Global.System.Resources.ResourceManager
        Friend Shared ReadOnly Property ResourceManager As Global.System.Resources.ResourceManager
            Get
                If s_resourceManager Is Nothing Then
                    s_resourceManager = New Global.System.Resources.ResourceManager(GetType(Resources))
                End If
                Return s_resourceManager
            End Get
        End Property
        Friend Shared Property Culture As Global.System.Globalization.CultureInfo
        <Global.System.Runtime.CompilerServices.MethodImpl(Global.System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)>
        Friend Shared Function GetResourceString(ByVal resourceKey As String, Optional ByVal defaultValue As String = Nothing) As String
            Return ResourceManager.GetString(resourceKey, Culture)
        End Function
        ''' <summary>value</summary>
        Friend Shared ReadOnly Property [Name] As String
          Get
            Return GetResourceString(""Name"")
          End Get
        End Property

    End Class
End Namespace
";

            await new VerifyVB.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

build_property.RootNamespace = {rootNamespace}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(VisualBasicResxGenerator), $"{rootNamespace}.Resources.Designer.vb", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [InlineData("")]
        [InlineData("NS")]
        [InlineData("NS1.NS2")]
        public async Task SingleString_RelativeDirCSharpAsync(string relativeDir)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var (namespaceName, typeName) = relativeDir switch
            {
                "" => ("TestProject", "Resources"),
                "NS" => ("TestProject", "NSResources"),
                "NS1.NS2" => ("TestProject.NS1", "NS2Resources"),
                _ => throw new NotSupportedException(),
            };

            var generated = $@"// <auto-generated/>

#nullable enable
using System.Reflection;


namespace {namespaceName}
{{
    internal static partial class {typeName}
    {{
        private static global::System.Resources.ResourceManager? s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof({typeName})));
        internal static global::System.Globalization.CultureInfo? Culture {{ get; set; }}
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;
        /// <summary>value</summary>
        internal static string? @Name => GetResourceString(""Name"");

    }}
}}
";

            await new VerifyCS.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.RelativeDir = {relativeDir}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(CSharpResxGenerator), $"{namespaceName}.{typeName}.Designer.cs", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [InlineData("")]
        [InlineData("NS")]
        [InlineData("NS1.NS2")]
        public async Task SingleString_RelativeDirVisualBasicAsync(string relativeDir)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var (namespaceName, typeName) = relativeDir switch
            {
                "" => ("TestProject", "Resources"),
                "NS" => ("TestProject", "NSResources"),
                "NS1.NS2" => ("TestProject.NS1", "NS2Resources"),
                _ => throw new NotSupportedException(),
            };

            var generated = $@"' <auto-generated/>

Imports System.Reflection


Namespace Global.{namespaceName}
    Friend Partial Class {typeName}
        Private Sub New
        End Sub
        
        Private Shared s_resourceManager As Global.System.Resources.ResourceManager
        Friend Shared ReadOnly Property ResourceManager As Global.System.Resources.ResourceManager
            Get
                If s_resourceManager Is Nothing Then
                    s_resourceManager = New Global.System.Resources.ResourceManager(GetType({typeName}))
                End If
                Return s_resourceManager
            End Get
        End Property
        Friend Shared Property Culture As Global.System.Globalization.CultureInfo
        <Global.System.Runtime.CompilerServices.MethodImpl(Global.System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)>
        Friend Shared Function GetResourceString(ByVal resourceKey As String, Optional ByVal defaultValue As String = Nothing) As String
            Return ResourceManager.GetString(resourceKey, Culture)
        End Function
        ''' <summary>value</summary>
        Friend Shared ReadOnly Property [Name] As String
          Get
            Return GetResourceString(""Name"")
          End Get
        End Property

    End Class
End Namespace
";

            await new VerifyVB.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.RelativeDir = {relativeDir}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(VisualBasicResxGenerator), $"{namespaceName}.{typeName}.Designer.vb", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [CombinatorialData]
        public async Task SingleString_OmitGetResourceStringCSharpAsync(bool omitGetResourceString)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var getResourceString = omitGetResourceString
                ? ""
                : @"        internal static global::System.Globalization.CultureInfo? Culture { get; set; }
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;";

            var generated = $@"// <auto-generated/>

#nullable enable
using System.Reflection;


namespace TestProject
{{
    internal static partial class Resources
    {{
        private static global::System.Resources.ResourceManager? s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof(Resources)));
{getResourceString}
        /// <summary>value</summary>
        internal static string? @Name => GetResourceString(""Name"");

    }}
}}
";

            var customGetResourceString = @"#nullable enable

namespace TestProject
{
    internal static partial class Resources
    {
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) => throw null!;
    }
}
";

            await new VerifyCS.Test
            {
                TestState =
                {
                    Sources = { omitGetResourceString ? customGetResourceString : "" },
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.OmitGetResourceString = {(omitGetResourceString ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(CSharpResxGenerator), "TestProject.Resources.Designer.cs", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [CombinatorialData]
        public async Task SingleString_OmitGetResourceStringVisualBasicAsync(bool omitGetResourceString)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var getResourceString = omitGetResourceString
                ? ""
                : @"        Friend Shared Property Culture As Global.System.Globalization.CultureInfo
        <Global.System.Runtime.CompilerServices.MethodImpl(Global.System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)>
        Friend Shared Function GetResourceString(ByVal resourceKey As String, Optional ByVal defaultValue As String = Nothing) As String
            Return ResourceManager.GetString(resourceKey, Culture)
        End Function";

            var generated = $@"' <auto-generated/>

Imports System.Reflection


Namespace Global.TestProject
    Friend Partial Class Resources
        Private Sub New
        End Sub
        
        Private Shared s_resourceManager As Global.System.Resources.ResourceManager
        Friend Shared ReadOnly Property ResourceManager As Global.System.Resources.ResourceManager
            Get
                If s_resourceManager Is Nothing Then
                    s_resourceManager = New Global.System.Resources.ResourceManager(GetType(Resources))
                End If
                Return s_resourceManager
            End Get
        End Property
{getResourceString}
        ''' <summary>value</summary>
        Friend Shared ReadOnly Property [Name] As String
          Get
            Return GetResourceString(""Name"")
          End Get
        End Property

    End Class
End Namespace
";

            var customGetResourceString = @"
Namespace Global.TestProject
    Friend Partial Class Resources
        Friend Shared Function GetResourceString(resourceKey As String, Optional defaultValue As String = Nothing)
            Return """"
        End Function
    End Class
End Namespace
";

            await new VerifyVB.Test
            {
                TestState =
                {
                    Sources = { omitGetResourceString ? customGetResourceString : "" },
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.OmitGetResourceString = {(omitGetResourceString ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(VisualBasicResxGenerator), "TestProject.Resources.Designer.vb", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [CombinatorialData]
        public async Task SingleString_AsConstantsCSharpAsync(bool asConstants)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var generatedName = asConstants
                ? @"internal const string @Name = ""Name"";"
                : @"internal static string? @Name => GetResourceString(""Name"");";

            var generated = $@"// <auto-generated/>

#nullable enable
using System.Reflection;


namespace TestProject
{{
    internal static partial class Resources
    {{
        private static global::System.Resources.ResourceManager? s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof(Resources)));
        internal static global::System.Globalization.CultureInfo? Culture {{ get; set; }}
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;
        /// <summary>value</summary>
        {generatedName}

    }}
}}
";

            await new VerifyCS.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.AsConstants = {(asConstants ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(CSharpResxGenerator), "TestProject.Resources.Designer.cs", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [CombinatorialData]
        public async Task SingleString_AsConstantsVisualBasicAsync(bool asConstants)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var generatedName = asConstants
                ? @"Friend Const [Name] As String = ""Name"""
                : @"Friend Shared ReadOnly Property [Name] As String
          Get
            Return GetResourceString(""Name"")
          End Get
        End Property";

            var generated = $@"' <auto-generated/>

Imports System.Reflection


Namespace Global.TestProject
    Friend Partial Class Resources
        Private Sub New
        End Sub
        
        Private Shared s_resourceManager As Global.System.Resources.ResourceManager
        Friend Shared ReadOnly Property ResourceManager As Global.System.Resources.ResourceManager
            Get
                If s_resourceManager Is Nothing Then
                    s_resourceManager = New Global.System.Resources.ResourceManager(GetType(Resources))
                End If
                Return s_resourceManager
            End Get
        End Property
        Friend Shared Property Culture As Global.System.Globalization.CultureInfo
        <Global.System.Runtime.CompilerServices.MethodImpl(Global.System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)>
        Friend Shared Function GetResourceString(ByVal resourceKey As String, Optional ByVal defaultValue As String = Nothing) As String
            Return ResourceManager.GetString(resourceKey, Culture)
        End Function
        ''' <summary>value</summary>
        {generatedName}

    End Class
End Namespace
";

            await new VerifyVB.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.AsConstants = {(asConstants ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(VisualBasicResxGenerator), "TestProject.Resources.Designer.vb", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [CombinatorialData]
        public async Task SingleString_IncludeDefaultValuesCSharpAsync(bool includeDefaultValues)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var includedDefaultValue = includeDefaultValues
                ? @", @""value"""
                : "";

            var generated = $@"// <auto-generated/>

#nullable enable
using System.Reflection;


namespace TestProject
{{
    internal static partial class Resources
    {{
        private static global::System.Resources.ResourceManager? s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof(Resources)));
        internal static global::System.Globalization.CultureInfo? Culture {{ get; set; }}
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;
        /// <summary>value</summary>
        internal static {(includeDefaultValues ? "string" : "string?")} @Name => GetResourceString(""Name""{includedDefaultValue});

    }}
}}
";

            await new VerifyCS.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.IncludeDefaultValues = {(includeDefaultValues ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(CSharpResxGenerator), "TestProject.Resources.Designer.cs", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [CombinatorialData]
        public async Task SingleString_IncludeDefaultValuesVisualBasicAsync(bool includeDefaultValues)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var includedDefaultValue = includeDefaultValues
                ? @", ""value"""
                : "";

            var generated = $@"' <auto-generated/>

Imports System.Reflection


Namespace Global.TestProject
    Friend Partial Class Resources
        Private Sub New
        End Sub
        
        Private Shared s_resourceManager As Global.System.Resources.ResourceManager
        Friend Shared ReadOnly Property ResourceManager As Global.System.Resources.ResourceManager
            Get
                If s_resourceManager Is Nothing Then
                    s_resourceManager = New Global.System.Resources.ResourceManager(GetType(Resources))
                End If
                Return s_resourceManager
            End Get
        End Property
        Friend Shared Property Culture As Global.System.Globalization.CultureInfo
        <Global.System.Runtime.CompilerServices.MethodImpl(Global.System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)>
        Friend Shared Function GetResourceString(ByVal resourceKey As String, Optional ByVal defaultValue As String = Nothing) As String
            Return ResourceManager.GetString(resourceKey, Culture)
        End Function
        ''' <summary>value</summary>
        Friend Shared ReadOnly Property [Name] As String
          Get
            Return GetResourceString(""Name""{includedDefaultValue})
          End Get
        End Property

    End Class
End Namespace
";

            await new VerifyVB.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.IncludeDefaultValues = {(includeDefaultValues ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(VisualBasicResxGenerator), "TestProject.Resources.Designer.vb", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [CombinatorialData]
        public async Task SingleString_EmitFormatMethodsCSharpAsync(
            [CombinatorialValues("0", "x", "replacement")] string placeholder,
            bool emitFormatMethods)
        {
            var code = ResxHeader
                + $@"  <data name=""Name"" xml:space=""preserve"">
    <value>value {{{placeholder}}}</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;

            var getResourceStringMethod = @"
        private static string GetResourceString(string resourceKey, string[]? formatterNames)
        {
           var value = GetResourceString(resourceKey) ?? """";
           if (formatterNames != null)
           {
               for (var i = 0; i < formatterNames.Length; i++)
               {
                   value = value.Replace(""{"" + formatterNames[i] + ""}"", ""{"" + i + ""}"");
               }
           }
           return value;
        }

";

            var replacementArgs = placeholder switch
            {
                "0" => "",
                _ => $@", new[] {{ ""{placeholder}"" }}",
            };
            var suffix = placeholder switch
            {
                "0" => @" ?? """"",
                _ => "",
            };
            var parameter = placeholder switch
            {
                "0" => "p0",
                _ => placeholder,
            };
            var formatMethod = $@"        /// <summary>value {{{placeholder}}}</summary>
        internal static string FormatName(object {parameter})
           => string.Format(Culture, GetResourceString(""Name""{replacementArgs}){suffix}, {parameter});

";

            var generated = $@"// <auto-generated/>

#nullable enable
using System.Reflection;


namespace TestProject
{{
    internal static partial class Resources
    {{
        private static global::System.Resources.ResourceManager? s_resourceManager;
        internal static global::System.Resources.ResourceManager ResourceManager => s_resourceManager ?? (s_resourceManager = new global::System.Resources.ResourceManager(typeof(Resources)));
        internal static global::System.Globalization.CultureInfo? Culture {{ get; set; }}
        [global::System.Runtime.CompilerServices.MethodImpl(global::System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)]
        [return: global::System.Diagnostics.CodeAnalysis.NotNullIfNotNull(""defaultValue"")]
        internal static string? GetResourceString(string resourceKey, string? defaultValue = null) =>  ResourceManager.GetString(resourceKey, Culture) ?? defaultValue;
{(emitFormatMethods ? getResourceStringMethod : "")}        /// <summary>value {{{placeholder}}}</summary>
        internal static string? @Name => GetResourceString(""Name"");
{(emitFormatMethods ? formatMethod : "")}
    }}
}}
";

            await new VerifyCS.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.EmitFormatMethods = {(emitFormatMethods ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(CSharpResxGenerator), "TestProject.Resources.Designer.cs", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }

        [Theory]
        [InlineData(true, Skip = "Not yet supported")]
        [InlineData(false)]
        public async Task SingleString_EmitFormatMethodsVisualBasicAsync(bool emitFormatMethods)
        {
            var code = ResxHeader
                + @"  <data name=""Name"" xml:space=""preserve"">
    <value>value</value>
    <comment>comment</comment>
  </data>"
                + ResxFooter;
            var generated = @"' <auto-generated/>

Imports System.Reflection


Namespace Global.TestProject
    Friend Partial Class Resources
        Private Sub New
        End Sub
        
        Private Shared s_resourceManager As Global.System.Resources.ResourceManager
        Friend Shared ReadOnly Property ResourceManager As Global.System.Resources.ResourceManager
            Get
                If s_resourceManager Is Nothing Then
                    s_resourceManager = New Global.System.Resources.ResourceManager(GetType(Resources))
                End If
                Return s_resourceManager
            End Get
        End Property
        Friend Shared Property Culture As Global.System.Globalization.CultureInfo
        <Global.System.Runtime.CompilerServices.MethodImpl(Global.System.Runtime.CompilerServices.MethodImplOptions.AggressiveInlining)>
        Friend Shared Function GetResourceString(ByVal resourceKey As String, Optional ByVal defaultValue As String = Nothing) As String
            Return ResourceManager.GetString(resourceKey, Culture)
        End Function
        ''' <summary>value</summary>
        Friend Shared ReadOnly Property [Name] As String
          Get
            Return GetResourceString(""Name"")
          End Get
        End Property

    End Class
End Namespace
";

            await new VerifyVB.Test
            {
                TestState =
                {
                    AdditionalFiles = { ("/0/Resources.resx", code) },
                    AnalyzerConfigFiles =
                    {
                        ("/.globalconfig", $@"
is_global = true

[/0/Resources.resx]
build_metadata.AdditionalFiles.EmitFormatMethods = {(emitFormatMethods ? "true" : "false")}
"),
                    },
                    GeneratedSources =
                    {
                        (typeof(VisualBasicResxGenerator), "TestProject.Resources.Designer.vb", SourceText.From(generated, Encoding.UTF8, SourceHashAlgorithm.Sha256)),
                    },
                },
            }.RunAsync();
        }
    }
}
