"""
Manages Docker container status checking for selected tools.
"""

import subprocess
from PySide6.QtGui import QIcon
from PySide6.QtCore import QSize, QTimer
from PySide6.QtWidgets import QWidget, QLabel
from src.path_resolver import PathResolver


class ContainerStatusController(QWidget):
    def __init__(self, docker_status_label: QLabel, docker_status_icon: QLabel, parent=None) -> None:
        super().__init__(parent)
        self.docker_status_label = docker_status_label
        self.docker_status_icon = docker_status_icon
        self.timer = QTimer(self)
        self.init_docker_status_update()

    def init_docker_status_update(self) -> None:
        self.timer.timeout.connect(self.check_docker_status)
        self.timer.start(5000)
        self.check_docker_status()

    def check_docker_status(self) -> None:
        running_icon = QIcon(PathResolver.resource_path('resources/assets/docker_working.png'))
        not_running_icon = QIcon(PathResolver.resource_path('resources/assets/docker_not_working.png'))
        icon_size = QSize(24, 24)
        try:
            subprocess.run(['docker', 'info'],
                           stdout=subprocess.PIPE,
                           stderr=subprocess.PIPE,
                           check=True)
            self.docker_status_label.setText("Docker is running")
            self.docker_status_icon.setPixmap(running_icon.pixmap(icon_size))
        except subprocess.CalledProcessError:
            self.docker_status_label.setText("Docker is not running")
            self.docker_status_icon.setPixmap(not_running_icon.pixmap(icon_size))
