/**********************************************************************
 * Copyright (c) by Heiner Jostkleigrewe
 * This program is free software: you can redistribute it and/or modify it under the terms of the 
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,  but WITHOUT ANY WARRANTY; without 
 *  even the implied warranty of  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See 
 *  the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program.  If not, 
 * see <http://www.gnu.org/licenses/>.
 * 
 * heiner@jverein.de
 * www.jverein.de
 **********************************************************************/
package de.jost_net.JVerein.server;

import java.rmi.RemoteException;
import java.util.Date;

import de.jost_net.JVerein.Einstellungen;
import de.jost_net.JVerein.keys.AfaMode;
import de.jost_net.JVerein.rmi.Buchungsart;
import de.jost_net.JVerein.rmi.Buchungsklasse;
import de.jost_net.JVerein.rmi.Konto;
import de.jost_net.JVerein.util.Geschaeftsjahr;
import de.willuhn.datasource.db.AbstractDBObject;
import de.willuhn.datasource.rmi.DBIterator;
import de.willuhn.logging.Logger;
import de.willuhn.util.ApplicationException;

public class KontoImpl extends AbstractDBObject implements Konto
{

  private static final long serialVersionUID = 1L;

  public KontoImpl() throws RemoteException
  {
    super();
  }

  @Override
  protected String getTableName()
  {
    return "konto";
  }

  @Override
  public String getPrimaryAttribute()
  {
    return "id";
  }

  @Override
  protected void deleteCheck()
  {
    //
  }

  @Override
  protected void insertCheck() throws ApplicationException
  {
    try
    {
      plausi();
      DBIterator<Konto> it = Einstellungen.getDBService()
          .createList(Konto.class);
      it.addFilter("nummer = ?", new Object[] { getNummer() });
      if (it.size() > 0)
      {
        throw new ApplicationException("Konto existiert bereits");
      }
    }
    catch (RemoteException e)
    {
      Logger.error("insert check of konto failed", e);
      throw new ApplicationException(
          "Konto kann nicht gespeichert werden. Siehe system log");
    }
  }

  @Override
  protected void updateCheck() throws ApplicationException
  {
    plausi();
  }

  private void plausi() throws ApplicationException
  {
    try
    {
      if (getBezeichnung() == null || getBezeichnung().length() == 0)
      {
        throw new ApplicationException("Bitte Bezeichnung eingeben");
      }
      if (getBezeichnung().length() > 255)
      {
        throw new ApplicationException(
            "Maximale Lnge der Bezeichnung: 255 Zeichen");
      }
      if (getNummer() == null || getNummer().length() == 0)
      {
        throw new ApplicationException("Bitte Nummer eingeben");
      }
      if (getAnlagenkonto())
      {
        if (getBetrag() != null && getBetrag() < 0.0)
        {
          throw new ApplicationException("Wert der Anlage darf nicht negativ sein");
        }
        if (getAnlagenart() == null)
        {
          throw new ApplicationException("Bitte Anlagen Buchungsart eingeben");
        }
        if (getAfaart() == null)
        {
          throw new ApplicationException("Bitte Afa Buchungsart eingeben");
        }
        if (getAnlagenklasse() == null)
        {
          throw new ApplicationException("Bitte Anlagen Buchungsklasse eingeben");
        }
        if (getAfaMode() == null)
        {
          throw new ApplicationException("Bitte Afa Mode eingeben");
        }
        if (getAfaMode() != null && getAfaMode() == AfaMode.ANGEPASST)
        {
          if (getAfaStart() == null)
          {
            throw new ApplicationException("Bitte Afa Erstes Jahr eingeben");
          }
          if (getAfaDauer() == null)
          {
            throw new ApplicationException("Bitte Afa Folgejahre eingeben");
          }
        }
      }
    }
    catch (RemoteException e)
    {
      Logger.error("insert check of konto failed", e);
      throw new ApplicationException(
          "Konto kann nicht gespeichert werden. Siehe system log");
    }
  }

  @Override
  protected Class<?> getForeignObject(String arg0)
  {
    return null;
  }

  @Override
  public String getNummer() throws RemoteException
  {
    return (String) getAttribute("nummer");
  }

  @Override
  public void setNummer(String nummer) throws RemoteException
  {
    setAttribute("nummer", nummer);
  }

  @Override
  public String getBezeichnung() throws RemoteException
  {
    return (String) getAttribute("bezeichnung");
  }

  @Override
  public void setBezeichnung(String bezeichnung) throws RemoteException
  {
    setAttribute("bezeichnung", bezeichnung);
  }

  @Override
  public Date getEroeffnung() throws RemoteException
  {
    return (Date) getAttribute("eroeffnung");
  }

  @Override
  public void setEroeffnung(Date eroeffnungdatum) throws RemoteException
  {
    setAttribute("eroeffnung", eroeffnungdatum);
  }

  @Override
  public Date getAufloesung() throws RemoteException
  {
    return (Date) getAttribute("aufloesung");
  }

  @Override
  public void setAufloesung(Date aufloesungsdatum) throws RemoteException
  {
    setAttribute("aufloesung", aufloesungsdatum);
  }

  @Override
  public Integer getHibiscusId() throws RemoteException
  {
    return (Integer) getAttribute("hibiscusid");
  }

  @Override
  public void setHibiscusId(Integer id) throws RemoteException
  {
    setAttribute("hibiscusid", id);
  }

  @Override
  public DBIterator<Konto> getKontenEinesJahres(Geschaeftsjahr gj)
      throws RemoteException
  {
    DBIterator<Konto> konten = Einstellungen.getDBService()
        .createList(Konto.class);
    konten.addFilter("(eroeffnung is null or eroeffnung <= ?)",
        new Object[] { gj.getEndeGeschaeftsjahr() });
    konten.addFilter(
        "(aufloesung is null or aufloesung >= ? )",
        new Object[] { gj.getBeginnGeschaeftsjahr() });
    konten.setOrder("order by bezeichnung");
    return konten;
  }
  
  @Override
  public DBIterator<Konto> getKontenVonBis(Date von, Date bis)
      throws RemoteException
  {
    DBIterator<Konto> konten = Einstellungen.getDBService()
        .createList(Konto.class);
    konten.addFilter("(eroeffnung is null or eroeffnung <= ?)",
        new Object[] { bis });
    konten.addFilter(
        "(aufloesung is null or aufloesung >= ? )",
        new Object[] { von });
    konten.setOrder("order by bezeichnung");
    return konten;
  }

  @Override
  public void delete() throws RemoteException, ApplicationException
  {
    super.delete();
    Cache.get(Konto.class, false).remove(this); // Aus Cache loeschen
  }

  @Override
  public void store() throws RemoteException, ApplicationException
  {
    super.store();
    Cache.get(Konto.class, false).put(this); // Cache aktualisieren
  }
  
  @Override
  public Buchungsart getBuchungsart() throws RemoteException
  {
    Long l = (Long) super.getAttribute("buchungsart");
    if (l == null)
    {
      return null; // Keine Buchungsart zugeordnet
    }

    Cache cache = Cache.get(Buchungsart.class, true);
    return (Buchungsart) cache.get(l);
  }

  @Override
  public Long getBuchungsartId() throws RemoteException
  {
    return (Long) super.getAttribute("buchungsart");
  }

  @Override
  public void setBuchungsartId(Long buchungsartId) throws RemoteException
  {
    setAttribute("buchungsart", buchungsartId);
  }
  
  @Override
  public Boolean getAnlagenkonto() throws RemoteException
  {
    return Util.getBoolean(getAttribute("anlagenkonto"));
  }

  @Override
  public void setAnlagenkonto(Boolean anlagenkonto)
      throws RemoteException
  {
    setAttribute("anlagenkonto", Boolean.valueOf(anlagenkonto));
  }
  
  @Override
  public Buchungsart getAnlagenart() throws RemoteException
  {
    Long l = (Long) super.getAttribute("anlagenart");
    if (l == null)
    {
      return null; // Keine Buchungsart zugeordnet
    }

    Cache cache = Cache.get(Buchungsart.class, true);
    return (Buchungsart) cache.get(l);
  }

  @Override
  public Long getAnlagenartId() throws RemoteException
  {
    return (Long) super.getAttribute("anlagenart");
  }

  @Override
  public void setAnlagenartId(Long anlagenartId) throws RemoteException
  {
    setAttribute("anlagenart", anlagenartId);
  }
  
  @Override
  public Buchungsklasse getAnlagenklasse() throws RemoteException
  {
    Long l = (Long) super.getAttribute("anlagenklasse");
    if (l == null)
    {
      return null; // Keine Buchungsklasse zugeordnet
    }

    Cache cache = Cache.get(Buchungsklasse.class, true);
    return (Buchungsklasse) cache.get(l);
  }

  @Override
  public Long getAnlagenklasseId() throws RemoteException
  {
    return (Long) super.getAttribute("anlagenklasse");
  }

  @Override
  public void setAnlagenklasseId(Long anlagenklasseId) throws RemoteException
  {
    setAttribute("anlagenklasse", anlagenklasseId);
  }
  
  @Override
  public Buchungsart getAfaart() throws RemoteException
  {
    Long l = (Long) super.getAttribute("afaart");
    if (l == null)
    {
      return null; // Keine Buchungsart zugeordnet
    }

    Cache cache = Cache.get(Buchungsart.class, true);
    return (Buchungsart) cache.get(l);
  }

  @Override
  public Long getAfaartId() throws RemoteException
  {
    return (Long) super.getAttribute("afaart");
  }

  @Override
  public void setAfaartId(Long afaartId) throws RemoteException
  {
    setAttribute("afaart", afaartId);
  }
  
  @Override
  public Integer getNutzungsdauer() throws RemoteException
  {
    return (Integer) getAttribute("nutzungsdauer");
  }

  @Override
  public void setNutzungsdauer(Integer nutzungsdauer) throws RemoteException
  {
    setAttribute("nutzungsdauer", nutzungsdauer);
  }
  
  @Override
  public Double getBetrag() throws RemoteException
  {
    return (Double) getAttribute("betrag");
  }

  @Override
  public void setBetrag(Double d) throws RemoteException
  {
    setAttribute("betrag", d);
  }
  
  @Override
  public String getKommentar() throws RemoteException
  {
    return (String) getAttribute("kommentar");
  }

  @Override
  public void setKommentar(String kommentar) throws RemoteException
  {
    setAttribute("kommentar", kommentar);
  }
  

  @Override
  public Date getAnschaffung() throws RemoteException
  {
    return (Date) getAttribute("anschaffung");
  }

  @Override
  public void setAnschaffung(Date anschaffung) throws RemoteException
  {
    setAttribute("anschaffung", anschaffung);
  }
  
  @Override
  public Double getAfaStart() throws RemoteException
  {
    return (Double) getAttribute("afastart");
  }

  @Override
  public void setAfaStart(Double afastart) throws RemoteException
  {
    setAttribute("afastart", afastart);
  }
  
  @Override
  public Double getAfaDauer() throws RemoteException
  {
    return (Double) getAttribute("afadauer");
  }

  @Override
  public void setAfaDauer(Double afadauer) throws RemoteException
  {
    setAttribute("afadauer", afadauer);
  }
  
  @Override
  public Double getAfaRestwert() throws RemoteException
  {
    return (Double) getAttribute("afarestwert");
  }

  @Override
  public void setAfaRestwert(Double afarestwert) throws RemoteException
  {
    setAttribute("afarestwert", afarestwert);
  }
  
  @Override
  public Integer getAfaMode() throws RemoteException
  {
    return (Integer) getAttribute("afamode");
  }

  @Override
  public void setAfaMode(Integer afamode) throws RemoteException
  {
    setAttribute("afamode", afamode);
  }
  
  @Override
  public Object getAttribute(String fieldName) throws RemoteException
  {
    if ("buchungsart".equals(fieldName))
      return getBuchungsart();
    return super.getAttribute(fieldName);
  }
}
