/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.model;

import ghidra.framework.data.LinkHandler;
import ghidra.framework.data.LinkedGhidraFolder;
import ghidra.framework.model.DomainFile;
import ghidra.framework.model.DomainFileFilter;
import ghidra.framework.model.DomainFolder;
import ghidra.framework.model.DomainFolderFilter;
import ghidra.framework.model.LinkFileInfo;
import ghidra.framework.store.FileSystem;
import ghidra.util.InvalidNameException;
import ghidra.util.Msg;
import ghidra.util.ReadOnlyException;
import java.io.IOException;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.lang3.StringUtils;

public class ProjectDataUtils {
    public static Iterable<DomainFile> descendantFiles(DomainFolder folder) {
        return new DomainFileIterator(folder, DomainFileFilter.NON_LINKED_FILE_FILTER);
    }

    public static Iterable<DomainFile> descendantFiles(DomainFolder folder, DomainFileFilter filter) {
        return new DomainFileIterator(folder, filter);
    }

    public static Iterable<DomainFolder> descendantFolders(DomainFolder folder) {
        return ProjectDataUtils.descendantFolders(folder, true, true);
    }

    public static Iterable<DomainFolder> descendantFolders(DomainFolder folder, boolean ignoreFolderLinks, boolean ignoreExternalLinks) {
        return new DomainFolderIterator(folder, ignoreFolderLinks, ignoreExternalLinks);
    }

    public static DomainFolder createDomainFolderPath(DomainFolder currentFolder, String path) throws InvalidNameException, IOException {
        String[] pathElements;
        if (!currentFolder.isInWritableProject()) {
            throw new ReadOnlyException("Folder is read-only: " + String.valueOf(currentFolder));
        }
        if (StringUtils.isBlank((CharSequence)path)) {
            return currentFolder;
        }
        DomainFolder folder = currentFolder;
        for (String pathElement : pathElements = path.split(FileSystem.SEPARATOR)) {
            LinkFileInfo linkInfo;
            if (pathElement.isEmpty()) continue;
            DomainFolder subFolder = folder.getFolder(pathElement);
            DomainFile file = folder.getFile(pathElement);
            if (file != null && file.isLink() && (linkInfo = file.getLinkInfo()).isFolderLink()) {
                if (subFolder != null) {
                    throw new IOException("Folder and folder-link name conflict encountered: " + String.valueOf(file));
                }
                if (linkInfo.isExternalLink()) {
                    throw new IOException("May not follow external folder-link: " + String.valueOf(file));
                }
                if (LinkHandler.getLinkFileStatus(file, null) == LinkHandler.LinkStatus.BROKEN) {
                    throw new IOException("May not follow broken folder-link: " + String.valueOf(file));
                }
                subFolder = linkInfo.getLinkedFolder();
            }
            if (subFolder == null) {
                subFolder = folder.createFolder(pathElement);
            }
            folder = subFolder;
        }
        return folder;
    }

    public static DomainFolder getDomainFolder(DomainFolder baseFolder, String relativePath) {
        return ProjectDataUtils.getDomainFolder(baseFolder, relativePath, DomainFolderFilter.ALL_INTERNAL_FOLDERS_FILTER);
    }

    public static DomainFolder getDomainFolder(DomainFolder baseFolder, String relativePath, DomainFolderFilter filter) {
        String[] pathElements;
        if (StringUtils.isBlank((CharSequence)relativePath)) {
            return baseFolder;
        }
        DomainFolder folder = baseFolder;
        for (String pathElement : pathElements = relativePath.split(FileSystem.SEPARATOR)) {
            LinkFileInfo linkInfo;
            if (pathElement.isEmpty()) continue;
            DomainFolder subFolder = folder.getFolder(pathElement);
            DomainFile file = folder.getFile(pathElement);
            if (file != null && file.isLink() && (linkInfo = file.getLinkInfo()).isFolderLink()) {
                if (filter.ignoreFolderLinks()) {
                    return null;
                }
                if (subFolder != null) {
                    Msg.error(ProjectDataUtils.class, (Object)("Folder and folder-link name conflict encountered: " + String.valueOf(file)));
                    return null;
                }
                if (linkInfo.isExternalLink() && filter.ignoreExternalLinks()) {
                    return null;
                }
                if (LinkHandler.getLinkFileStatus(file, null) == LinkHandler.LinkStatus.BROKEN) {
                    Msg.warn(ProjectDataUtils.class, (Object)("Skipping broken folder-link: " + file.getPathname()));
                    return null;
                }
                subFolder = linkInfo.getLinkedFolder();
            }
            if (subFolder == null) {
                return null;
            }
            folder = subFolder;
        }
        return folder;
    }

    public static String getUniqueName(DomainFolder folder, String baseName) {
        int MAX_TRY_COUNT = 1000;
        for (int tryNum = 0; tryNum < MAX_TRY_COUNT; ++tryNum) {
            String tryName = baseName + (tryNum > 0 ? Integer.toString(tryNum) : "");
            if (folder.getFile(tryName) != null || folder.getFolder(tryName) != null) {
                continue;
            }
            return tryName;
        }
        return null;
    }

    private static class DomainFileIterator
    implements Iterator<DomainFile>,
    Iterable<DomainFile> {
        private Deque<DomainFile> fileQueue = new LinkedList<DomainFile>();
        private Deque<DomainFolder> folderQueue = new LinkedList<DomainFolder>();
        private DomainFileFilter filter;

        DomainFileIterator(DomainFolder startFolder, DomainFileFilter filter) {
            Objects.requireNonNull(startFolder, "folder not specified");
            Objects.requireNonNull(filter, "domain file filter not specified");
            this.folderQueue.add(startFolder);
            this.filter = filter;
        }

        private void queueNextFiles() {
            DomainFolder folder;
            while (this.fileQueue.isEmpty() && (folder = this.folderQueue.poll()) != null) {
                DomainFolder[] folders = folder.getFolders();
                for (int i = folders.length - 1; i >= 0; --i) {
                    DomainFolder subfolder = folders[i];
                    this.folderQueue.addFirst(subfolder);
                }
                for (DomainFile df : folder.getFiles()) {
                    if (df.isLink()) {
                        AtomicReference<LinkHandler.LinkStatus> linkStatus = new AtomicReference<LinkHandler.LinkStatus>();
                        if (this.skipLinkFile(df, linkStatus)) continue;
                        if (df.getLinkInfo().isFolderLink()) {
                            LinkedGhidraFolder linkedFolder = this.resolveFolderLink(df, linkStatus.get());
                            if (linkedFolder == null) continue;
                            this.folderQueue.addFirst(linkedFolder);
                            continue;
                        }
                    }
                    if (!this.filter.accept(df)) continue;
                    this.fileQueue.addLast(df);
                }
            }
        }

        private LinkedGhidraFolder resolveFolderLink(DomainFile folderLinkFile, LinkHandler.LinkStatus status) {
            if (status == LinkHandler.LinkStatus.BROKEN) {
                Msg.warn((Object)this, (Object)("Skipping broken folder-link: " + folderLinkFile.getPathname()));
                return null;
            }
            if (status == LinkHandler.LinkStatus.EXTERNAL && !this.filter.followExternallyLinkedFolders()) {
                return null;
            }
            return folderLinkFile.getLinkInfo().getLinkedFolder();
        }

        private boolean skipLinkFile(DomainFile linkFile, AtomicReference<LinkHandler.LinkStatus> returnedLinkStatus) {
            LinkFileInfo linkInfo = linkFile.getLinkInfo();
            boolean isFolderLink = linkInfo.isFolderLink();
            if (isFolderLink && this.filter.ignoreFolderLinks()) {
                return true;
            }
            LinkHandler.LinkStatus linkStatus = LinkHandler.getLinkFileStatus(linkFile, null);
            if (linkStatus == LinkHandler.LinkStatus.BROKEN && this.filter.ignoreBrokenLinks()) {
                return true;
            }
            if (linkStatus == LinkHandler.LinkStatus.EXTERNAL) {
                return true;
            }
            if (linkStatus == LinkHandler.LinkStatus.BROKEN) {
                Msg.warn((Object)this, (Object)("Skipping broken link-file: " + linkFile.getPathname()));
                return true;
            }
            returnedLinkStatus.set(linkStatus);
            return false;
        }

        @Override
        public boolean hasNext() {
            this.queueNextFiles();
            return !this.fileQueue.isEmpty();
        }

        @Override
        public DomainFile next() {
            return this.fileQueue.poll();
        }

        @Override
        public Iterator<DomainFile> iterator() {
            return this;
        }
    }

    private static class DomainFolderIterator
    implements Iterator<DomainFolder>,
    Iterable<DomainFolder> {
        private Deque<DomainFolder> folderQueue = new LinkedList<DomainFolder>();
        private DomainFolder nextFolder;
        private boolean ignoreFolderLinks;
        private boolean ignoreExternalLinks;

        DomainFolderIterator(DomainFolder startFolder, boolean ignoreFolderLinks, boolean ignoreExternalLinks) {
            this.folderQueue.add(startFolder);
            this.ignoreFolderLinks = ignoreFolderLinks;
            this.ignoreExternalLinks = ignoreExternalLinks;
        }

        private void queueNextFiles() {
            if (this.nextFolder == null && !this.folderQueue.isEmpty()) {
                this.nextFolder = this.folderQueue.poll();
                DomainFolder[] folders = this.nextFolder.getFolders();
                for (int i = folders.length - 1; i >= 0; --i) {
                    DomainFolder subfolder = folders[i];
                    this.folderQueue.addFirst(subfolder);
                }
                if (!this.ignoreFolderLinks) {
                    for (DomainFile df : this.nextFolder.getFiles()) {
                        LinkedGhidraFolder linkedFolder = this.resolveFolderLink(df);
                        if (linkedFolder == null) continue;
                        this.folderQueue.addFirst(linkedFolder);
                    }
                }
            }
        }

        private LinkedGhidraFolder resolveFolderLink(DomainFile file) {
            LinkFileInfo linkInfo = file.getLinkInfo();
            if (linkInfo == null || !linkInfo.isFolderLink()) {
                return null;
            }
            LinkHandler.LinkStatus linkStatus = LinkHandler.getLinkFileStatus(file, null);
            if (linkStatus == LinkHandler.LinkStatus.BROKEN) {
                Msg.warn((Object)this, (Object)("Skipping broken folder-link: " + file.getPathname()));
                return null;
            }
            if (linkStatus == LinkHandler.LinkStatus.EXTERNAL && this.ignoreExternalLinks) {
                return null;
            }
            return linkInfo.getLinkedFolder();
        }

        @Override
        public boolean hasNext() {
            this.queueNextFiles();
            return this.nextFolder != null;
        }

        @Override
        public DomainFolder next() {
            DomainFolder tmp = this.nextFolder;
            this.nextFolder = null;
            return tmp;
        }

        @Override
        public Iterator<DomainFolder> iterator() {
            return this;
        }
    }
}

