/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util.extensions;

import generic.jar.ResourceFile;
import ghidra.framework.Application;
import ghidra.util.Msg;
import ghidra.util.exception.CancelledException;
import ghidra.util.extensions.ExtensionDetails;
import ghidra.util.extensions.Extensions;
import ghidra.util.task.TaskMonitor;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import utilities.util.FileUtilities;
import utility.application.ApplicationLayout;

public class ExtensionUtils {
    private static final int ZIPFILE = 1347093252;
    public static String PROPERTIES_FILE_NAME = "extension.properties";
    public static String PROPERTIES_FILE_NAME_UNINSTALLED = "extension.properties.uninstalled";
    private static final Logger log = LogManager.getLogger(ExtensionUtils.class);
    private static Extensions extensions;

    public static void initializeExtensions() {
        extensions = ExtensionUtils.getAllInstalledExtensions();
        extensions.cleanupExtensionsMarkedForRemoval();
        extensions.reportDuplicateExtensions();
    }

    public static ExtensionDetails getExtension(String path) {
        File pathDir = new File(path);
        Set<ExtensionDetails> installedExtensions = ExtensionUtils.getActiveInstalledExtensions();
        for (ExtensionDetails ext : installedExtensions) {
            File installDir = ext.getInstallDir();
            if (!FileUtilities.isPathContainedWithin((File)installDir, (File)pathDir)) continue;
            return ext;
        }
        return null;
    }

    public static boolean isExtension(File file) {
        return ExtensionUtils.getExtension(file, true) != null;
    }

    public static boolean install(ExtensionDetails extension, File file, TaskMonitor monitor) {
        boolean success = false;
        try {
            if (file.isFile()) {
                success = ExtensionUtils.unzipToInstallationFolder(extension, file, monitor);
            }
            success = ExtensionUtils.copyToInstallationFolder(extension, file, monitor);
        }
        catch (CancelledException e) {
            log.info("Extension installation cancelled by user");
        }
        catch (IOException e) {
            Msg.showError(ExtensionUtils.class, null, (String)"Error Installing Extension", (Object)"Unexpected error installing extension", (Throwable)e);
        }
        if (success) {
            extensions.add(extension);
        }
        return success;
    }

    public static Set<ExtensionDetails> getActiveInstalledExtensions() {
        return ExtensionUtils.getAllInstalledExtensions().getActiveExtensions();
    }

    public static Set<ExtensionDetails> getInstalledExtensions() {
        return ExtensionUtils.getAllInstalledExtensions().get();
    }

    public static Extensions getAllInstalledExtensions() {
        if (extensions != null) {
            return extensions;
        }
        log.trace("Finding all installed extensions...");
        extensions = new Extensions(log);
        ApplicationLayout layout = Application.getApplicationLayout();
        for (ResourceFile installDir : layout.getExtensionInstallationDirs()) {
            if (!installDir.isDirectory()) continue;
            log.trace("Checking extension installation dir '" + String.valueOf(installDir));
            File dir = installDir.getFile(false);
            List<File> propFiles = ExtensionUtils.findExtensionPropertyFiles(dir);
            for (File propFile : propFiles) {
                ExtensionDetails extension = ExtensionUtils.createExtensionFromProperties(propFile);
                if (extension == null) continue;
                File extInstallDir = propFile.getParentFile();
                extension.setInstallDir(extInstallDir);
                log.trace("Loading extension '" + extension.getName() + "' from: " + String.valueOf(extInstallDir));
                extensions.add(extension);
            }
        }
        log.trace(() -> "All installed extensions: " + extensions.getAsString());
        return extensions;
    }

    public static ExtensionDetails getExtension(File file, boolean quiet) {
        if (file == null) {
            log.error("Cannot get an extension; null file");
            return null;
        }
        try {
            return ExtensionUtils.tryToGetExtension(file);
        }
        catch (IOException e) {
            if (quiet) {
                log.trace("Exception trying to read an extension from " + String.valueOf(file), (Throwable)e);
            } else {
                log.error("Exception trying to read an extension from " + String.valueOf(file), (Throwable)e);
            }
            return null;
        }
    }

    public static void reload() {
        log.trace("Clearing extensions cache");
        ExtensionUtils.clearCache();
        ExtensionUtils.getAllInstalledExtensions();
    }

    public static void clearCache() {
        extensions = null;
    }

    public static Set<ExtensionDetails> getArchiveExtensions() {
        log.trace("Finding archived extensions");
        ApplicationLayout layout = Application.getApplicationLayout();
        ResourceFile archiveDir = layout.getExtensionArchiveDir();
        if (archiveDir == null) {
            log.trace("No extension archive dir found");
            return Collections.emptySet();
        }
        ResourceFile[] archiveFiles = archiveDir.listFiles();
        if (archiveFiles == null) {
            log.trace("No files in extension archive dir: " + String.valueOf(archiveDir));
            return Collections.emptySet();
        }
        HashSet<ExtensionDetails> results = new HashSet<ExtensionDetails>();
        ExtensionUtils.findExtensionsInZips(archiveFiles, results);
        ExtensionUtils.findExtensionsInFolder(archiveDir.getFile(false), results);
        return results;
    }

    public static ExtensionDetails createExtensionFromProperties(File file) {
        try {
            return ExtensionUtils.tryToLoadExtensionFromProperties(file);
        }
        catch (IOException e) {
            log.error("Error loading extension properties from " + file.getAbsolutePath(), (Throwable)e);
            return null;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static ExtensionDetails createExtensionDetailsFromArchive(ResourceFile resourceFile) {
        File file = resourceFile.getFile(false);
        if (!ExtensionUtils.isZip(file)) {
            return null;
        }
        try (ZipFile zipFile = new ZipFile(file);){
            Properties props = ExtensionUtils.getProperties(zipFile);
            if (props == null) return null;
            ExtensionDetails extension = ExtensionUtils.createExtensionDetails(props);
            extension.setArchivePath(file.getAbsolutePath());
            ExtensionDetails extensionDetails = extension;
            return extensionDetails;
        }
        catch (IOException e) {
            log.error("Unable to read zip file to get extension properties: " + String.valueOf(file), (Throwable)e);
        }
        return null;
    }

    private static void findExtensionsInZips(ResourceFile[] archiveFiles, Set<ExtensionDetails> results) {
        for (ResourceFile file : archiveFiles) {
            ExtensionDetails extension = ExtensionUtils.createExtensionDetailsFromArchive(file);
            if (extension == null) {
                log.trace("Skipping archive file; not an extension: " + String.valueOf(file));
                continue;
            }
            ExtensionDetails existingExtension = ExtensionUtils.get(results, extension);
            if (existingExtension == null) {
                results.add(extension);
                continue;
            }
            String name = extension.getName();
            String newZip = FilenameUtils.getBaseName((String)extension.getArchivePath());
            String existingZip = FilenameUtils.getBaseName((String)existingExtension.getArchivePath());
            ResourceFile dir = file.getParentFile();
            log.error("Skipping extension '%s' found in zip '%s'. Extension by that name already found in zip '%s'. Archive dir %s\n".formatted(name, newZip, existingZip, dir));
        }
    }

    private static ExtensionDetails get(Set<ExtensionDetails> set, ExtensionDetails extension) {
        if (set.contains(extension)) {
            return set.stream().filter(e -> e.equals(extension)).findFirst().get();
        }
        return null;
    }

    private static void findExtensionsInFolder(File dir, Set<ExtensionDetails> results) {
        List<File> propFiles = ExtensionUtils.findExtensionPropertyFiles(dir);
        for (File propFile : propFiles) {
            ExtensionDetails extension = ExtensionUtils.createExtensionFromProperties(propFile);
            if (extension == null) continue;
            File extDir = propFile.getParentFile();
            extension.setArchivePath(extDir.getAbsolutePath());
            if (results.contains(extension)) {
                log.error("Skipping duplicate extension \"" + extension.getName() + "\" found at " + extension.getInstallPath());
            }
            results.add(extension);
        }
    }

    private static ExtensionDetails tryToGetExtension(File file) throws IOException {
        File propertyFile;
        if (file == null) {
            log.error("Cannot get an extension; null file");
            return null;
        }
        if (file.isDirectory() && file.canRead() && (propertyFile = new File(file, PROPERTIES_FILE_NAME)).isFile()) {
            return ExtensionUtils.tryToLoadExtensionFromProperties(propertyFile);
        }
        if (ExtensionUtils.isZip(file)) {
            try (ZipFile zipFile = new ZipFile(file);){
                Properties props = ExtensionUtils.getProperties(zipFile);
                if (props != null) {
                    ExtensionDetails extensionDetails = ExtensionUtils.createExtensionDetails(props);
                    return extensionDetails;
                }
                throw new IOException("No extension.properties file found in zip");
            }
        }
        return null;
    }

    private static ExtensionDetails tryToLoadExtensionFromProperties(File file) throws IOException {
        Properties props = new Properties();
        try (FileInputStream in = new FileInputStream(file.getAbsolutePath());){
            props.load(in);
            ExtensionDetails extensionDetails = ExtensionUtils.createExtensionDetails(props);
            return extensionDetails;
        }
    }

    private static ExtensionDetails createExtensionDetails(Properties props) {
        String name = props.getProperty("name");
        String desc = props.getProperty("description");
        String author = props.getProperty("author");
        String date = props.getProperty("createdOn");
        String version = props.getProperty("version");
        return new ExtensionDetails(name, desc, author, date, version);
    }

    private static Properties getProperties(ZipFile zipFile) throws IOException {
        Properties props = null;
        Enumeration zipEntries = zipFile.getEntries();
        while (zipEntries.hasMoreElements()) {
            ZipArchiveEntry entry = (ZipArchiveEntry)zipEntries.nextElement();
            Properties nextProperties = ExtensionUtils.getProperties(zipFile, entry);
            if (nextProperties == null) continue;
            if (props != null) {
                throw new IOException("Zip file contains multiple extension properties files");
            }
            props = nextProperties;
        }
        return props;
    }

    private static Properties getProperties(ZipFile zipFile, ZipArchiveEntry entry) throws IOException {
        String path = entry.getName();
        List parts = FileUtilities.pathToParts((String)path);
        if (parts.size() != 2) {
            return null;
        }
        if (!entry.getName().endsWith(PROPERTIES_FILE_NAME)) {
            return null;
        }
        InputStream propFile = zipFile.getInputStream(entry);
        Properties prop = new Properties();
        prop.load(propFile);
        return prop;
    }

    private static List<File> findExtensionPropertyFiles(File installDir) {
        ArrayList<File> results = new ArrayList<File>();
        FileUtilities.forEachFile((File)installDir, f -> {
            if (!f.isDirectory() || f.getName().equals("Skeleton")) {
                return;
            }
            File pf = ExtensionUtils.getPropertyFile(f);
            if (pf != null) {
                results.add(pf);
            }
        });
        return results;
    }

    private static File getPropertyFile(File dir) {
        File f = new File(dir, PROPERTIES_FILE_NAME_UNINSTALLED);
        if (f.exists()) {
            return f;
        }
        f = new File(dir, PROPERTIES_FILE_NAME);
        if (f.exists()) {
            return f;
        }
        return null;
    }

    private static boolean isZip(File file) {
        boolean bl;
        if (file == null) {
            log.error("Cannot check for extension zip; null file");
            return false;
        }
        if (file.isDirectory()) {
            return false;
        }
        if (file.length() < 4L) {
            return false;
        }
        DataInputStream in = new DataInputStream(new BufferedInputStream(new FileInputStream(file)));
        try {
            int test = in.readInt();
            bl = test == 1347093252;
        }
        catch (Throwable throwable) {
            try {
                try {
                    in.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                log.trace("Unable to check if file is a zip file: " + String.valueOf(file) + ". " + e.getMessage());
                return false;
            }
        }
        in.close();
        return bl;
    }

    private static boolean copyToInstallationFolder(ExtensionDetails extension, File sourceFolder, TaskMonitor monitor) throws IOException, CancelledException {
        log.trace("Copying extension from " + String.valueOf(sourceFolder));
        ApplicationLayout layout = Application.getApplicationLayout();
        ResourceFile installDir = (ResourceFile)layout.getExtensionInstallationDirs().get(0);
        File installDirRoot = installDir.getFile(false);
        File destinationFolder = new File(installDirRoot, sourceFolder.getName());
        if (ExtensionUtils.hasExistingExtension(destinationFolder, monitor)) {
            return false;
        }
        log.trace("Copying extension to " + String.valueOf(destinationFolder));
        FileUtilities.copyDir((File)sourceFolder, (File)destinationFolder, (TaskMonitor)monitor);
        extension.setInstallDir(destinationFolder);
        return true;
    }

    private static boolean unzipToInstallationFolder(ExtensionDetails extension, File file, TaskMonitor monitor) throws CancelledException, IOException {
        log.trace("Unzipping extension from " + String.valueOf(file));
        ApplicationLayout layout = Application.getApplicationLayout();
        ResourceFile installDir = (ResourceFile)layout.getExtensionInstallationDirs().get(0);
        File installDirRoot = installDir.getFile(false);
        File destinationFolder = new File(installDirRoot, extension.getName());
        if (ExtensionUtils.hasExistingExtension(destinationFolder, monitor)) {
            return false;
        }
        try (ZipFile zipFile = new ZipFile(file);){
            Enumeration entries = zipFile.getEntries();
            while (entries.hasMoreElements()) {
                monitor.checkCancelled();
                ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
                String filePath = String.valueOf(installDir) + File.separator + entry.getName();
                File destination = new File(filePath);
                if (entry.isDirectory()) {
                    destination.mkdirs();
                    continue;
                }
                ExtensionUtils.writeZipEntryToFile(zipFile, entry, destination);
            }
        }
        extension.setInstallDir(destinationFolder);
        return true;
    }

    private static boolean hasExistingExtension(File extensionFolder, TaskMonitor monitor) {
        if (extensionFolder.exists()) {
            Msg.showWarn(ExtensionUtils.class, null, (String)"Duplicate Extension Folder", (Object)("Attempting to install a new extension over an existing directory.\nEither remove the extension for that directory from the UI\nor close Ghidra and delete the directory and try installing again.\n\nDirectory: " + String.valueOf(extensionFolder)));
            return true;
        }
        return false;
    }

    private static void writeZipEntryToFile(ZipFile zFile, ZipArchiveEntry entry, File destination) throws IOException {
        try (BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(destination));){
            IOUtils.copy((InputStream)zFile.getInputStream(entry), (OutputStream)outputStream);
            if (entry.getPlatform() != 3) {
                return;
            }
            int mode = entry.getUnixMode();
            if (mode != 0) {
                Set<PosixFilePermission> perms = ExtensionUtils.getPermissions(mode);
                try {
                    Files.setPosixFilePermissions(destination.toPath(), perms);
                }
                catch (UnsupportedOperationException unsupportedOperationException) {
                    // empty catch block
                }
            }
        }
    }

    private static Set<PosixFilePermission> getPermissions(int unixMode) {
        HashSet<PosixFilePermission> permissions = new HashSet<PosixFilePermission>();
        if ((unixMode & 0x100) != 0) {
            permissions.add(PosixFilePermission.OWNER_READ);
        }
        if ((unixMode & 0x80) != 0) {
            permissions.add(PosixFilePermission.OWNER_WRITE);
        }
        if ((unixMode & 0x40) != 0) {
            permissions.add(PosixFilePermission.OWNER_EXECUTE);
        }
        if ((unixMode & 0x20) != 0) {
            permissions.add(PosixFilePermission.GROUP_READ);
        }
        if ((unixMode & 0x10) != 0) {
            permissions.add(PosixFilePermission.GROUP_WRITE);
        }
        if ((unixMode & 8) != 0) {
            permissions.add(PosixFilePermission.GROUP_EXECUTE);
        }
        if ((unixMode & 4) != 0) {
            permissions.add(PosixFilePermission.OTHERS_READ);
        }
        if ((unixMode & 2) != 0) {
            permissions.add(PosixFilePermission.OTHERS_WRITE);
        }
        if ((unixMode & 1) != 0) {
            permissions.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return permissions;
    }
}

