﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/organizations/OrganizationsRequest.h>
#include <aws/organizations/Organizations_EXPORTS.h>
#include <aws/organizations/model/Tag.h>

#include <utility>

namespace Aws {
namespace Organizations {
namespace Model {

/**
 */
class TagResourceRequest : public OrganizationsRequest {
 public:
  AWS_ORGANIZATIONS_API TagResourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "TagResource"; }

  AWS_ORGANIZATIONS_API Aws::String SerializePayload() const override;

  AWS_ORGANIZATIONS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the resource to add a tag to.</p> <p>You can specify any of the
   * following taggable resources.</p> <ul> <li> <p>Amazon Web Services account –
   * specify the account ID number.</p> </li> <li> <p>Organizational unit – specify
   * the OU ID that begins with <code>ou-</code> and looks similar to:
   * <code>ou-<i>1a2b-34uvwxyz</i> </code> </p> </li> <li> <p>Root – specify the root
   * ID that begins with <code>r-</code> and looks similar to: <code>r-<i>1a2b</i>
   * </code> </p> </li> <li> <p>Policy – specify the policy ID that begins with
   * <code>p-</code> andlooks similar to: <code>p-<i>12abcdefg3</i> </code> </p>
   * </li> </ul>
   */
  inline const Aws::String& GetResourceId() const { return m_resourceId; }
  inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
  template <typename ResourceIdT = Aws::String>
  void SetResourceId(ResourceIdT&& value) {
    m_resourceIdHasBeenSet = true;
    m_resourceId = std::forward<ResourceIdT>(value);
  }
  template <typename ResourceIdT = Aws::String>
  TagResourceRequest& WithResourceId(ResourceIdT&& value) {
    SetResourceId(std::forward<ResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to add to the specified resource.</p> <p>For each tag in the
   * list, you must specify both a tag key and a value. The value can be an empty
   * string, but you can't set it to <code>null</code>.</p>  <p>If any one of
   * the tags is not valid or if you exceed the maximum allowed number of tags for a
   * resource, then the entire request fails.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  TagResourceRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  TagResourceRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceId;

  Aws::Vector<Tag> m_tags;
  bool m_resourceIdHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Organizations
}  // namespace Aws
