﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/textract/Textract_EXPORTS.h>
#include <aws/textract/model/BoundingBox.h>
#include <aws/textract/model/Point.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Textract {
namespace Model {

/**
 * <p>Information about where the following items are located on a document page:
 * detected page, text, key-value pairs, tables, table cells, and selection
 * elements.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/Geometry">AWS
 * API Reference</a></p>
 */
class Geometry {
 public:
  AWS_TEXTRACT_API Geometry() = default;
  AWS_TEXTRACT_API Geometry(Aws::Utils::Json::JsonView jsonValue);
  AWS_TEXTRACT_API Geometry& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_TEXTRACT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>An axis-aligned coarse representation of the location of the recognized item
   * on the document page.</p>
   */
  inline const BoundingBox& GetBoundingBox() const { return m_boundingBox; }
  inline bool BoundingBoxHasBeenSet() const { return m_boundingBoxHasBeenSet; }
  template <typename BoundingBoxT = BoundingBox>
  void SetBoundingBox(BoundingBoxT&& value) {
    m_boundingBoxHasBeenSet = true;
    m_boundingBox = std::forward<BoundingBoxT>(value);
  }
  template <typename BoundingBoxT = BoundingBox>
  Geometry& WithBoundingBox(BoundingBoxT&& value) {
    SetBoundingBox(std::forward<BoundingBoxT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Within the bounding box, a fine-grained polygon around the recognized
   * item.</p>
   */
  inline const Aws::Vector<Point>& GetPolygon() const { return m_polygon; }
  inline bool PolygonHasBeenSet() const { return m_polygonHasBeenSet; }
  template <typename PolygonT = Aws::Vector<Point>>
  void SetPolygon(PolygonT&& value) {
    m_polygonHasBeenSet = true;
    m_polygon = std::forward<PolygonT>(value);
  }
  template <typename PolygonT = Aws::Vector<Point>>
  Geometry& WithPolygon(PolygonT&& value) {
    SetPolygon(std::forward<PolygonT>(value));
    return *this;
  }
  template <typename PolygonT = Point>
  Geometry& AddPolygon(PolygonT&& value) {
    m_polygonHasBeenSet = true;
    m_polygon.emplace_back(std::forward<PolygonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides a numerical value corresponding to the rotation of the text.</p>
   */
  inline double GetRotationAngle() const { return m_rotationAngle; }
  inline bool RotationAngleHasBeenSet() const { return m_rotationAngleHasBeenSet; }
  inline void SetRotationAngle(double value) {
    m_rotationAngleHasBeenSet = true;
    m_rotationAngle = value;
  }
  inline Geometry& WithRotationAngle(double value) {
    SetRotationAngle(value);
    return *this;
  }
  ///@}
 private:
  BoundingBox m_boundingBox;

  Aws::Vector<Point> m_polygon;

  double m_rotationAngle{0.0};
  bool m_boundingBoxHasBeenSet = false;
  bool m_polygonHasBeenSet = false;
  bool m_rotationAngleHasBeenSet = false;
};

}  // namespace Model
}  // namespace Textract
}  // namespace Aws
